﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ldn.h>

namespace nns { namespace ldn
{
    /**
     * @brief           ローカル通信の状態です。
     */
    enum LdnSubScene
    {
        //! 無効なサブシーンです。
        LdnSubScene_None,

        //! ローカル通信ライブラリを初期化中です。
        LdnSubScene_Initializing,

        //! ローカル通信ライブラリを初期化済です。
        LdnSubScene_Initialized,

        //! ステーションとして接続先のネットワークを選択中です。
        LdnSubScene_Station,

        //! ステーションとしてネットワークを探索中です。
        LdnSubScene_StationScanning,

        //! ステーションとしてネットワークに接続試行中です。
        LdnSubScene_StationConnecting,

        //! ステーションとしてネットワークに接続しています。
        LdnSubScene_StationConnected,

        //! ステーションとして接続先のネットワークから切断されました。
        LdnSubScene_StationDisconnected,

        //! ステーションとしての動作を終了中です。
        LdnSubScene_StationClosing,

        //! アクセスポイントとしてネットワークを構築中です。
        LdnSubScene_AccessPointCreating,

        //! アクセスポイントとしてネットワークを展開しています。
        LdnSubScene_AccessPointCreated,

        //! アクセスポイントとして周囲のネットワークをスキャンしています。
        LdnSubScene_AccessPointScanning,

        //! アクセスポイントとして指定されたステーションを切断中です。
        LdnSubScene_AccessPointRejecting,

        //! アクセスポイントとしての動作を終了中です。
        LdnSubScene_AccessPointClosing,

        //! ローカル通信ライブラリを終了中です。
        LdnSubScene_Finalizing,

        //! エラー状態です。
        LdnSubScene_Error,
    };

    /**
     * @brief           通信で交換した情報を基に構築したノード毎のカウンタ情報です。
     */
    struct Counter
    {
        //! 最初に受信したフレームのシーケンス番号です。
        int32_t start;

        //! 最新のシーケンス番号です。
        int32_t current;

        //! 今までに受信したフレームの総数です。
        int32_t count;

        //! 受信時のパケット損失率です。
        float loss;
    };

    /**
     * @brief           ローカル通信中に保持するデータです。
     */
    struct LdnData
    {
        //! 現在のサブシーンです。
        int16_t subScene;

        //! 選択されているメニュー項目の X 位置です。
        int8_t x;

        //! 選択されているメニュー項目の Y 位置です。
        int8_t y;

        //! 最後に発生したエラーです。
        nn::Result lastError;

        //! 現在のサブシーンに入ってからの経過フレーム数です。
        int64_t subSceneFrame;

        //! ネットワーク情報です。
        nn::ldn::NetworkInfo network;

        //! ネットワーク情報の差分です。
        nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];

        //! スキャンで発見したネットワークです。
        nn::ldn::NetworkInfo scanResult[nn::ldn::ScanResultCountMax];

        //! スキャンで発見したネットワークの総数です。
        int8_t scanResultCount;

        //! データ更新の有無です。
        bool isUpdated;

        //! 切断理由です。
        int8_t disconnectReason;

        //! 切断対象のステーションの NodeId です。
        int8_t rejectNodeId;

        //! 現在のシーケンス番号です。
        int32_t sequenceNumber;

        //! 接続直後のシーケンス番号です。
        int32_t initialNumber;

        //! 自身の IPv4 アドレスです。
        nn::ldn::Ipv4Address address;

        //! 自身のサブネットマスクです。
        nn::ldn::SubnetMask mask;
        NN_PADDING4;

        //! 各ノードに対応するカウンタです。
        Counter counters[nn::ldn::NodeCountMax];
    };

}} // namespace nns::ldn
