﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/hid.h>
#include "DebugPad.h"

namespace nns { namespace ldn { namespace
{
    uint16_t SetIf(uint16_t state, Button button, bool cond)
    {
        return static_cast<uint16_t>(cond ? state | button : state);
    }

}}} // namespace nns::ldn::<unnamed>

namespace nns { namespace ldn
{
    DebugPad::DebugPad() NN_NOEXCEPT
        : m_PreviousButtonState(0),
          m_IsInitialized(false)
    {
    }

    DebugPad::~DebugPad() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            Finalize();
        }
    }

    void DebugPad::Initialize() NN_NOEXCEPT
    {
        NN_ASSERT(!m_IsInitialized, "debug pad has already been initialized.");
        m_IsInitialized = true;
        nn::hid::InitializeDebugPad();
    }

    void DebugPad::Finalize() NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized, "debug pad has yet to been initialized.");
        m_IsInitialized = false;
    }

    void DebugPad::Update(PadState* pOutState) NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized, "debug pad has yet to been initialized.");
        NN_ASSERT_NOT_NULL(pOutState);

        // デバッグパッドの入力を取得します。
        nn::hid::DebugPadState padState;
        nn::hid::GetDebugPadState(&padState);

        // nn::hid::DebugPadState から PadState に変換します。
        uint16_t down = 0;
        down = SetIf(down, Button_A, padState.buttons.Test<nn::hid::DebugPadButton::A>());
        down = SetIf(down, Button_B, padState.buttons.Test<nn::hid::DebugPadButton::B>());
        down = SetIf(down, Button_X, padState.buttons.Test<nn::hid::DebugPadButton::X>());
        down = SetIf(down, Button_Y, padState.buttons.Test<nn::hid::DebugPadButton::Y>());
        down = SetIf(down, Button_L, padState.buttons.Test<nn::hid::DebugPadButton::L>());
        down = SetIf(down, Button_R, padState.buttons.Test<nn::hid::DebugPadButton::R>());
        down = SetIf(down, Button_ZL, padState.buttons.Test<nn::hid::DebugPadButton::ZL>());
        down = SetIf(down, Button_ZR, padState.buttons.Test<nn::hid::DebugPadButton::ZR>());
        down = SetIf(down, Button_Plus, padState.buttons.Test<nn::hid::DebugPadButton::Start>());
        down = SetIf(down, Button_Minus, padState.buttons.Test<nn::hid::DebugPadButton::Select>());
        down = SetIf(down, Button_Left, padState.buttons.Test<nn::hid::DebugPadButton::Left>());
        down = SetIf(down, Button_Up, padState.buttons.Test<nn::hid::DebugPadButton::Up>());
        down = SetIf(down, Button_Right, padState.buttons.Test<nn::hid::DebugPadButton::Right>());
        down = SetIf(down, Button_Down, padState.buttons.Test<nn::hid::DebugPadButton::Down>());
        pOutState->button.down = down;
        pOutState->button.trigger = (~m_PreviousButtonState) & down;
        m_PreviousButtonState = down;
    }

}} // namespace nns::ldn
