﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/ldn.h>
#include "CommandLineParser.h"

namespace nns { namespace ldn { namespace
{
    const size_t LongNameLengthMax = 16;

    bool IsOption(const char* arg, char shortName, const char* longName) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(arg);
        NN_ASSERT_NOT_EQUAL(shortName, '-');
        return (shortName != 0 && arg[0] == '-' && arg[1] == shortName) ||
            (longName != nullptr && std::strncmp(arg, longName, LongNameLengthMax) == 0);
    }

    void GetString(char* out, const char* name, const char* str, size_t bufferSize) NN_NOEXCEPT
    {
        // 文字列を出力します。
        std::strncpy(out, str, bufferSize);

        // 文字列のバイト数を取得します。
        size_t bytes = std::strlen(str);
        if (bufferSize <= bytes)
        {
            NN_LOG("FAILED: %s should be smaller than %u Bytes\n", name, bufferSize);
            std::exit(-1);
        }
        else
        {
            // TODO: UTF-8 の 10 文字以内であることを検証する
        }
    }

    int GetInteger(const char* name, const char* str, int min, int max) NN_NOEXCEPT
    {
        // 文字列を整数値に変換します。
        char* end;
        int n = std::strtol(str, &end, 10);

        // 変換に失敗した場合にはエラーメッセージを出力して終了します。
        if (end == str || *end != 0)
        {
            NN_LOG("FAILED: %s should be decimal number\n", name);
            std::exit(-1);
        }
        else if (n < min)
        {
            NN_LOG("FAILED: %s (=%d) should be greater than or equal to %d\n", name, n, min);
            std::exit(-1);
        }
        else if (max < n)
        {
            NN_LOG("FAILED: %s (=%d) should be less than or equal to %d\n", name, n, max);
            std::exit(-1);
        }
        return n;
    }

}}} // namespace nnt::ldn::<unnamed>

namespace nns { namespace ldn
{
    void Parse(Config* pOutConfig, int argc, char* argv[]) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutConfig);
        NN_SDK_ASSERT(0 < argc);
        NN_SDK_ASSERT_NOT_NULL(argv);

        // デフォルト値を設定します。
        std::memset(pOutConfig, 0, sizeof(Config));
        std::strncpy(pOutConfig->userName, "Anonymous", nn::ldn::UserNameBytesMax);
        pOutConfig->nodeCountMax = nn::ldn::NodeCountMax;
        pOutConfig->channel = nn::ldn::AutoChannel;
        pOutConfig->localCommunicationVersion = 0;
        pOutConfig->sceneId = 0;
        pOutConfig->localCommunicationId = nn::ldn::DefaultLocalCommunicationId;
        pOutConfig->port = 12345;
        pOutConfig->packetRate = 30;
        pOutConfig->isEncrypted = false;

        // コマンドライン引数を最後まで解析します。
        int i = 1;
        while (i < argc)
        {
            // ユーザ名です。
            if (IsOption(argv[i], 'u', "user") && i + 1 < argc)
            {
                GetString(pOutConfig->userName, "Username", argv[i + 1],
                          nn::ldn::UserNameBytesMax);
                i += 2;
            }

            // ノードの最大数です。
            else if (IsOption(argv[i], 'n', "node") && i + 1 < argc)
            {
                pOutConfig->nodeCountMax = static_cast<int8_t>(
                    GetInteger("The number of nodes", argv[i + 1], 1, nn::ldn::NodeCountMax));
                i += 2;
            }

            // 無線チャンネルです。
            else if (IsOption(argv[i], 'c', "channel") && i + 1 < argc)
            {
                int channel = GetInteger("Channel", argv[i + 1], 0, 32767);
                if (channel != 0 && channel != 1 && channel != 6 && channel != 11 &&
                    channel != 36 && channel != 40 && channel != 44 && channel != 48)
                {
                    NN_LOG("FAILED: %s (=%d) should be 0, 1, 6, 11, 36, 40, 44 or 48\n",
                           "Channel", channel);
                }
                pOutConfig->channel = static_cast<int16_t>(channel);
                i += 2;
            }

            // ローカル通信バージョンです。
            else if (IsOption(argv[i], 'v', "version") && i + 1 < argc)
            {
                pOutConfig->localCommunicationVersion = static_cast<int16_t>(
                    GetInteger("Local communication version", argv[i + 1],
                    nn::ldn::LocalCommunicationVersionMin, nn::ldn::LocalCommunicationVersionMax));
                i += 2;
            }

            // シーン識別子です。
            else if (IsOption(argv[i], 's', "scene") && i + 1 < argc)
            {
                pOutConfig->sceneId = static_cast<int16_t>(
                    GetInteger("Scene ID", argv[i + 1], 0, 65535));
                i += 2;
            }

            // ポート番号です。
            else if (IsOption(argv[i], 'p', "port") && i + 1 < argc)
            {
                pOutConfig->port = static_cast<uint16_t>(
                    GetInteger("Port number", argv[i + 1], 0, 65535));
                i += 2;
            }

            // パケットの送信レートです。
            else if (IsOption(argv[i], 'r', "rate") && i + 1 < argc)
            {
                pOutConfig->packetRate = static_cast<uint8_t>(
                    GetInteger("Packet rate", argv[i + 1], 1, 255));
                i += 2;
            }

            // 暗号化の設定です。
            else if (IsOption(argv[i], 'e', "encrypt"))
            {
                pOutConfig->isEncrypted = true;
                ++i;
            }

            // 無効な引数です。
            else
            {
                NN_LOG("FAILED: %s is an invalid argument\n", argv[i]);
                std::exit(-1);
            }
        }
    }

}} // namespace nnt::ldn
