﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define __STDC_FORMAT_MACROS
#include <cinttypes>
#include <cstdio>
#include <nn/nn_Log.h>

#include "DrawText.h"
#include "Graphics.h"

void DrawSessionInfo(const nn::lcs::SessionInfo& info, int* x, int* y) NN_NOEXCEPT
{
    char str[256];

    ::std::sprintf(str, "ホストのデバイス名 : %s  -  配信しているアプリケーション数 : %d  -  現在の端末数 : %d / 参加できる最大端末数 : %d - 通信品質 : %d",
        info.hostUserName, info.contentsCount, info.nodeCount, info.nodeCountMax, info.linkLevel);
    WriteText((*x) + 2, (*y)++, str, Color::White);
    for (int i = 0; i < info.contentsCount; ++i)
    {
        ::std::sprintf(str, "ID : %" PRIx64 " / Version : %s",
            info.contents[i].applicationId.value, info.contents[i].displayVersion);
        WriteText((*x) + 3, (*y)++, str, Color::White);
    }
}

void DrawProgress(ApplicationResource app, int line) NN_NOEXCEPT
{
    char str[256];
    int x = 0;
    int y = line;
    int count = 0;

    nn::lcs::Progress progress;
    if (nn::lcs::GetProgress(&progress).IsFailure())
    {
        return;
    }
    while (count < app.nodeCount)
    {
        if (app.nodeInfo[count].index == progress.transferIndex)
        {
            break;
        }
        ++count;
    }

    if (progress.transferRole == nn::lcs::TransferRole_Sender)
    {
        if (progress.downloadedSize == progress.size)
        {
            ::std::sprintf(str, "相手の操作を待っています");
        }
        else
        {
            if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::System::Index))
            {
                ::std::sprintf(str, "%s へシステムを送信中です", app.nodeInfo[count].userName);
            }
            else if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::Application::Index))
            {
                ::std::sprintf(str, "%s へアプリケーションを送信中です", app.nodeInfo[count].userName);
            }
            else if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::Patch::Index))
            {
                ::std::sprintf(str, "%s へパッチを送信中です", app.nodeInfo[count].userName);
            }
        }
    }
    else if (progress.transferRole == nn::lcs::TransferRole_Receiver)
    {
        if (progress.downloadedSize == progress.size)
        {
            ::std::sprintf(str, "相手の操作を待っています");
        }
        else
        {
            if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::System::Index))
            {
                ::std::sprintf(str, "%s からシステムを受信中です", app.nodeInfo[count].userName);
            }
            else if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::Application::Index))
            {
                ::std::sprintf(str, "%s からアプリケーションを受信中です", app.nodeInfo[count].userName);
            }
            else if (progress.info.contentsFlag.Test(nn::lcs::ContentsType::Patch::Index))
            {
                ::std::sprintf(str, "%s からパッチを受信中です", app.nodeInfo[count].userName);
            }
        }

    }
    else
    {
        ::std::sprintf(str, "配信を待機中です");
    }
    WriteText(x, y++, str, Color::White);
    ::std::sprintf(str, "%" PRIu64 " / %" PRIu64 "", progress.downloadedSize, progress.size);
    WriteText(x, y++, str, Color::White);
}

void DrawSessionState(int line) NN_NOEXCEPT
{
    int x = 0;
    int y = line;

    nn::lcs::SessionState state;
    if (nn::lcs::GetSessionState(&state).IsSuccess())
    {
        WriteText(x, y++, "-- セッションの状態 --", Color::White);
        switch (state)
        {
        case nn::lcs::SessionState_None:
            WriteText(x, y++, "・配信開始前です", Color::White);
            break;
        case nn::lcs::SessionState_ForceLup:
            WriteText(x, y++, "・強制システム更新中です", Color::White);
            break;
        case nn::lcs::SessionState_ShareContents:
            WriteText(x, y++, "・コンテンツを配信中です", Color::White);
            break;
        case nn::lcs::SessionState_Waiting:
            WriteText(x, y++, "・一時停止中です", Color::White);
            break;
        default:
            break;
        }
    }
}

void DrawNodeProgress(ApplicationResource app, int line) NN_NOEXCEPT
{
    char str[256];
    int x = 0;
    int y = line;

    WriteText(x, y++, "-- 他の端末の受信情報 --", Color::White);
    WriteText(x, y++, "ユーザ名 : 受信したコンテンツ数 / 受信するコンテンツ数", Color::White);

    if (nn::lcs::GetNodes(app.nodeInfo, &app.nodeCount, nn::lcs::NodeCountMax).IsSuccess())
    {
        nn::lcs::NodeProgress nodeProgress;
        for (int i = 0; i < app.nodeCount; ++i)
        {
            if (nn::lcs::GetNodeProgress(&nodeProgress, app.nodeInfo[i].index).IsSuccess())
            {
                ::std::sprintf(str, "%s : %d / %d",
                    app.nodeInfo[i].userName, nodeProgress.downloadedContentCount, nodeProgress.contentCount);
                WriteText(1, y++, str, Color::White);
            }
        }
    }
}

void DrawIsContentShareSucceeded(int line) NN_NOEXCEPT
{
    int x = 0;
    int y = line;

    nn::lcs::SessionInfo info = {};
    if (nn::lcs::GetSessionInfo(&info).IsSuccess())
    {
        if (info.isContentShareSucceeded)
        {
            WriteText(x, y++, "コンテンツの配信に成功", Color::White);
        }
    }
}

void DrawDownloadedContent(ApplicationResource app) NN_NOEXCEPT
{
    char str[256];
    int count;
    int x = 0;
    int y = 3;

    nn::lcs::ContentsInfo info[nn::lcs::DownloadableContentsCountMax];
    if (nn::lcs::GetDownloadedContents(info, &count, nn::lcs::DownloadableContentsCountMax).IsFailure())
    {
        return;
    }
    else
    {
        if (count == 0)
        {
            WriteText(x, y++, "受信したコンテンツはありません", Color::White);
        }
        else
        {
            WriteText(x, y++, "-- 受信したコンテンツ --", Color::White);
            for (int i = 0; i < count; ++i)
            {
                if (info[i].contentsFlag.Test(nn::lcs::ContentsType::System::Index))
                {
                    WriteText(x, y++, "システム", Color::White);
                }
                else
                {
                    if (info[i].contentsFlag.Test(nn::lcs::ContentsType::Application::Index))
                    {
                        WriteText(x, y, "アプリケーション", Color::White);
                    }
                    if (info[i].contentsFlag.Test(nn::lcs::ContentsType::Patch::Index))
                    {
                        WriteText(x, y, "パッチ", Color::White);
                    }

                    char str[256];
                    ::std::sprintf(str, "ID:%" PRIx64 " / Version : %s",
                        info[i].applicationId.value, info[i].displayVersion);
                    WriteText(x + 10, y++, str, Color::White);
                }
            }
        }
    }

    y = 10;
    WriteText(x, y++, "-- 他の端末の受信情報 --", Color::White);
    WriteText(1, y++, "ユーザ名 : 受信したコンテンツ数 / 受信するコンテンツ数", Color::White);
    nn::lcs::NodeProgress nodeProgress;
    for (int i = 0; i < app.nodeCount; ++i)
    {
        if (nn::lcs::GetNodeProgress(&nodeProgress, app.nodeInfo[i].index).IsFailure())
        {
            return;
        }
        ::std::sprintf(str, "%s : %d / %d",
            app.nodeInfo[i].userName, nodeProgress.downloadedContentCount, nodeProgress.contentCount);
        WriteText(1, y++, str, Color::White);
    }
}

void DrawText(ApplicationResource app) NN_NOEXCEPT
{
    char buffer[ApplicationControlDataSize];
    int x = 0;
    int y = 0;
    char str[256];

    BeginText();

    if (app.appState == ApplicationState_Initialized)
    {
        WriteText(x, y++, "おすそわけ通信です", Color::White);
        char name[nn::lcs::UserNameBytesMax] = {};
        size_t nameSize;
        app.config.GetName(name, &nameSize, nn::lcs::UserNameBytesMax);
        WriteText(x, y, "ユーザ名 : ", Color::White);
        WriteText((x + 8), y++, name, Color::White);

        for (int i = 0; i < InitCount; ++i)
        {
            if (app.cursor == i)
            {
                WriteText(0, y, ">", Color::Orange);
            }
            WriteText(2, y++, strInit[i], Color::White);
        }
    }
    else if (app.appState == ApplicationState_ListUpApplication)
    {
        WriteText(x, y++, "↑↓でアプリケーションを選択できます", Color::White);
        WriteText(x, y++, "選択したアプリケーションで A ボタンでセッションを構築します", Color::White);
        WriteText(x, y++, "B ボタンで初期状態に戻ります", Color::White);
        y++;

        if (app.sharableAppCount == 0)
        {
            WriteText(x, y++, "配信可能なアプリケーションが見つかりませんでした。", Color::White);
        }
        else
        {
            for (int i = 0; i < app.sharableAppCount; ++i)
            {
                if (app.cursor == i)
                {
                    WriteText(0, y, ">", Color::Orange);
                }
                size_t size;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::GetApplicationControlData(
                    &size, buffer, ApplicationControlDataSize,
                    nn::ns::ApplicationControlSource::Storage, app.sharableAppId[i]));

                nn::ns::ApplicationControlDataAccessor accessor(buffer, size);
                auto& acProperty = accessor.GetProperty();
                ::std::sprintf(str, "%s [ID:%" PRIx64 "] / Version : %s",
                    acProperty.title[0].name, app.sharableAppId[i].value, acProperty.displayVersion);
                WriteText(3, y++, str, Color::White);
            }
        }
    }
    else if (app.appState == ApplicationState_HostOpened)
    {
        WriteText(x, y++, "Y ボタンでクライアントの参加設定を変更します", Color::White);
        WriteText(x, y++, "X ボタンで配信を開始します", Color::White);
        WriteText(x, y++, "B ボタンでセッションを破棄して、初期状態に戻ります", Color::White);
        y++;

        WriteText(x, y++, "-- 現在のセッションの情報 --", Color::White);
        ::std::sprintf(str, "自身のインデックス : %x", nn::lcs::GetMyIndex());
        WriteText(x, y++, str, Color::White);

        DrawSessionInfo(app.sessionInfo[0], &x, &y);

        if (app.isAcceptPolicy)
        {
            WriteText(1, y++, "クライアントの参加設定は AcceptPolicy_AlwaysAccept です", Color::White);
        }
        else
        {
            WriteText(1, y++, "クライアントの参加設定は AcceptPolicy_AlwaysReject です", Color::White);
        }
        y++;

        WriteText(x, y++, "-- 必要な空きストレージサイズ --", Color::White);
        ::std::sprintf(str, " %zu Byte", app.requiredStorageSize);
        WriteText(1, y++, str, Color::White);
        y++;

        WriteText(x, y++, "-- 現在の端末の情報 --", Color::White);
        if (app.nodeCount == 0)
        {
            WriteText(x, y++, "端末は接続されていません。", Color::White);
        }
        for (int i = 0; i < app.nodeCount; ++i)
        {
            ::std::sprintf(str, "%s[Index : %x]", app.nodeInfo[i].userName, app.nodeInfo[i].index);
            WriteText(2, y++, str, Color::White);
        }
    }
    else if (app.appState == ApplicationState_Scaned)
    {
        WriteText(x, y++, "↑↓でセッションを選択できます", Color::White);
        WriteText(x, y++, "選択したセッションで A ボタンを押すとセッションへ参加します", Color::White);
        WriteText(x, y++, "X ボタンで再度スキャンを行います", Color::White);
        WriteText(x, y++, "B ボタンで初期状態に戻ります", Color::White);
        y++;

        if (app.sessionCount == 0)
        {
            WriteText(x, y++, "LCS のセッションが見つかりませんでした。", Color::White);
        }
        else
        {
            for (int i = 0; i < app.sessionCount; ++i)
            {
                // 必要に応じて、詳細情報を取得してください。
                // 取得方法については GetAppDetailInfo() を参考にしてください。

                WriteText(x, y++, "---------------------------------------------------------------------", Color::White);
                if (app.cursor == i)
                {
                    WriteText(0, y, ">", Color::Orange);
                }
                DrawSessionInfo(app.sessionInfo[i], &x, &y);
            }
            WriteText(x, y++, "---------------------------------------------------------------------", Color::White);
        }
    }
    else if (app.appState == ApplicationState_ClientJoined)
    {
        WriteText(x, y++, "B ボタンでセッションを離脱して、初期状態に戻ります", Color::White);
        y++;
        WriteText(x, y++, "-- 現在のセッションの情報 --", Color::White);
        ::std::sprintf(str, "自身のインデックス : %x", nn::lcs::GetMyIndex());
        WriteText(1, y++, str, Color::White);
        ::std::sprintf(str, "現在の端末数 : %d / 参加できる最大端末数 : %d",
            app.sessionInfo[0].nodeCount, app.sessionInfo[0].nodeCountMax);
        WriteText(1, y++, str, Color::White);
        ::std::sprintf(str, "配信しているアプリケーション数 : %d", app.sessionInfo[0].contentsCount);
        WriteText(1, y++, str, Color::White);
        for (int i = 0; i < app.sessionInfo[0].contentsCount; ++i)
        {
            ::std::sprintf(str, "ID : %" PRIx64 " / Version : %s",
                app.sessionInfo[0].contents[i].applicationId.value, app.sessionInfo[0].contents[i].displayVersion);
            WriteText(2, y++, str, Color::White);
        }
        y++;

        WriteText(x, y++, "-- 必要な空きストレージサイズ --", Color::White);
        ::std::sprintf(str, " %zu Byte", app.requiredStorageSize);
        WriteText(1, y++, str, Color::White);
        y++;

        WriteText(x, y++, "-- 現在の端末の情報 --", Color::White);
        if (app.nodeCount == 0)
        {
            WriteText(x, y++, "端末は接続されていません。", Color::White);
        }
        for (int i = 0; i < app.nodeCount; ++i)
        {
            ::std::sprintf(str, "%s[Index : %x]", app.nodeInfo[i].userName, app.nodeInfo[i].index);
            WriteText(2, y++, str, Color::White);
        }
    }
    else if (app.appState == ApplicationState_Transferring)
    {
        WriteText(x, y++, "B ボタンでキャンセルできます", Color::White);
        DrawProgress(app, 3);
        DrawSessionState(8);
        DrawNodeProgress(app, 12);
        DrawIsContentShareSucceeded(22);
    }
    else if (app.appState == ApplicationState_Suspending)
    {
        WriteText(x, y++, "中断が発生しました", Color::White);
        if (app.suspendedReason == nn::lcs::SuspendedReason_IncompatibleContentsInfo)
        {
            ::std::sprintf(str, "コンテンツの情報が取得できません。本体更新が必要です。");
        }
        else if (app.suspendedReason == nn::lcs::SuspendedReason_NeedTerminateApplication)
        {
            ::std::sprintf(str, "アプリを終了してください。");
        }
        else if (app.suspendedReason == nn::lcs::SuspendedReason_EulaRequired)
        {
            size_t eulaSize = 0;
            nn::Result result = nn::lcs::GetDownloadedEulaDataSize(&eulaSize, "JPja/Eula.msbt.szs");
            if (result.IsSuccess())
            {
                ::std::sprintf(str, "ユーザの同意が必要です。/ EULA サイズ : %zu Byte", eulaSize);
            }
            else
            {
                ::std::sprintf(str, "ユーザの同意が必要です。/ EULA サイズを取得できませんでした。(Error : %08x)",
                    result.GetInnerValueForDebug());
            }
        }
        else if (app.suspendedReason == nn::lcs::SuspendedReason_RebootRequired)
        {
            ::std::sprintf(str, "再起動が必要です。");
        }
        else if (app.suspendedReason == nn::lcs::SuspendedReason_StorageSpaceNotEnough)
        {
            size_t RequiredSize = 0;
            nn::lcs::GetRequiredStorageSize(&RequiredSize);
            ::std::sprintf(str, "ストレージに十分な空き容量がありません。/ 必要容量 : %zu Byte", RequiredSize);
        }
        else if (app.suspendedReason == nn::lcs::SuspendedReason_StorageSpaceNotEnoughOnRequredNode)
        {
            size_t RequiredSize = 0;
            nn::lcs::GetRequiredStorageSize(&RequiredSize);
            ::std::sprintf(str, "ストレージに十分な空き容量がありません。/ 必要容量 : %zu Byte", RequiredSize);
        }

        WriteText(2, y++, str, Color::White);
        y++;
        WriteText(x, y++, "Y ボタンで配信を再開します", Color::White);
        if (app.suspendedReason == nn::lcs::SuspendedReason_RebootRequired ||
            app.suspendedReason == nn::lcs::SuspendedReason_StorageSpaceNotEnough)
        {
            WriteText(x, y++, "X ボタンで再合流用のコンテキストを保存し、一時離脱を行います", Color::White);
        }
        WriteText(x, y++, "B ボタンで初期状態に戻ります", Color::White);
    }
    else if (app.appState == ApplicationState_Resuming)
    {
        WriteText(x, y++, "再合流中です", Color::White);
        WriteText(x, y++, "B ボタンでキャンセルできます", Color::White);
    }
    else if (app.appState == ApplicationState_SessionEnd)
    {
        WriteText(x, y++, "コンテンツの配信が完了しました", Color::White);
        WriteText(x, y++, "B ボタンで初期状態に戻ります", Color::White);
        DrawDownloadedContent(app);
    }

    EndText();
} // NOLINT(impl/function_size)
