﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PointingModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

PointingModeState::PointingModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_PointingProcessorState()
{
    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_PointingProcessorState.samplingNumber, nullptr);
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);
    auto* pStatus = &m_PointingProcessorState.pointingStatus;
    auto* pPosition = &m_PointingProcessorState.position;
    m_ReadOnlyMenu.emplace_back("Status",
        [pStatus](std::stringstream& sstr) {
            sstr << ((*pStatus == nn::irsensor::PointingStatus_DataValid) ? "DataValid" : "LackOfObject");
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
    m_ReadOnlyMenu.emplace_back("Position",
        [pPosition](std::stringstream& sstr) {
            sstr << "(" << static_cast<float>(pPosition->x) << "," << static_cast<float>(pPosition->y) << ")";
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

void PointingModeState::Start()
{
    nn::irsensor::RunPointingProcessor(m_IrCameraHandle);
}

void PointingModeState::Update()
{
    int count;
    nn::Result result = nn::irsensor::GetPointingProcessorStates(&m_PointingProcessorState, &count, 1, m_IrCameraHandle);
    HandleResult(result);
    auto sampling = m_PointingProcessorState.samplingNumber / 3;
    m_Statistics.Update(&sampling, IrSensorModeStatistics::ExpectedPointingModeFramerate);
}

namespace
{

void WritePointingProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::PointingProcessorState* pPointingState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pPointingState);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green ={ { 0, 255, 0, 255 } };
    const nn::util::Uint8x4 Red ={ { 255, 0, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 座標を描画
    if (pPointingState->pointingStatus == nn::irsensor::PointingStatus_DataValid)
    {
        pPrimitiveRenderer->SetColor(Green);
    }
    else
    {
        pPrimitiveRenderer->SetColor(Red);
    }

    nn::util::VectorSet(&begin,
        xStart + (xEnd - xStart) * (1.0f - ((pPointingState->position.x + 1.0f) / 2.0f)),
        yStart + (yEnd - yStart) * ((pPointingState->position.y + 1.0f) / 2.0f), 0.0f);
    pPrimitiveRenderer->DrawSphere(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface_Solid,
        nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
        begin,
        0.01f);
}

}

void PointingModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WritePointingProcessorState(pPrimitiveRenderer, pCommandBuffer, &m_PointingProcessorState, index);
}
