﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ClusteringModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

ClusteringModeState::ClusteringModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_ClusteringProcessorState()
    , m_ClusteringProcessorConfig()
{
    nn::irsensor::GetClusteringProcessorDefaultConfig(&m_ClusteringProcessorConfig);
    nn::irsensor::ClusteringProcessorConfig* pClusteringConfig = &m_ClusteringProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pClusteringConfig->irCameraConfig,
        nn::irsensor::ClusteringProcessorExposureTimeMin,
        nn::irsensor::ClusteringProcessorExposureTimeMax
    );

    m_ReadWriteMenu.emplace_back("Object Pixel Count Min",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectPixelCountMin;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectPixelCountMin += delta * 16;
            pClusteringConfig->objectPixelCountMin =
            std::min(std::max(pClusteringConfig->objectPixelCountMin, 0), pClusteringConfig->objectPixelCountMax - 1);
        }
    );
    m_ReadWriteMenu.emplace_back("Object Pixel Count Max",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectPixelCountMax;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectPixelCountMax += delta * 16;
            pClusteringConfig->objectPixelCountMax =
            std::max(std::min(pClusteringConfig->objectPixelCountMax, nn::irsensor::ClusteringProcessorObjectPixelCountMax), pClusteringConfig->objectPixelCountMin + 1);
        }
    );
    m_ReadWriteMenu.emplace_back("Object Intensity Min",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << pClusteringConfig->objectIntensityMin;
        },
        [pClusteringConfig](int8_t delta) {
            pClusteringConfig->objectIntensityMin += delta * 8;
            pClusteringConfig->objectIntensityMin =
            std::max(std::min(pClusteringConfig->objectIntensityMin, nn::irsensor::IrCameraIntensityMax), 0);
        }
    );
    m_ReadWriteMenu.emplace_back("External Light Filter",
        [pClusteringConfig](std::stringstream& sstr) {
            sstr << (pClusteringConfig->isExternalLightFilterEnabled ? "Enabled" : "Disabled");
        },
        [pClusteringConfig](int8_t delta) {
            NN_UNUSED(delta);
            pClusteringConfig->isExternalLightFilterEnabled = !pClusteringConfig->isExternalLightFilterEnabled;
        }
    );


    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_ClusteringProcessorState.samplingNumber, &m_ClusteringProcessorState.ambientNoiseLevel);
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);
    int8_t* pObjectCount = &m_ClusteringProcessorState.objectCount;
    m_ReadOnlyMenu.emplace_back("Object Count",
        [pObjectCount](std::stringstream& sstr) {
            sstr << static_cast<int>(*pObjectCount);
        },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

void ClusteringModeState::Start()
{
    nn::irsensor::RunClusteringProcessor(m_IrCameraHandle, m_ClusteringProcessorConfig);
}

void ClusteringModeState::Update()
{
    int count;
    nn::Result result = nn::irsensor::GetClusteringProcessorStates(&m_ClusteringProcessorState, &count, 1, m_IrCameraHandle);
    HandleResult(result);
    m_ClusteringProcessorState.objectCount = (m_ClusteringProcessorState.objectCount >= nn::irsensor::ClusteringProcessorObjectCountMax) ? 0 : m_ClusteringProcessorState.objectCount;
    m_Statistics.Update(&m_ClusteringProcessorState.samplingNumber, IrSensorModeStatistics::ExpectedClusteringModeFramerate);
}

namespace
{

void WriteClusteringProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::ClusteringProcessorState* pClusteringState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pClusteringState);

    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Green ={ { 0, 255, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    const float xStart = -0.2f, xEnd = -0.8f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 各面を描画
    for (auto i = 0; i < pClusteringState->objectCount; ++i)
    {
        const nn::irsensor::ClusteringData* pObject = &pClusteringState->objects[i];
        pPrimitiveRenderer->SetColor(Green);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * pObject->bound.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->bound.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * (pObject->bound.x + pObject->bound.width) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->bound.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (pObject->bound.x + pObject->bound.width) / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (pObject->bound.y + pObject->bound.height) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * pObject->bound.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (pObject->bound.y + pObject->bound.height) / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * pObject->bound.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->bound.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}


}

void ClusteringModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WriteClusteringProcessorState(pPrimitiveRenderer, pCommandBuffer, &m_ClusteringProcessorState, index);
}
