﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "IrSensorModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

namespace
{

const char* AmbientNoiseLevel[] =
{
    "Low",
    "Middle",
    "High",
    "Unknown",
};

}

MenuItem::MenuItem(const char* name, MenuItem::ReaderType reader, MenuItem::WriterType writer)
    : m_Name(name)
    , m_Reader(reader)
    , m_Writer(writer)
{
}

void MenuItem::Increment(int8_t delta) const
{
    m_Writer(delta);
}

void MenuItem::Read(std::stringstream& sstr) const
{
    m_Reader(sstr);
}

const char* MenuItem::GetName() const
{
    return m_Name.c_str();
}

const float IrSensorModeStatistics::ExpectedMomentModeFramerate = 70.0f;
const float IrSensorModeStatistics::ExpectedClusteringModeFramerate = 70.0f;
const float IrSensorModeStatistics::ExpectedHandAnalysisFramerate = 66.6f;

void IrSensorModeStatistics::Update(const int64_t* pCurrentSamplingNumber, const float ExpectedFrameRate)
{
    // 3 秒ごとに Packet Loss 向けデータを初期化します
    if ( ( m_FramerateCounter % ResetIntervalsInFrame ) == 0)
    {
        m_FramerateFirstTick = nn::os::GetSystemTick();
        m_FramerateFirstSample = *pCurrentSamplingNumber;
    }
    nn::os::Tick currentTick = nn::os::GetSystemTick() - m_FramerateFirstTick;
    int64_t currentSample = *pCurrentSamplingNumber - m_FramerateFirstSample;

    // UpdateIntervalsInFrame 周期ごとに、Packet Loss を更新します
    if ( m_FramerateCounter % UpdateIntervalsInFrame == ( UpdateIntervalsInFrame - 1 ) )
    {
        m_FramerateComputation = 1000.0f * currentSample / nn::os::ConvertToTimeSpan(currentTick).GetMilliSeconds();
        m_PacketDropPercentage = 1.0f - m_FramerateComputation / ExpectedFrameRate;
        m_PacketDropPercentage = std::min(std::max(m_PacketDropPercentage, 0.0f), 1.0f);
    }
    m_FramerateCounter++;
}

IrSensorModeState::IrSensorModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : m_IrCameraHandle(irCameraHandle)
{
    m_ReadWriteMenu.emplace_back("Current Mode",
        [pNextProcessor](std::stringstream& sstr) {
            switch (*pNextProcessor)
            {
            case IrSensorMode_PositionReader:
                sstr << "Position Reader";
                break;
            default:
                NN_ASSERT(false);
            }
        },
        [pNextProcessor, pMenuSelection](int8_t delta) {
            int8_t mode = static_cast<int8_t>(*pNextProcessor) + delta;
            *pNextProcessor = static_cast<IrSensorMode>(std::min(std::max(mode, static_cast<int8_t>(IrSensorMode_PositionReader)), static_cast<int8_t>(IrSensorMode_Count - 1)));
            *pMenuSelection = 0;
        }
    );
}
void IrSensorModeState::AddCommonReadWriteMenu(std::vector<MenuItem>* pMenu, ::nn::irsensor::IrCameraConfig* pConfig, ::nn::TimeSpanType exposureMin, ::nn::TimeSpanType exposureMax)
{
    pMenu->emplace_back("Exposure Time",
        [pConfig](std::stringstream& sstr) {
            sstr << pConfig->exposureTime.GetMicroSeconds();
        },
        [pConfig, exposureMin, exposureMax](int8_t delta) {
            pConfig->exposureTime += ::nn::TimeSpanType::FromMicroSeconds(delta * 10);
            pConfig->exposureTime = std::min(std::max(pConfig->exposureTime, exposureMin), exposureMax);
        }
    );
    pMenu->emplace_back("Light Target",
        [pConfig](std::stringstream& sstr) {
            switch (pConfig->lightTarget)
            {
            case nn::irsensor::IrCameraLightTarget_AllObjects:
                sstr << "AllObjects";
                break;
            case nn::irsensor::IrCameraLightTarget_FarObjects:
                sstr << "FarObjects";
                break;
            case nn::irsensor::IrCameraLightTarget_NearObjects:
                sstr << "NearObjects";
                break;
            case nn::irsensor::IrCameraLightTarget_None:
                sstr << "None";
                break;
            default:
                break;
            }
        },
        [pConfig](int8_t delta) {
            int8_t lightTarget = static_cast<int8_t>(pConfig->lightTarget) + static_cast<int8_t>(delta);
            lightTarget = std::max(std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)), static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects));
            pConfig->lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
        }
    );
    pMenu->emplace_back("Digital Gain",
        [pConfig](std::stringstream& sstr) {
            sstr << pConfig->gain;
        },
        [pConfig](int8_t delta) {
            pConfig->gain += delta;
            pConfig->gain = std::min(std::max(pConfig->gain, nn::irsensor::IrCameraGainMin), nn::irsensor::IrCameraGainMax);
        }
    );
    pMenu->emplace_back("Negative Image",
        [pConfig](std::stringstream& sstr) {
            sstr << (pConfig->isNegativeImageUsed ? "ON" : "OFF");
        },
        [pConfig](int8_t delta) {
            NN_UNUSED(delta);
            pConfig->isNegativeImageUsed = !pConfig->isNegativeImageUsed;
        }
    );
}

void IrSensorModeState::AddCommonReadOnlyMenu(std::vector<MenuItem>* pMenu, int64_t* pSamplingNumber, nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel)
{
    pMenu->emplace_back("Sampling Number",
        [pSamplingNumber](std::stringstream& sstr) {
        sstr << *pSamplingNumber;
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
    pMenu->emplace_back("Ambient Noise Level",
        [pAmbientNoiseLevel](std::stringstream& sstr) {
        sstr << AmbientNoiseLevel[*pAmbientNoiseLevel];
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

void IrSensorModeState::AddStatisticsMenu(std::vector<MenuItem>* pMenu, float* pPacketLoss)
{
    pMenu->emplace_back("Packet Loss",
        [pPacketLoss](std::stringstream& sstr) {
        sstr << (*pPacketLoss) * 100.0f << "%";
    },
        [](int8_t delta) { NN_UNUSED(delta); }
    );
}

const std::vector<MenuItem>& IrSensorModeState::GetReadWriteMenu()
{
    return m_ReadWriteMenu;
}

const std::vector<MenuItem>& IrSensorModeState::GetReadOnlyMenu()
{
    return m_ReadOnlyMenu;
}

void IrSensorModeState::Stop()
{
}

void IrSensorModeState::HandleResult(nn::Result result)
{
    if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
    {
        // NotReady のエラーが返る場合は、データの準備中ですので、正しいデータが来るまでリトライします。
        return;
    }
    else if (nn::irsensor::ResultIrsensorUnavailable::Includes(result))
    {
        // コントローラにモーションIRカメラが非搭載の場合は、
        // ユーザに適切なコントローラを接続するよう案内します。
        return;
    }
    else if (nn::irsensor::ResultIrsensorUnconnected::Includes(result))
    {
        // コントローラが非接続の場合は、ユーザに案内し、接続されるまでリトライすることを推奨します。
        return;
    }
    else if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
    {
        // DeviceError が発生した場合は、モーションIRカメラは使用できませんので、一度 プロセッサを停止します。
        // この場合、詳細なエラー内容を示したエラービューア、またはコントローラのアップデートが表示されるため、
        // ユーザは問題の解決を試みます。
        //Stop();
        //Start();
        return;
    }

}
