﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "ImguiDemo_SixAxisSensorPointer.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

namespace {

template <typename T>
T Clamp(T value, T min, T max)
{
    if (value < min)
    {
        value = min;
    }
    else if (max < value)
    {
        value = max;
    }

    return value;
}

template <typename T>
T LinearInterpolation(T x1, T x2, T coefficient)
{
    return (x1 * (1 - coefficient)) + (x2 * coefficient);
}

// バイアス
const float DistanceX = 6.0f;
const float DistanceY = 5.0f;

} // namespace

SixAxisSensorPointer::SixAxisSensorPointer() NN_NOEXCEPT
: m_Width(0.0f)
, m_Height(0.0f)
, m_RotationBase()
, m_CursorOffset()
{
}

void SixAxisSensorPointer::Initialize(float width, float height)
{
    m_Width = width;
    m_Height = height;

    m_CursorCenterX = width / 2.0f;
    m_CursorCenterY = height / 2.0f;
}

void SixAxisSensorPointer::Finalize()
{
}

void SixAxisSensorPointer::Update(const nn::hid::DirectionState& direction) NN_NOEXCEPT
{
    // 画面座標に変換
    ::nn::util::Vector3f pointingDirection;
    ::nn::util::VectorLoad(&pointingDirection, direction.y);
    ::nn::util::Vector3f rotatedPointingDirection = m_RotationBase.Transform(pointingDirection);

    float x = 0.0f;
    float y = 0.0f;

    {
        float cosHorizontalAngle = rotatedPointingDirection.GetX();
        x = cosHorizontalAngle;
        x = x * DistanceX;
    }
    {
        float cosVerticalAngle = rotatedPointingDirection.GetZ();
        y = cosVerticalAngle;
        y = y * DistanceY;
    }

    m_CursorOffset.x = LinearInterpolation(0.0f, m_Width * 0.5f, x);
    m_CursorOffset.y = LinearInterpolation(0.0f, -m_Height * 0.5f, y);
}

void SixAxisSensorPointer::Reset(
    const nn::hid::DirectionState& direction,
    float cursorCenterX, float cursorCenterY) NN_NOEXCEPT
{
    nn::util::Matrix4x3f mat = nn::util::Matrix4x3f::Identity();

    nn::util::Vector3f rowX;
    nn::util::Vector3f rowY;
    nn::util::Vector3f rowZ;

    nn::util::VectorLoad(&rowX, direction.x);
    nn::util::VectorLoad(&rowY, direction.y);
    nn::util::VectorLoad(&rowZ, direction.z);

    mat.SetAxisX(rowX);
    mat.SetAxisY(rowY);
    mat.SetAxisZ(rowZ);
    mat.Transpose(); // Transpose <-> Invert

    m_RotationBase = mat;

    m_CursorOffset.x = 0.0f;
    m_CursorOffset.y = 0.0f;

    m_CursorCenterX = cursorCenterX;
    m_CursorCenterY = cursorCenterY;
}

::nn::util::Vector3f SixAxisSensorPointer::GetCursor() const NN_NOEXCEPT
{
    ::nn::util::Vector3f pointer;

    pointer.SetX(m_CursorCenterX + m_CursorOffset.x);
    pointer.SetY(m_CursorCenterY + m_CursorOffset.y);
    return pointer;
}


#endif // defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
