﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ImguiDemo_IrSensor.h"

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)

#include <nn/nn_Assert.h>
#include <nn/irsensor.h>


NN_STATIC_ASSERT(ImageBufferSize == NN_ARRAY_SIZE(nn::irsensor::HandAnalysisImageState::image));

namespace
{
bool g_IsIrSensorInitialized = false;
nn::irsensor::IrCameraHandle g_IrCameraHandle = {};

const int HandAnalysisStateCount = 1;
nn::irsensor::HandAnalysisImageState g_HandAnalysisImageState[HandAnalysisStateCount];
nn::irsensor::HandAnalysisSilhouetteState g_HandAnalysisSilhouetteState[HandAnalysisStateCount];
}

void InitializeIrSensor(const ::nn::hid::NpadIdType& npadIdType)
{
    NN_ASSERT(!g_IsIrSensorInitialized);

    nn::Result result;
    g_IrCameraHandle = nn::irsensor::GetIrCameraHandle(npadIdType);
    nn::irsensor::Initialize(g_IrCameraHandle);

    nn::irsensor::HandAnalysisConfig handAnalysisConfig;
    handAnalysisConfig.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;

    result = nn::irsensor::RunHandAnalysis(g_IrCameraHandle, handAnalysisConfig);
    NN_ASSERT(result.IsSuccess());

    g_IsIrSensorInitialized = true;
}

void FinalizeIrSensor()
{
    NN_ASSERT(g_IsIrSensorInitialized);

    nn::irsensor::Finalize(g_IrCameraHandle);

    g_IsIrSensorInitialized = false;
}

void UpdateIrSensor()
{
    nn::Result result;
    int returnCount = 0;

    result = nn::irsensor::GetHandAnalysisImageState(
        &g_HandAnalysisImageState[0],
        &returnCount,
        HandAnalysisStateCount,
        0,
        g_IrCameraHandle);
    NN_ASSERT(
        ((!result.IsSuccess()) && (returnCount == 0))
        || ((result.IsSuccess()) && (returnCount > 0)));

    result = nn::irsensor::GetHandAnalysisSilhouetteState(
        &g_HandAnalysisSilhouetteState[0],
        &returnCount,
        HandAnalysisStateCount,
        0,
        g_IrCameraHandle);
    NN_ASSERT(
        ((!result.IsSuccess()) && (returnCount == 0))
        || ((result.IsSuccess()) && (returnCount > 0)));
}


int GetIrSensorHandCount()
{
    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState =
        &g_HandAnalysisSilhouetteState[0];
    return pHandAnalysisSilhouetteState->handCount;
}

int GetIrSensorHandFingerCount(int handIndex)
{
    NN_ASSERT(handIndex < nn::irsensor::IrHandAnalysisHandCountMax);

    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState =
        &g_HandAnalysisSilhouetteState[0];

    int fingerCount = -1;

    if (handIndex < pHandAnalysisSilhouetteState->handCount)
    {
        nn::irsensor::Hand* pHand = &pHandAnalysisSilhouetteState->hands[handIndex];

        fingerCount = 0;

        for (int fingerIndex = 0; fingerIndex < nn::irsensor::HandFinger_Count; ++fingerIndex)
        {
            if (pHand->fingers[fingerIndex].isValid)
            {
                fingerCount++;
            }
        }

        for (int fingerTouchingIndex = 0;
            fingerTouchingIndex < nn::irsensor::HandTouchingFingers_Count;
            ++fingerTouchingIndex)
        {
            if (pHand->areFingersTouching[fingerTouchingIndex])
            {
                fingerCount++;
            }
        }
    }

    return fingerCount;
}



IrSensorHandInput GetIrSensorHandInput()
{
    IrSensorHandInput result = IrSensorHandInput_None;

    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState =
        &g_HandAnalysisSilhouetteState[0];
    int handCount = pHandAnalysisSilhouetteState->handCount;

    for (int handIndex = 0; handIndex < handCount; ++handIndex)
    {
        int fingerCount = GetIrSensorHandFingerCount(handIndex);

        if (fingerCount == 0)
        {
            result = IrSensorHandInput_Validate;
        }
        else if (fingerCount == 5)
        {
            result = IrSensorHandInput_Cancel;
        }
    }

    return result;
}


void CopyIrSensorImageR16(void* pImageBuffer, ptrdiff_t imageBufferPitch, size_t imageBufferSize)
{
    NN_ASSERT(imageBufferSize >= sizeof(nn::irsensor::HandAnalysisImageState::image));

    uint8_t* pDestinationBuffer = reinterpret_cast<uint8_t*>(pImageBuffer);
    const uint16_t* pSourceBuffer = g_HandAnalysisImageState[0].image;

    for (int line = 0; line < nn::irsensor::IrHandAnalysisImageHeight; ++line)
    {
        memcpy(
            pDestinationBuffer, pSourceBuffer,
            sizeof(uint16_t) * nn::irsensor::IrHandAnalysisImageWidth);

        pDestinationBuffer += imageBufferPitch;
        pSourceBuffer += nn::irsensor::IrHandAnalysisImageWidth;
    }
}

#endif





