﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Log.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

const size_t KIBI = 1024;
const size_t MEBI = 1024 * KIBI;
const size_t GIBI = 1024 * MEBI;

size_t findMaxMallocSize(size_t minimum, size_t maximum)
{
    size_t allocSize = 0;
    size_t minBoundary = minimum;
    size_t maxBoundary = maximum;

    while (minBoundary < maxBoundary)
    {
        allocSize = (minBoundary + maxBoundary) / 2;
        NN_LOG("Attempting to allocate %llu bytes... ", allocSize);

        void *memHandle = malloc(allocSize);
        if (memHandle != NULL)
        {
            NN_LOG("succeeded. (0x%p)\n", memHandle);
            minBoundary = allocSize + 1;
        }
        else
        {
            NN_LOG("failed.\n");
            maxBoundary = allocSize;
        }

        free(memHandle);
    }

    return allocSize - 1;
}

void PrintAllocData(nn::gfx::util::DebugFontTextWriter &fontWriter, unsigned int frame, size_t allocSize)
{
    NN_UNUSED(frame);
    double allocInGibibytes = static_cast<double>(allocSize) / static_cast<double>(GIBI);

    const nn::util::Color4u8Type lightBlue = {{ 173, 216, 230, 255 }};

    fontWriter.SetScale(3.0f, 3.0f);
    fontWriter.SetTextColor(lightBlue);
    fontWriter.SetCursor(300, 200);
    fontWriter.Print("%0.2f GiB\n", allocInGibibytes);

    fontWriter.SetScale(1.5f, 1.5f);
    fontWriter.Print("%llu bytes\n", allocSize);
}

struct CommandUserData
{
    size_t allocSize;
    nn::gfx::util::DebugFontTextWriter *fontWriter;
    unsigned int frameCount;
};

void CommandGenerator(nns::gfx::GraphicsFramework *framework, int bufferIndex, void *userData)
{
    CommandUserData *data = reinterpret_cast<CommandUserData*>(userData);

    // "Print" to the font writer
    PrintAllocData(*data->fontWriter, data->frameCount, data->allocSize);

    framework->BeginFrame(bufferIndex);

    nn::gfx::ColorTargetView *target  = framework->GetColorTargetView();
    nn::gfx::CommandBuffer *cmdBuffer = framework->GetRootCommandBuffer(bufferIndex);

    // Set up a rendering state
    cmdBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
    cmdBuffer->SetRenderTargets(1, &target, nullptr);
    cmdBuffer->SetViewportScissorState(framework->GetViewportScissorState());

    // Draw text onto the buffer
    data->fontWriter->Draw(cmdBuffer);

    framework->EndFrame(bufferIndex, true);
}

extern "C"
int nnMain(int argc, char *argv[])
{
    NN_UNUSED(argc);
    NN_UNUSED(argv);

    // Find our maximum memory size
    size_t maxAllocation = findMaxMallocSize(1, 8 * GIBI);

#if defined(NN_SDK_BUILD_DEVELOP)
    NN_LOG("##teamcity[buildStatisticValue key='Maximum Allocation Size (Develop)' value='%llu']\n", maxAllocation);
#elif defined(NN_SDK_BUILD_RELEASE)
    NN_LOG("##teamcity[buildStatisticValue key='Maximum Allocation Size (Release)' value='%llu']\n", maxAllocation);
#elif defined(NN_SDK_BUILD_DEBUG)
    NN_LOG("##teamcity[buildStatisticValue key='Maximum Allocation Size (Debug)' value='%llu']\n", maxAllocation);
#endif

    // Initialize the core graphics system
    const size_t gfxMemorySize = 8 * MEBI;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(gfxMemorySize);

    // Set up the sample graphics framework
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // Configure the debug font writer
    nn::gfx::util::DebugFontTextWriterInfo fontInfo;
    fontInfo.SetDefault();
    fontInfo.SetCharCountMax(64);
    fontInfo.SetBufferCount(2);
    fontInfo.SetUserMemoryPoolEnabled(true);

    // Allocate heap for the font
    size_t fontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(gfw.GetDevice(), fontInfo);
    nn::util::BytePtr fontHeap(new unsigned char[fontHeapSize]);

    // Generate font memory pool
    size_t fontPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(gfw.GetDevice(), fontInfo);
    nn::gfx::MemoryPool *memPool = gfw.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
    ptrdiff_t memPoolOffset = gfw.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, fontPoolSize, 1);

    // Create the debug font writer
    nn::gfx::util::DebugFontTextWriter fontWriter;
    fontWriter.Initialize(gfw.GetDevice(), fontInfo, fontHeap.Get(), fontHeapSize, memPool, memPoolOffset, fontPoolSize);

    // Allocate descriptors for texture and sampler
    int texDescriptor = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samDescriptor = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    // Assign descriptors to the font writer
    fontWriter.SetDisplayWidth(gfw.GetDisplayWidth());
    fontWriter.SetDisplayHeight(gfw.GetDisplayHeight());
    fontWriter.SetTextureDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), texDescriptor);
    fontWriter.SetSamplerDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samDescriptor);

    // Configure a command builder
    CommandUserData userData;
    userData.allocSize = maxAllocation;
    userData.fontWriter = &fontWriter;
    gfw.SetMakeCommandCallback(CommandGenerator, &userData);

    // Spin for 15 seconds
    for (int frame = 0; frame < 15 * 60; ++frame)
    {
        userData.frameCount = frame;
        gfw.ProcessFrame();
    }
    gfw.QueueFinish();

    // Clean up and exit
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samDescriptor);
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, texDescriptor);

    fontWriter.Finalize();

    gfw.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memPoolOffset);

    delete[] reinterpret_cast<unsigned char*>(fontHeap.Get());
    fontHeap.Reset(nullptr);

    gfw.Finalize();

    return 0;
}
