﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace Nintendo.Htcs
{
    // HTCS のポートと WinSock のポートの対応
    public class PortMapItem
    {
        public PortMapItem(HtcsPortDescriptor htcsPortDescriptor, IPEndPoint endPoint)
        {
            this.HtcsPortDescriptor = htcsPortDescriptor;
            this.EndPoint = endPoint;
        }

        public HtcsPortDescriptor HtcsPortDescriptor { get; private set; }
        public IPEndPoint EndPoint { get; private set; }

        internal const string XElementName = "PortMapItem";

        internal PortMapItem(XElement xElement)
        {
            this.HtcsPortDescriptor = new HtcsPortDescriptor(xElement);

            var address = IPAddress.Parse(xElement.Element("IPAddress").Value);
            var port = int.Parse(xElement.Element("TcpPortNumber").Value, CultureInfo.InvariantCulture);
            this.EndPoint = new IPEndPoint(address, port);
        }

        internal XElement ToXElement()
        {
            return new XElement(XElementName,
                HtcsPortDescriptor.HtcsPeerName.ToXElement(),
                new XElement("HtcsPortName", HtcsPortDescriptor.HtcsPortName),
                new XElement("IPAddress", EndPoint.Address.ToString()),
                new XElement("TcpPortNumber", EndPoint.Port.ToString(CultureInfo.InvariantCulture))
            );
        }

        public override string ToString()
        {
            return string.Format(CultureInfo.InvariantCulture, "{0}, {1}", HtcsPortDescriptor, EndPoint);
        }
    }

    public struct HtcsPortDescriptor : IEquatable<HtcsPortDescriptor>, IComparable<HtcsPortDescriptor>, IComparable
    {
        public HtcsPortDescriptor(HtcsPeerName htcsPeerName, string htcsPortName)
            : this()
        {
            this.HtcsPeerName = htcsPeerName;
            this.HtcsPortName = htcsPortName;
        }

        public HtcsPeerName HtcsPeerName { get; private set; }
        public string HtcsPortName { get; private set; }

        public override bool Equals(object obj)
        {
            if (!(obj is HtcsPortDescriptor))
            {
                return false;
            }

            return ((IEquatable<HtcsPortDescriptor>)(this)).Equals((HtcsPortDescriptor)obj);
        }

        public override int GetHashCode()
        {
            return this.HtcsPeerName.GetHashCode() ^ this.HtcsPortName.GetHashCode();
        }

        public override string ToString()
        {
            return string.Format(
                CultureInfo.InvariantCulture, "{0}:{1}", this.HtcsPeerName.ToString(), this.HtcsPortName);
        }

        public bool Equals(HtcsPortDescriptor other)
        {
            return this.HtcsPeerName.Equals(other.HtcsPeerName) && this.HtcsPortName.Equals(other.HtcsPortName); ;
        }

        public static bool operator==(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.Equals(rhs);
        }

        public static bool operator !=(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return !lhs.Equals(rhs);
        }

        public static bool operator <(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.CompareTo(rhs) < 0;
        }

        public static bool operator >(HtcsPortDescriptor lhs, HtcsPortDescriptor rhs)
        {
            return lhs.CompareTo(rhs) > 0;
        }

        public int CompareTo(HtcsPortDescriptor other)
        {
            var c1 = this.HtcsPeerName.CompareTo(other.HtcsPeerName);
            if (c1 != 0)
            {
                return c1;
            }

            return string.CompareOrdinal(this.HtcsPortName, other.HtcsPortName);
        }

        public int CompareTo(object obj)
        {
            if (!(obj is HtcsPortDescriptor))
            {
                throw new ArgumentException("obj must be a HtcsPortDescriptor", "obj");
            }

            return ((IComparable<HtcsPortDescriptor>)(this)).CompareTo((HtcsPortDescriptor)obj);
        }

        internal HtcsPortDescriptor(XElement xElement)
            : this()
        {
            this.HtcsPeerName = new HtcsPeerName(xElement.Element(HtcsPeerName.XElementName));
            this.HtcsPortName = xElement.Element("HtcsPortName").Value;
        }
    }

    public struct HtcsPeerName : IEquatable<HtcsPeerName>, IComparable<HtcsPeerName>, IComparable
    {
        // ホストが指定した場合、どのターゲットからの接続も受け付けることを示す
        public static readonly HtcsPeerName Any = new HtcsPeerName(string.Empty);

        public HtcsPeerName(string idValue)
            : this()
        {
            this.Value = idValue;
        }

        public string Value { get; private set; }

        public override bool Equals(object obj)
        {
            return this.Value.Equals(obj);
        }

        public override int GetHashCode()
        {
            return this.Value.GetHashCode();
        }

        public override string ToString()
        {
            return this.Value.ToString();
        }

        public bool Equals(HtcsPeerName other)
        {
            return this.Value.Equals(other.Value);
        }

        public static bool operator ==(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.Equals(rhs);
        }

        public static bool operator !=(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return !lhs.Equals(rhs);
        }

        public static bool operator <(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.CompareTo(rhs) < 0;
        }

        public static bool operator >(HtcsPeerName lhs, HtcsPeerName rhs)
        {
            return lhs.CompareTo(rhs) > 0;
        }

        public int CompareTo(HtcsPeerName other)
        {
            return string.CompareOrdinal(this.Value, other.Value);
        }

        public int CompareTo(object obj)
        {
            if (!(obj is HtcsPeerName))
            {
                throw new ArgumentException("obj must be a HtcsPeerName", "obj");
            }

            return this.CompareTo((HtcsPeerName)obj);
        }

        internal const string XElementName = "HtcsPeerName";

        internal HtcsPeerName(XElement xElement)
            : this()
        {
            this.Value = xElement.Value;    // TODO: validate
        }

        internal XElement ToXElement()
        {
            return new XElement(XElementName, Value);
        }
    }
}
