﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidVibrationBasic.cpp,PageSampleHidVibrationBasic}

    @brief
    シンプルに振動子を制御するためのサンプルプログラム
 */

/**
    @page PageSampleHidVibrationBasic シンプルな振動子制御
    @tableofcontents

    @brief
    シンプルに振動子を制御するためのサンプルプログラムの解説です。

    @section PageSampleHidVibrationBasic_SectionBrief 概要
    コントローラの振動子を制御する方法の説明をします。

    @section PageSampleHidVibrationBasic_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidVibrationBasic @endlink 以下にあります。

    @section PageSampleHidVibrationBasic_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。
    1 台以上のコントローラを接続する必要があります。

    @section PageSampleHidVibrationBasic_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    接続後、コントローラのボタンを押している間、振動子に振動値が送信され、コントローラが振動します。
    右振動子は A 、B 、X 、Y 、L 、ZL のボタン、左振動子は上、下、左、右、R 、ZR のボタンを押すことで振動します。
    送信される振動値は各コントローラで同時に押しているボタンの数に応じて変わります。
    振動はボタンを押しているコントローラでのみ発生します。ボタンを押していないコントローラでは振動しません。
    - 0 個：振動なし
    - 1 個：周波数 160 Hz で振幅が 1.00 と 0.00 に交互に切り替わる振動が発生します
    - 2 個：周波数 160 Hz で振幅が動的に変化する振動を発生します
    - 3 個：振幅 0.50 周波数が 160 Hz と 320 Hz に交互に切り替わる振動が発生します
    - それ以上：振幅が 0.50 で周波数が動的に変化する振動が発生します
    画面に、実際に発生している振動値が出力されます。

    なお、「 nn::hid::SendVibrationValue 関数で設定した振動値」と「 nn::hid::GetActualVibrationValue 関数で取得した振動値」が必ずしも一致しない背景については、
    NintendoSDK ドキュメントの「 API リファレンス 」以下の nn::hid::GetActualVibrationValue 関数の項目をご覧ください。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。

    @section PageSampleHidVibrationBasic_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。

    @section PageSampleHidVibrationBasic_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidVibrationBasic_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を初期化
    - NpadID から振動子のハンドルを取得
    - 振動子の情報を取得
    - 振動子に振動値を送信
    - 振動子で発生している実際の振動値を取得
 */

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include "GraphicsSystem.h"
#include "VibrationValueDrawer.h"
#include "NpadController.h"

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include "NpadJoyController.h"
#endif

namespace
{
    const int FrameRate = 60;
    const char ProgramName[] = "HidVibrationBasic Sample";

    const nn::util::Unorm8x4 TextColor = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 ActiveColor = { { 255, 255, 255, 255 } };
    const nn::util::Unorm8x4 DeactiveColor = { { 32, 32, 32, 255 } };

    GraphicsSystem* g_pGraphicsSystem;
    nn::mem::StandardAllocator* g_pAppAllocator;
    nn::Bit8* g_pAppMemory;

    void InitializeGraphics() NN_NOEXCEPT
    {
        // Memory
        g_pAppAllocator = new nn::mem::StandardAllocator();
        const size_t appMemorySize = 128 * 1024 * 1024;
        g_pAppMemory = new nn::Bit8[appMemorySize];
        g_pAppAllocator->Initialize(g_pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        const size_t graphicsMemorySize = 256 * 1024 * 1024;
        void* pGraphicsMemory = nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);
        nv::SetGraphicsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
        nv::SetGraphicsDevtoolsAllocator(nns::gfx::GraphicsFramework::DefaultAllocateFunction, nns::gfx::GraphicsFramework::DefaultFreeFunction, nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);
        nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif
        // Graphics
        g_pGraphicsSystem = new ::GraphicsSystem();
        g_pGraphicsSystem->SetApplicationHeap(g_pAppAllocator);
        g_pGraphicsSystem->Initialize();
    }

    void DrawTimeSeriesGraph(NpadControllerBase* p, int ctrlIdx, bool isLow) NN_NOEXCEPT
    {
        int vibrationDeviceCount = p->GetVibrationDeviceCount();

        float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
        float baseY = 50.0f + 160.0f * (ctrlIdx % 4);

        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        pTextWriter->SetTextColor(p->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("%s : Time Series amplitude%s value", p->GetName(), isLow ? "Low" : "High");

        for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
        {
            const VibrationState& v = p->GetVibrationState(vibIdx);

            float x = baseX + 20.0f + 280.0f * vibIdx;
            float y = baseY + 25.0f;

            pTextWriter->SetTextColor(p->IsConnected() ? ActiveColor : DeactiveColor);
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(x, y);
            switch (v.deviceInfo.position)
            {
            case nn::hid::VibrationDevicePosition_Left:
                pTextWriter->Print("Left");
                break;
            case nn::hid::VibrationDevicePosition_Right:
                pTextWriter->Print("Right");
                break;
            default:
                pTextWriter->Print("Unknown");
                break;
            }

            VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
            const VibrationValueBuffer& buffer = p->GetVibrationValueBuffer(vibIdx);
            vibDrawer.SetPosition(x, y + 15.0f);
            vibDrawer.SetBrightColor(p->IsConnected());
            vibDrawer.DrawTimeSeriesGraph(buffer, isLow);
        }
    }

    void DrawVibrationValue(NpadControllerBase* p, int ctrlIdx) NN_NOEXCEPT
    {
        int vibrationDeviceCount = p->GetVibrationDeviceCount();

        float baseX = 30.0f + 600.0f * (ctrlIdx / 4);
        float baseY = 50.0f + 160.0f * (ctrlIdx % 4);

        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        pTextWriter->SetTextColor(p->IsConnected() ? ActiveColor : DeactiveColor);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("%s", p->GetName());

        for (int vibIdx = 0; vibIdx < vibrationDeviceCount; vibIdx++)
        {
            const VibrationState& v = p->GetVibrationState(vibIdx);

            float x = baseX + 20.0f + 280.0f * vibIdx;
            float y = baseY + 25.0f;

            pTextWriter->SetTextColor(p->IsConnected() ? ActiveColor : DeactiveColor);
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(x, y);
            switch (v.deviceInfo.position)
            {
            case nn::hid::VibrationDevicePosition_Left:
                pTextWriter->Print("Left");
                break;
            case nn::hid::VibrationDevicePosition_Right:
                pTextWriter->Print("Right");
                break;
            default:
                pTextWriter->Print("Unknown");
                break;
            }
            pTextWriter->Print(" (Pattern=%d)", v.vibrationPatternId);

            VibrationValueDrawer vibDrawer(pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
            vibDrawer.SetPosition(x, y + 15.0f);
            vibDrawer.SetScale(1.0f);
            vibDrawer.SetBrightColor(p->IsConnected());
            vibDrawer.DrawVibrationValue(v.actualVibrationValue, true);
        }
    }

    void UpdateGraphics(std::vector<NpadControllerBase*> controllers) NN_NOEXCEPT
    {
        g_pGraphicsSystem->BeginDraw();

        nn::gfx::util::DebugFontTextWriter* pTextWriter = &g_pGraphicsSystem->GetDebugFont();
        pTextWriter->Draw(&g_pGraphicsSystem->GetCommandBuffer());
        pTextWriter->SetScale(1.5f, 1.5f);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->SetCursor(10.0f, 10.0f);
        pTextWriter->Print("%s", ProgramName);

        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetTextColor(TextColor);
        pTextWriter->SetCursor(800.0f, 10.0f);
        pTextWriter->Print("If you push any button, Vibration will start working.\n");
        pTextWriter->Print("Push (+) and (-) Button to shutdown this application.");

        bool isFirstActiveDevice = true;
        int ctrlIdx = 0;

        for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
            it != controllers.end();
            ++it)
        {
            if ((*it)->IsConnected() == true)
            {
                DrawVibrationValue(*it, ctrlIdx++);
                if (isFirstActiveDevice)
                {
                    DrawTimeSeriesGraph(*it, ctrlIdx++, true);
                    DrawTimeSeriesGraph(*it, ctrlIdx++, false);
                    isFirstActiveDevice = false;
                }
            }
        }

        static int counter = 0;
        if (ctrlIdx == 0 && counter < 60)
        {
            pTextWriter->SetScale(2.0f, 2.0f);
            pTextWriter->SetTextColor(TextColor);
            pTextWriter->SetCursor(400.0f, 360.0f);
            pTextWriter->Print("Controller Not Connecting");
        }
        counter = counter % 90 + 1;

        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000L * 1000L * 1000L / FrameRate));
    }

    void FinalizeGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->Finalize();
        delete g_pGraphicsSystem;

        g_pAppAllocator->Finalize();
        delete g_pAppAllocator;

        delete[] g_pAppMemory;
    }

} // anonymous-namespace

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    InitializeGraphics();

    nn::hid::InitializeNpad();

    // 使用する操作形態を設定します
    nn::hid::SetSupportedNpadStyleSet(
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask
#else
        nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask
#endif
    );

    // 使用する Npad を設定します
    nn::hid::NpadIdType npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::Handheld
    };

    nn::hid::SetSupportedNpadIdType(npadIds, NN_ARRAY_SIZE(npadIds));

    // 管理対象の操作形態に関する初期化
    std::vector<NpadControllerBase*> controllers;
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    controllers.push_back(new NpadJoyDualController(nn::hid::NpadId::No1, "Id:1 JoyDual"));
    controllers.push_back(new NpadJoyDualController(nn::hid::NpadId::No2, "Id:2 JoyDual"));
    controllers.push_back(new NpadJoyDualController(nn::hid::NpadId::No3, "Id:3 JoyDual"));
    controllers.push_back(new NpadJoyDualController(nn::hid::NpadId::No4, "Id:4 JoyDual"));
#endif
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No1, "Id:1 FullKey"));
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No2, "Id:2 FullKey"));
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No3, "Id:3 FullKey"));
    controllers.push_back(new NpadFullKeyController(nn::hid::NpadId::No4, "Id:4 FullKey"));

    controllers.push_back(new NpadHandheldController(nn::hid::NpadId::Handheld, "Id:Handheld Handheld"));

    // 管理対象の操作形態に関する初期化
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        (*it)->Initialize();
    }

    bool isQuitRequired = false;

    while (!isQuitRequired)
    {
        for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
            it != controllers.end();
            ++it)
        {
            (*it)->Update();

            if ((*it)->IsQuitRequired())
            {
                isQuitRequired = true;
            }
        }

        UpdateGraphics(controllers);
    }

    // 管理対象の操作形態に関する削除
    for (std::vector<NpadControllerBase*>::iterator it = controllers.begin();
        it != controllers.end();
        ++it)
    {
        delete *it;
    }

    FinalizeGraphics();

    NN_LOG("%s Done\n", ProgramName);
}
