﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidSixAxisSensorCalibration_Main.cpp,PageSampleSixAxisSensorCalibration}
 *
 * @brief
 *  6軸センサーのユーザーキャリブレーションのサンプルプログラム
 */

/**
 * @page PageSampleSixAxisSensorCalibration 6軸センサーのユーザーキャリブレーション
 * @tableofcontents
 *
 * @brief
 *  6軸センサーのユーザーキャリブレーションのサンプルプログラムの解説です。
 *
 * @section PageSampleSixAxisSensorCalibration_SectionBrief 概要
 *  ここでは、6軸センサーのユーザーキャリブレーションのサンプルプログラムの説明をします。
 *
 *  6軸センサーのユーザーキャリブレーションの使い方については、Hid の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleSixAxisSensorCalibration_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidSixAxisSensorCalibration Samples/Sources/Applications/HidSixAxisSensorCalibration @endlink 以下にあります。
 *
 * @section PageSampleSixAxisSensorCalibration_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleSixAxisSensorCalibration_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleSixAxisSensorCalibration_SectionPrecaution 注意事項
 *  とくになし
 *
 * @section PageSampleSixAxisSensorCalibration_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleSixAxisSensorCalibration_SectionDetail 解説
 *
 * @subsection PageSampleSixAxisSensorCalibration_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HidSixAxisSensorCalibration_Main.cpp
 *  @includelineno HidSixAxisSensorCalibration_Main.cpp
 *
 * @subsection PageSampleSixAxisSensorCalibration_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - キャリブレーションするデバイスを選択する
 *  - UniquePad でデバイスを特定する
 *  - コントローラが静止するまで待つ
 *  - キャリブレーション処理を開始する
 *  - キャリブレーション中の状態をプリントする
 *
 *  本サンプルプログラムを起動してコントローラーを接続すると、キャリブレーションを行う 6 軸センサーの選択が開始されます。
 *  A ボタンか Right ボタンを押したデバイスが、キャリブレーションの対象デバイスとなります。
 *  対象のデバイスの静止判定が完了すると、キャリブレーションが開始します。
 *
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_Result.h>

#include "HidSixAxisSensorCalibration_Context.h"

extern "C" void nnMain()
{
    // サポートする NpadId
    const ::nn::hid::NpadIdType NpadIds[] = {
        ::nn::hid::NpadId::No1,
        ::nn::hid::NpadId::No2,
        ::nn::hid::NpadId::No3,
        ::nn::hid::NpadId::No4,
        ::nn::hid::NpadId::No5,
        ::nn::hid::NpadId::No6,
        ::nn::hid::NpadId::No7,
        ::nn::hid::NpadId::No8,
        ::nn::hid::NpadId::Handheld
    };

    // Npad の初期化
    nn::hid::InitializeNpad();

    nn::hid::SetSupportedNpadIdType(NpadIds,
                                    sizeof(NpadIds) / sizeof(NpadIds[0]));
    ::nn::hid::SetSupportedNpadStyleSet(::nn::hid::NpadStyleFullKey::Mask |
                                        ::nn::hid::NpadStyleJoyDual::Mask |
                                        ::nn::hid::NpadStyleJoyLeft::Mask |
                                        ::nn::hid::NpadStyleJoyRight::Mask);

    // イベントの登録
    ::nn::os::MultiWaitType multiWait;
    ::nn::os::SystemEventType connectionEvent;
    ::nn::os::MultiWaitHolderType connectionEventHolder;
    ::nn::os::TimerEventType timerEvent;
    ::nn::os::MultiWaitHolderType timerEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    // UniquePad の接続イベントを登録
    ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&connectionEventHolder,
                                        &connectionEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait,
                                  &connectionEventHolder);

    // タイマーイベントを登録
    ::nn::os::InitializeTimerEvent(&timerEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timerEventHolder,
                                        &timerEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait,
                                  &timerEventHolder);
    const int64_t IntervalMilliSeconds = 15;
    ::nn::os::StartPeriodicTimerEvent(&timerEvent,
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSeconds),
                                      ::nn::TimeSpan::FromMilliSeconds(IntervalMilliSeconds));

    bool runs = true;
    Context context;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        // UniquePad の接続イベントがシグナルされた
        if (pHolder == &connectionEventHolder)
        {
            ::nn::os::ClearSystemEvent(&connectionEvent);
            context.HandleConnection();
        }

        // タイマーイベントがシグナルされた
        if(pHolder == &timerEventHolder)
        {
            ::nn::os::ClearTimerEvent(&timerEvent);
            context.Update();
        }
    }
}
