﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidSimple_Main.cpp,PageSampleHidSimple}
 *
 * @brief
 *  nn::hid ライブラリの使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleHidSimple ヒューマンインターフェイスデバイスの状態取得
 * @tableofcontents
 *
 * @brief
 * nn::hid ライブラリを用いてヒューマンインターフェイスデバイスの状態を取得するサンプルプログラムの解説です。
 *
 * @section PageSampleHidSimple_SectionBrief 概要
 * ゲームパッド、デバッグコントローラ、タッチスクリーンの状態を表示します。
 *
 * @section PageSampleHidSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidSimple
 * Samples/Sources/Applications/HidSimple @endlink 以下にあります。
 *
 * @section PageSampleHidSimple_SectionNecessaryEnvironment 必要な環境
 * Windows 環境においては、 ゲームパッドの操作に XInput 互換のゲームパッドを、
 * デバッグコントローラの操作にキーボードを、 タッチスクリーンの操作にマウスを利用します。
 * ゲームパッドは最大で 4 台まで利用可能で、 5 台目はデバッグコントローラとして認識されます。
 * また、タッチメッセージが有効な環境においては、これもタッチスクリーンの入力として扱われます。
 *
 * @section PageSampleHidSimple_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとウィンドウが立ち上がります。
 * ウインドウ上にはそれぞれのデバイスの状態が表示されており、
 * ゲームパッドやデバッグコントローラを操作すると状態の変化が反映されます。
 * タッチスクリーンについては、ウインドウをタッチすると検出されたタッチの情報が表示されます。
 *
 * @section PageSampleHidSimple_SectionPrecaution 注意事項
 * コンソールウィンドウには何も表示されません。
 * キー割り当ての初期値は、 ゲームパッドは XInput 互換のゲームパッドを、
 * デバッグコントローラは ELECOM JC-W01UWH を前提とした値をとなっています。
 *
 * @section PageSampleHidSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleHidSimple_SectionDetail 解説
 * 処理の流れは nns::hid サンプルライブラリのソースコード中にコメントで示されています。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nns/hid.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "HidSimple_ApplicationHeap.h"
#include "HidSimple_Color.h"
#include "HidSimple_FontSystem.h"
#include "HidSimple_GraphicsSystem.h"
#include "HidSimple_WindowMessage.h"

namespace {

const size_t ApplicationHeapSize = 64 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

const nn::settings::DebugPadKeyboardMap DebugPadKeyboardMap =
{
    nn::hid::KeyboardKey::A::Index,
    nn::hid::KeyboardKey::B::Index,
    nn::hid::KeyboardKey::X::Index,
    nn::hid::KeyboardKey::Y::Index,
    nn::hid::KeyboardKey::L::Index,
    nn::hid::KeyboardKey::R::Index,
    0,
    0,
    nn::hid::KeyboardKey::P::Index,
    nn::hid::KeyboardKey::M::Index,
    nn::hid::KeyboardKey::LeftArrow::Index,
    nn::hid::KeyboardKey::UpArrow::Index,
    nn::hid::KeyboardKey::RightArrow::Index,
    nn::hid::KeyboardKey::DownArrow::Index,
};

void WriteGamePadState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                       const nns::hid::GamePad* const pGamePad,
                       const float offsetX,
                       const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (pGamePad == 0)
    {
        return;
    }

    const nn::util::Unorm8x4& textColor = pGamePad->IsConnected() ? Color::White
                                                                  : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("GamePad");

    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::A::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 245, offsetY + 77);
    pTextWriter->Print("a");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::B::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 225, offsetY + 100);
    pTextWriter->Print("b");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::X::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 225, offsetY + 55);
    pTextWriter->Print("x");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Y::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 205, offsetY + 77);
    pTextWriter->Print("y");

    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Start::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 145, offsetY + 60);
    pTextWriter->Print("start");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Select::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 75, offsetY + 60);
    pTextWriter->Print("select");

    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::ZL::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 30);
    pTextWriter->Print("ZL");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::ZR::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 230, offsetY + 30);
    pTextWriter->Print("ZR");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::L::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 35, offsetY + 30);
    pTextWriter->Print("L");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::R::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 210, offsetY + 30);
    pTextWriter->Print("R");

    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Up::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 25, offsetY + 60);
    pTextWriter->Print("|");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Down::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 25, offsetY + 100);
    pTextWriter->Print("|");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Left::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 80);
    pTextWriter->Print("--");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::Right::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 40, offsetY + 80);
    pTextWriter->Print("--");

    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::StickL::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 73 + 20 * pGamePad->GetLeftStick().x,
                           offsetY + 105 - 20 * pGamePad->GetLeftStick().y);
    pTextWriter->Print("*");
    pTextWriter->SetTextColor(pGamePad->HasAnyButtons(
        nns::hid::Button::StickR::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 180 + 20 * pGamePad->GetRightStick().x,
                           offsetY + 105 - 20 * pGamePad->GetRightStick().y);
    pTextWriter->Print("*");
    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 145);
    pTextWriter->Print("(%8.5f,%8.5f)", pGamePad->GetLeftStick().x,
                                        pGamePad->GetLeftStick().y);
    pTextWriter->SetCursor(offsetX + 135, offsetY + 145);
    pTextWriter->Print("(%8.5f,%8.5f)", pGamePad->GetRightStick().x,
                                        pGamePad->GetRightStick().y);
}

void WriteDebugPadState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                        const nns::hid::DebugPad* const pDebugPad,
                        const float offsetX,
                        const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (pDebugPad == 0)
    {
        return;
    }

    const nn::util::Unorm8x4& textColor =
        pDebugPad->IsConnected() ? Color::White : Color::Gray;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("DebugPad");

    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::A::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 245, offsetY + 77);
    pTextWriter->Print("a");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::B::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 225, offsetY + 100);
    pTextWriter->Print("b");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::X::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 225, offsetY + 55);
    pTextWriter->Print("x");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Y::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 205, offsetY + 77);
    pTextWriter->Print("y");

    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Start::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 145, offsetY + 60);
    pTextWriter->Print("start");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Select::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 75, offsetY + 60);
    pTextWriter->Print("select");

    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::ZL::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 30);
    pTextWriter->Print("ZL");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::ZR::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 230, offsetY + 30);
    pTextWriter->Print("ZR");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::L::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 35, offsetY + 30);
    pTextWriter->Print("L");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::R::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 210, offsetY + 30);
    pTextWriter->Print("R");

    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Up::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 25, offsetY + 60);
    pTextWriter->Print("|");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Down::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 25, offsetY + 100);
    pTextWriter->Print("|");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Left::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 80);
    pTextWriter->Print("--");
    pTextWriter->SetTextColor(pDebugPad->HasAnyButtons(
        nns::hid::Button::Right::Mask) ? Color::Orange : textColor);
    pTextWriter->SetCursor(offsetX + 40, offsetY + 80);
    pTextWriter->Print("--");

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetCursor(offsetX + 73 + 20 * pDebugPad->GetLeftStick().x,
                           offsetY + 105 - 20 * pDebugPad->GetLeftStick().y);
    pTextWriter->Print("*");
    pTextWriter->SetCursor(offsetX + 180 + 20 * pDebugPad->GetRightStick().x,
                           offsetY + 105 - 20 * pDebugPad->GetRightStick().y);
    pTextWriter->Print("*");
    pTextWriter->SetScale(0.7f, 0.7f);
    pTextWriter->SetCursor(offsetX + 5, offsetY + 145);
    pTextWriter->Print("(%8.5f,%8.5f)", pDebugPad->GetLeftStick().x,
                                        pDebugPad->GetLeftStick().y);
    pTextWriter->SetCursor(offsetX + 135, offsetY + 145);
    pTextWriter->Print("(%8.5f,%8.5f)", pDebugPad->GetRightStick().x,
                                        pDebugPad->GetRightStick().y);
}

void WriteTouchScreenState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                           const nns::hid::TouchScreen* const pTouchScreen,
                           const float offsetX,
                           const float offsetY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (pTouchScreen == 0)
    {
        return;
    }

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("TouchScreen");

    if (pTouchScreen->GetTouchStates().size() == 0)
    {
        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->SetCursor(offsetX + 5, offsetY + 30);
        pTextWriter->Print("Touch not found");
        return;
    }

    const std::vector<nns::hid::TouchScreen::TouchState
                      >& states = pTouchScreen->GetTouchStates();

    for (size_t i = 0; i < states.size(); ++i)
    {
        const nns::hid::TouchScreen::TouchState& state = states[i];
        pTextWriter->SetCursor(offsetX + 5,
                               offsetY + 30 * static_cast<float>(i + 1));
        pTextWriter->Print("id %08X (%4.0f, %4.0f)",
                           state.fingerId,
                           state.position.x,
                           state.position.y);
    }

    pTextWriter->SetTextColor(Color::Green);

    const float barOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
    const float barOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
    const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
    const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
    const float plusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
    const float plusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

    for (size_t i = 0; i < states.size(); ++i)
    {
        const nns::hid::TouchScreen::TouchState& state = states[i];
        pTextWriter->SetCursor(state.position.x - barOffsetX - 45,
                               state.position.y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(state.position.x - barOffsetX + 45,
                               state.position.y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(state.position.x - minusOffsetX,
                               state.position.y - minusOffsetY - 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(state.position.x - minusOffsetX,
                               state.position.y - minusOffsetY + 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(state.position.x - plusOffsetX - 45,
                               state.position.y - plusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(state.position.x - plusOffsetX - 45,
                               state.position.y - plusOffsetY + 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(state.position.x - plusOffsetX + 45,
                               state.position.y - plusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(state.position.x - plusOffsetX + 45,
                               state.position.y - plusOffsetY + 45);
        pTextWriter->Print("+");
    }
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault(&controllerManager);

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize(ApplicationHeapSize);

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;

        case WindowMessage_Active:
            nn::settings::SetDebugPadKeyboardMap(DebugPadKeyboardMap);
            break;

        case WindowMessage_Inactive:
            nn::settings::ResetDebugPadKeyboardMap();
            break;

        default:
            break;
        }

        if (quits)
        {
            break;
        }

        controllerManager.Update();

        for (int i = 0; i < 4; ++i)
        {
            WriteGamePadState(&textWriter,
                              reinterpret_cast<nns::hid::GamePad*>(
                                  controllerManager.GetController(
                                      nns::hid::ControllerId_GamePad, i)),
                              25, 15 + 175 * static_cast<float>(i));
        }

        WriteDebugPadState(&textWriter,
                           reinterpret_cast<nns::hid::DebugPad*>(
                               controllerManager.GetController(
                                   nns::hid::ControllerId_DebugPad, 0)),
                           645, 15);

        WriteTouchScreenState(&textWriter,
                              reinterpret_cast<nns::hid::TouchScreen*>(
                                  controllerManager.GetController(
                                      nns::hid::ControllerId_TouchScreen, 0)),
                              645, 200);

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
