﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_Palma.h>
#include <nn/hid/hid_Result.palma.h>
#include "HidNpadController.h"
#include "PalmaDeviceController.h"
#include "RomFileSystem.h"

namespace {
    NN_ALIGNAS(nn::os::MemoryPageSize) char g_FileReadBuffer[32 * 1024];    //!< 書き込む Activity の各種ファイルを読み込むためのバッファ

    //!< 書き込む RgbLedPattern ファイル
    const char* const PalmaRgbLedPatternFiles[SampleActivityPattern_Max] = {
        "rom:/RgbLed/Sample01.RgbLed.bin",
        "rom:/RgbLed/Sample02.RgbLed.bin"
    };

    //!< 書き込む Wave ファイル
    const char* const PalmaWaveFiles[SampleActivityPattern_Max] = {
        "rom:/Wave/Sample01.wav",
        "rom:/Wave/Sample02.wav"
    };

    enum ActivationTaskSequence
    {
        ActivationTaskSequence_None,
        ActivationTaskSequence_EnableStep,
        ActivationTaskSequence_Max
    };
    enum WriteTaskSequence
    {
        WriteTaskSequence_None,
        WriteTaskSequence_WriteRgbLedEntry,
        WriteTaskSequence_WriteWaveEntry,
        WriteTaskSequence_WriteActivityEntry,
        WriteTaskSequence_WriteDataBaseIdentificationVersion,
        WriteTaskSequence_Completed,
        WriteTaskSequence_Max
    };
    enum PropertyTaskSequence
    {
        PropertyTaskSequence_None,
        PropertyTaskSequence_GetStep,
        PropertyTaskSequence_GetUniqueCode,
        PropertyTaskSequence_GetDataBaseIdentificationVersion,
        PropertyTaskSequence_Max
    };

    //!< Palma のデバイス情報
    struct PalmaDevice
    {
        bool isValid;                              //!< 接続して有効な状態かどうか
        nn::os::SystemEventType operationEvent;    //!< Palma の操作完了を通知するイベント
        nn::hid::PalmaConnectionHandle handle;     //!< Palma の操作をするためのハンドル
        nn::hid::NpadIdType id;                    //!< NpadId
        nn::hid::PalmaFrModeType nextFrModeType;   //!< 設定する FrModeType
        SampleActivityPattern playActivityPattern; //!< 再生する対象の Activity
        SampleActivityPattern writeActivityPattern; //!< 書き込む対象の Activity
        ActivationTaskSequence activationTask;     //!< アクティベートでのタスクシーケンス
        WriteTaskSequence writeTask;               //!< 書き込みのタスクシーケンス
        PropertyTaskSequence propertyTask;         //!< 情報取得のタスクシーケンス
    };

    const int32_t SampleDatabaseVersion = 0x80000000;    //!< サンプルの Activity を書き込んだ時のデータベース識別バージョン

    //!< サンプルで扱う Activity の情報
    struct SamplePalmaActivityParameter
    {
        int32_t databaseVersion;            //!< データベース識別バージョン
        nn::hid::PalmaActivityIndex index;  //!< アクティビティのエントリのインデックス
        nn::hid::PalmaActivityEntry entry;  //!< アクティビティのエントリの内容
    };

    //!< データベースの後方(511 と 510 番) にサンプルの Activity を書き込むための設定
    SamplePalmaActivityParameter g_SamplePalmaActivityParameter[SampleActivityPattern_Max] = {
        { SampleDatabaseVersion, 510,{ 510, nn::hid::PalmaWaveSet_Small, 24 } },
        { SampleDatabaseVersion, 511,{ 511, nn::hid::PalmaWaveSet_Small, 25 } }
    };

    PalmaDevice g_PalmaDevices[NpadIdCountMax];

    int HandlePalmaOperationEvent(int index) NN_NOEXCEPT;    //!< Operation イベントをハンドリングします

    void ParsePalmaOperationInfo(int index, const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT;    //!< OperationInfo を取得して、解析します

    void ProceedActivationTaskSequence(int index) NN_NOEXCEPT;    //!< タスクを進めます

    ActivationTaskSequence GetNextActivationTask(int index) NN_NOEXCEPT;    //!< 次のタスクを取得します

    void ProceedWriteTaskSequence(int index) NN_NOEXCEPT;    //!< タスクを進めます

    WriteTaskSequence GetNextWriteTask(int index) NN_NOEXCEPT;    //!< 次のタスクを取得します

    void ProceedPropertyTaskSequence(int index) NN_NOEXCEPT;    //!< タスクを進めます

    PropertyTaskSequence GetNextPropertyTask(int index) NN_NOEXCEPT;    //!< 次のタスクを取得します

    void HandleResult(int index, nn::Result result) NN_NOEXCEPT;    //!< Result をハンドリングします

    int EnableStep(int index) NN_NOEXCEPT;    //!< Palma の Step を有効化します

    int GetStep(int index) NN_NOEXCEPT;    //!< Palma の Step を取得します

    int GetUniqueCode(int index) NN_NOEXCEPT;    //!< Palma の UniqueCode を取得します

    int GetDataBaseIdentificationVersion(int index) NN_NOEXCEPT;    //!< Palma のデータベース識別バージョンを取得します

    int WriteRgbLedEntry(int index) NN_NOEXCEPT;    //!< Palma に RgbLed エントリを書き込みます

    int WriteWaveEntry(int index) NN_NOEXCEPT;    //!< Palma に Wave エントリを書き込みます

    int WriteActivityEntry(int index) NN_NOEXCEPT;    //!< Palma に Activity エントリを書き込みます

    int WriteDataBaseIdentificationVersion(int index) NN_NOEXCEPT;    //!< Palma にデータベース識別バージョンを書き込みます

    const char* PalmaWaveSetToString(nn::hid::PalmaWaveSet type) NN_NOEXCEPT;

    const char* PalmaOperationTypeToString(nn::hid::PalmaOperationType type) NN_NOEXCEPT;

    void PrintPalmaOperationInfo(const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT;

    void PrintPalmaOperationInfoDetail(const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT;

    bool IsActivated(int index) NN_NOEXCEPT;

    nn::hid::PalmaFrModeType GetNext(nn::hid::PalmaFrModeType current) NN_NOEXCEPT;
}

void InitializePalmaDeviceController() NN_NOEXCEPT
{
    nn::hid::SetPalmaBoostMode(true);

    for (auto& palma : g_PalmaDevices)
    {
        palma.isValid = false;
        palma.nextFrModeType = nn::hid::PalmaFrModeType_Off;
        palma.playActivityPattern = SampleActivityPattern_1;
        palma.activationTask = ActivationTaskSequence_None;
        palma.writeTask = WriteTaskSequence_None;
        palma.propertyTask = PropertyTaskSequence_None;
    }
}

void FinalizePalmaDeviceController() NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        DeactivatePalmaDevice(i);
    }
}

void ActivatePalmaDevice(int index, const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    if (IsActivated(index))
    {
        return;
    }
    auto& palma = g_PalmaDevices[index];

    if (nn::hid::GetPalmaConnectionHandle(&palma.handle, id).IsFailure())
    {
        // 切断や対象の NpadId が Palma に対応していないため、ハンドルを取得できません
        NN_LOG("[PalmaDevice] Cannot get PalmaConnectionHandle.\n");
        return;
    }

    if (nn::hid::InitializePalma(palma.handle).IsFailure())
    {
        // Palma の初期化に失敗しました。切断やハンドルが無効になっている可能性があります
        NN_LOG("[PalmaDevice] InitializePalma failed.\n");
        return;
    }

    if (nn::hid::BindPalmaOperationCompleteEvent(palma.handle, &palma.operationEvent, nn::os::EventClearMode_ManualClear).IsFailure())
    {
        // Palma のイベントオブジェクトにバインドできませんでした。切断やハンドルが無効になっている可能性があります
        NN_LOG("[PalmaDevice] Cannot bind PalmaEvent.\n");
        return;
    }

    palma.id = id;
    palma.isValid = true;

    palma.activationTask = GetNextActivationTask(index);
    ProceedActivationTaskSequence(index);
}

void DeactivatePalmaDevice(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return;
    }

    auto& palma = g_PalmaDevices[index];

    nn::os::DestroySystemEvent(&palma.operationEvent);
    palma.isValid = false;
    palma.nextFrModeType = nn::hid::PalmaFrModeType_Off;
    palma.playActivityPattern = SampleActivityPattern_1;
    palma.activationTask = ActivationTaskSequence_None;
    palma.writeTask = WriteTaskSequence_None;
    palma.propertyTask = PropertyTaskSequence_None;
}

void UpdatePalmaDevices() NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        HandlePalmaOperationEvent(i);
    }
}

void DetachAllPalma() NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        auto& palma = g_PalmaDevices[i];
        if (palma.isValid)
        {
            nn::hid::DisconnectNpad(palma.id);
            DeactivatePalmaDevice(i);
        }
    }
}

int PlayPalmaActivity(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::PlayPalmaActivity(palma.handle, g_SamplePalmaActivityParameter[palma.playActivityPattern].index);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WritePalmaActivity(int index, SampleActivityPattern pattern) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    if (palma.writeTask != WriteTaskSequence_None)
    {
        // 処理中のため、何もせずに返す
        return -1;
    }
    palma.writeActivityPattern = pattern;
    palma.writeTask = GetNextWriteTask(index);
    ProceedWriteTaskSequence(index);
    return 0;
}

int GetPalmaProperties(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    if (palma.propertyTask != PropertyTaskSequence_None)
    {
        // 処理中のため、何もせずに返す
        return -1;
    }
    palma.propertyTask = GetNextPropertyTask(index);
    ProceedPropertyTaskSequence(index);
    return 0;
}

int ReadPalmaApplicationSection(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    // Application セクションの 0 番地から読み込む
    auto result = nn::hid::ReadPalmaApplicationSection(palma.handle, 0, nn::hid::PalmaApplicationSectionAccessUnitSizeMax);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WritePalmaApplicationSection(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    // Application セクションの 0 番地から書き込む
    char sampleData[] = "HidNpadPalmaSimple";
    nn::hid::PalmaApplicationSectionAccessBuffer buffer;
    memcpy(buffer.raw, sampleData, strlen(sampleData));
    auto result = nn::hid::WritePalmaApplicationSection(palma.handle, 0, strlen(sampleData), buffer);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int SetPalmaFrModeType(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::SetPalmaFrModeType(palma.handle, palma.nextFrModeType);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    palma.nextFrModeType = GetNext(palma.nextFrModeType);
    return 0;
}

void PairPalmaDevice(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return;
    }

    auto& palma = g_PalmaDevices[index];
    nn::hid::PairPalma(palma.handle);
}

int ReadPalmaPlayLog(int index, SamplePlayLogFieldType fieldType) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::ReadPalmaPlayLog(palma.handle, static_cast<nn::hid::PalmaPlayLogFieldIndex>(fieldType));
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int ResetPalmaPlayLog(int index, SamplePlayLogFieldType fieldType) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::ResetPalmaPlayLog(palma.handle, static_cast<nn::hid::PalmaPlayLogFieldIndex>(fieldType));
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int SuspendPalmaFeature(int index, bool isSuspended) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];
    nn::hid::PalmaFeatureSet featureFlags = nn::util::MakeBitFlagSet<32, nn::hid::PalmaFeature>();
    featureFlags.Set<nn::hid::PalmaFeature::FrMode>(isSuspended);
    featureFlags.Set<nn::hid::PalmaFeature::RumbleFeedback>(isSuspended);
    featureFlags.Set<nn::hid::PalmaFeature::Step>(isSuspended);
    featureFlags.Set<nn::hid::PalmaFeature::MuteSwitch>(isSuspended);
    auto result = nn::hid::SuspendPalmaFeature(palma.handle, featureFlags);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

namespace {
int HandlePalmaOperationEvent(int index) NN_NOEXCEPT
{
    if (!IsActivated(index))
    {
        return -1;
    }

    auto& palma = g_PalmaDevices[index];

    if (nn::os::TryWaitSystemEvent(&palma.operationEvent))
    {
        nn::os::ClearSystemEvent(&palma.operationEvent);

        // PALMA の実行結果を取得する
        nn::hid::PalmaOperationInfo palmaOperationInfo;
        auto result = nn::hid::GetPalmaOperationInfo(&palmaOperationInfo, palma.handle);
        if (result.IsSuccess())
        {
            PrintPalmaOperationInfo(palmaOperationInfo);
            ParsePalmaOperationInfo(index, palmaOperationInfo);
            return 1;
        }
        else
        {
            HandleResult(index, result);
            return -1;
        }
    }
    return 0;
}

void ParsePalmaOperationInfo(int index, const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    switch (info.type)
    {
        case nn::hid::PalmaOperationType_EnableStep:
        {
            palma.activationTask = GetNextActivationTask(index);
            ProceedActivationTaskSequence(index);
            break;
        }
        case nn::hid::PalmaOperationType_ReadStep:
        case nn::hid::PalmaOperationType_ReadUniqueCode:
        case nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion:
        {
            palma.propertyTask = GetNextPropertyTask(index);
            ProceedPropertyTaskSequence(index);
            break;
        }
        case nn::hid::PalmaOperationType_WriteRgbLedPatternEntry:
        case nn::hid::PalmaOperationType_WriteWaveEntry:
        case nn::hid::PalmaOperationType_WriteActivityEntry:
        case nn::hid::PalmaOperationType_WriteDataBaseIdentificationVersion:
        {
            palma.writeTask = GetNextWriteTask(index);
            ProceedWriteTaskSequence(index);
            break;
        }
        case nn::hid::PalmaOperationType_PlayActivity:
        case nn::hid::PalmaOperationType_SetFrModeType:
        case nn::hid::PalmaOperationType_ReadApplicationSection:
        case nn::hid::PalmaOperationType_WriteApplicationSection:
        case nn::hid::PalmaOperationType_SuspendFeature:
        default:
            break;
    }
}

void ProceedActivationTaskSequence(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    int result = -1;
    switch (palma.activationTask)
    {
        case ActivationTaskSequence_EnableStep:
        {
            result = EnableStep(index);
            break;
        }
        case ActivationTaskSequence_None:
        default:
            break;
    }
    if (result < 0)
    {
        palma.activationTask = ActivationTaskSequence_None;
    }
}

ActivationTaskSequence GetNextActivationTask(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    if (palma.isValid == false)
    {
        return ActivationTaskSequence_None;
    }

    auto max = static_cast<uint32_t>(ActivationTaskSequence_Max);
    auto current = static_cast<uint32_t>(palma.activationTask);
    return static_cast<ActivationTaskSequence>((current + 1) % max);
}

void ProceedWriteTaskSequence(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    int result = -1;
    switch (palma.writeTask)
    {
        case WriteTaskSequence_WriteRgbLedEntry:
        {
            result = WriteRgbLedEntry(index);
            break;
        }
        case WriteTaskSequence_WriteWaveEntry:
        {
            result = WriteWaveEntry(index);
            break;
        }
        case WriteTaskSequence_WriteActivityEntry:
        {
            result = WriteActivityEntry(index);
            break;
        }
        case WriteTaskSequence_WriteDataBaseIdentificationVersion:
        {
            result = WriteDataBaseIdentificationVersion(index);
            break;
        }
        case WriteTaskSequence_Completed:
        {
            // 書き込んだ Activity を再生するように切り替えます
            palma.playActivityPattern = palma.writeActivityPattern;
            palma.writeTask = WriteTaskSequence_None;
            result = 0;
            break;
        }
        case WriteTaskSequence_None:
        default:
            break;
    }
    if (result < 0)
    {
        palma.writeTask = WriteTaskSequence_None;
    }
}

WriteTaskSequence GetNextWriteTask(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    if (palma.isValid == false)
    {
        return WriteTaskSequence_None;
    }

    auto max = static_cast<uint32_t>(WriteTaskSequence_Max);
    auto current = static_cast<uint32_t>(palma.writeTask);
    return static_cast<WriteTaskSequence>((current + 1) % max);
}

void ProceedPropertyTaskSequence(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    int result = -1;
    switch (palma.propertyTask)
    {
        case PropertyTaskSequence_GetStep:
        {
            result = GetStep(index);
            break;
        }
        case PropertyTaskSequence_GetUniqueCode:
        {
            result = GetUniqueCode(index);
            break;
        }
        case PropertyTaskSequence_GetDataBaseIdentificationVersion:
        {
            result = GetDataBaseIdentificationVersion(index);
            break;
        }
        case PropertyTaskSequence_None:
        default:
            break;
    }
    if (result < 0)
    {
        palma.propertyTask = PropertyTaskSequence_None;
    }
}

PropertyTaskSequence GetNextPropertyTask(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    if (palma.isValid == false)
    {
        return PropertyTaskSequence_None;
    }

    auto max = static_cast<uint32_t>(PropertyTaskSequence_Max);
    auto current = static_cast<uint32_t>(palma.propertyTask);
    return static_cast<PropertyTaskSequence>((current + 1) % max);
}

void HandleResult(int index, nn::Result result) NN_NOEXCEPT
{
    if (nn::hid::ResultPalmaBusy::Includes(result))
    {
        // PALMA がほかの BLE 操作を処理で新たなリクエストを受け付けられません
    }
    else if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
    {
        // 切断などの要因で、PALMA のハンドルが無効になっています
        // 対象の Palma のシステムイベントを無効化します
        DeactivatePalmaDevice(index);
    }
    else if (nn::hid::ResultPalmaNoOperationInfo::Includes(result))
    {
        // 操作が実行されていないか、実行が完了していないため、操作結果が得られません
    }
}

int EnableStep(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::EnablePalmaStep(palma.handle, true);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int GetStep(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::ReadPalmaStep(palma.handle);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int GetUniqueCode(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::ReadPalmaUniqueCode(palma.handle);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int GetDataBaseIdentificationVersion(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto result = nn::hid::GetPalmaDataBaseIdentificationVersion(palma.handle);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WriteRgbLedEntry(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto rgbLedIndex = g_SamplePalmaActivityParameter[palma.writeActivityPattern].entry.rgbLedPatternIndex;

    size_t fileSize;
    LoadFile(&fileSize, g_FileReadBuffer, sizeof(g_FileReadBuffer), PalmaRgbLedPatternFiles[palma.writeActivityPattern]);

    auto result = nn::hid::WritePalmaRgbLedPatternEntry(palma.handle, rgbLedIndex, g_FileReadBuffer, fileSize);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WriteWaveEntry(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto waveIndex = g_SamplePalmaActivityParameter[palma.writeActivityPattern].entry.waveIndex;
    auto waveSet = g_SamplePalmaActivityParameter[palma.writeActivityPattern].entry.waveSet;

    size_t fileSize;
    LoadFile(&fileSize, g_FileReadBuffer, sizeof(g_FileReadBuffer), PalmaWaveFiles[palma.writeActivityPattern]);

    // ファイルサイズを MemoryPageSize の整数倍になるように整える
    auto alignedBufferSize = (fileSize / nn::os::MemoryPageSize + 1) * nn::os::MemoryPageSize;
    NN_ABORT_UNLESS_LESS_EQUAL(alignedBufferSize, sizeof(g_FileReadBuffer));

    auto result = nn::hid::WritePalmaWaveEntry(palma.handle, waveSet, waveIndex, g_FileReadBuffer, alignedBufferSize, fileSize);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WriteActivityEntry(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto activityIndex = g_SamplePalmaActivityParameter[palma.writeActivityPattern].index;
    auto& activityEntry = g_SamplePalmaActivityParameter[palma.writeActivityPattern].entry;

    auto result = nn::hid::WritePalmaActivityEntry(palma.handle, activityIndex, &activityEntry);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

int WriteDataBaseIdentificationVersion(int index) NN_NOEXCEPT
{
    auto& palma = g_PalmaDevices[index];
    auto version = g_SamplePalmaActivityParameter[palma.writeActivityPattern].databaseVersion;

    auto result = nn::hid::SetPalmaDataBaseIdentificationVersion(palma.handle, version);
    if (result.IsFailure())
    {
        HandleResult(index, result);
        return -1;
    }
    return 0;
}

const char* PalmaWaveSetToString(nn::hid::PalmaWaveSet type) NN_NOEXCEPT
{
    switch (type)
    {
        case nn::hid::PalmaWaveSet_Small:
            return "Small";
        case nn::hid::PalmaWaveSet_Medium:
            return "Medium";
        case nn::hid::PalmaWaveSet_Large:
            return "Large";
        default:
            return "Unknown";
    }
}

const char* PalmaOperationTypeToString(nn::hid::PalmaOperationType type) NN_NOEXCEPT
{
    switch (type)
    {
        case nn::hid::PalmaOperationType_PlayActivity:
            return "PlayActivity";
        case nn::hid::PalmaOperationType_SetFrModeType:
            return "SetFrModeType";
        case nn::hid::PalmaOperationType_ReadStep:
            return "ReadStep";
        case nn::hid::PalmaOperationType_EnableStep:
            return "EnableStep";
        case nn::hid::PalmaOperationType_ResetStep:
            return "ResetStep";
        case nn::hid::PalmaOperationType_ReadApplicationSection:
            return "ReadApplicationSection";
        case nn::hid::PalmaOperationType_WriteApplicationSection:
            return "WriteApplicationSection";
        case nn::hid::PalmaOperationType_ReadUniqueCode:
            return "ReadUniqueCode";
        case nn::hid::PalmaOperationType_SetUniqueCodeInvalid:
            return "SetUniqueCodeInvalid";
        case nn::hid::PalmaOperationType_WriteActivityEntry:
            return "WriteActivityEntry";
        case nn::hid::PalmaOperationType_WriteRgbLedPatternEntry:
            return "WriteRbgLedPatternEntry";
        case nn::hid::PalmaOperationType_WriteWaveEntry:
            return "WriteWaveEntry";
        case nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion:
            return "ReadDataBaseIdentificationVersion";
        case nn::hid::PalmaOperationType_WriteDataBaseIdentificationVersion:
            return "WriteDataBaseIdentificationVersion";
        case nn::hid::PalmaOperationType_SuspendFeature:
            return "SuspendFeature";
        case nn::hid::PalmaOperationType_ReadPlayLog:
            return "ReadPlayLog";
        case nn::hid::PalmaOperationType_ResetPlayLog:
            return "ResetPlayLog";
        default:
            return "Unknown";
    }
}

const char* PalmaResultToString(nn::Result result) NN_NOEXCEPT
{
    if (result.IsSuccess())
    {
        return "Success";
    }
    if (nn::hid::ResultPalmaBusy::Includes(result))
    {
        return "Busy";
    }
    else if (nn::hid::ResultPalmaInvalidHandle::Includes(result))
    {
        return "InvalidHandle";
    }
    else if (nn::hid::ResultPalmaNoOperationInfo::Includes(result))
    {
        return "NoOperationInfo";
    }
    else if (nn::hid::ResultPalmaUniqueCodeInvalid::Includes(result))
    {
        return "UniqueCodeInvalid";
    }
    else if (nn::hid::ResultPalmaOperationNotSupported::Includes(result))
    {
        return "OperationNotSupported";
    }
    else if (nn::hid::ResultPalmaInvalidFieldIndex::Includes(result))
    {
        return "Invalid Field Index";
    }
    else if (nn::hid::ResultPalmaOperationTimeout::Includes(result))
    {
        return "Timeout";
    }
    return "Failure";
}

void PrintPalmaOperationInfo(const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT
{
    NN_LOG("[PalmaDevice] [%s] ", PalmaOperationTypeToString(info.type));
    NN_LOG("Result = %s", PalmaResultToString(info.result));
    PrintPalmaOperationInfoDetail(info);
    NN_LOG("\n");
}

void PrintPalmaOperationInfoDetail(const nn::hid::PalmaOperationInfo& info) NN_NOEXCEPT
{
    switch (info.type)
    {
        case nn::hid::PalmaOperationType_PlayActivity:
        {
            auto& detailInfo = info.individual.playActivity;
            NN_LOG(" , Index = %d", detailInfo.index);
            break;
        }
        case nn::hid::PalmaOperationType_SetFrModeType:
        {
            auto& detailInfo = info.individual.setFrModeType;
            NN_LOG(" , FrModeType = %d", detailInfo.frModeType);
            break;
        }
        case nn::hid::PalmaOperationType_ReadStep:
        {
            auto& detailInfo = info.individual.readStep;
            NN_LOG(" , Steps = %d", detailInfo.step);
            break;
        }
        case nn::hid::PalmaOperationType_EnableStep:
        {
            auto& detailInfo = info.individual.enablePalmaStep;
            NN_LOG(" , Palma Step = %s", detailInfo.isEnabled ? "Enabled" : "Disabled");
            break;
        }
        case nn::hid::PalmaOperationType_ReadApplicationSection:
        {
            auto& detailInfo = info.individual.readApplicationSection;
            NN_LOG(" , StartAddress = 0x%04X , Size = %d\n", detailInfo.address, detailInfo.size);
            NN_LOG("Hex:\n");
            for (int i = 0; i < detailInfo.size; ++i)
            {
                if ((i != 0) && (i % 16 == 0))
                {
                    NN_LOG("\n");
                }
                NN_LOG("%02X ", detailInfo.buffer.raw[i]);
            }
            break;
        }
        case nn::hid::PalmaOperationType_WriteApplicationSection:
        {
            auto& detailInfo = info.individual.writeApplicationSection;
            NN_LOG(" , StartAddress = 0x%04X , Size = %d", detailInfo.address, detailInfo.size);
            break;
        }
        case nn::hid::PalmaOperationType_ReadUniqueCode:
        {
            auto& detailInfo = info.individual.readUniqueCode;
            NN_LOG(" , 0x");
            for (int i = 0; i < 16; ++i)
            {
                if ((i != 0) && (i % 4 == 0))
                {
                    NN_LOG("-");
                }
                NN_LOG("%02X", detailInfo.uniqueCode[i]);
            }
            break;
        }
        case nn::hid::PalmaOperationType_WriteActivityEntry:
        {
            auto& detailInfo = info.individual.writeActivityEntry;
            NN_LOG(" , ActivityEntry Index = %d", detailInfo.index);
            break;
        }
        case nn::hid::PalmaOperationType_WriteRgbLedPatternEntry:
        {
            auto& detailInfo = info.individual.writeRgbLedPatternEntry;
            NN_LOG(" , LedEntry Index = %d", detailInfo.index);
            break;
        }
        case nn::hid::PalmaOperationType_WriteWaveEntry:
        {
            auto& detailInfo = info.individual.writeWaveEntry;
            NN_LOG(" , WaveEntry %s , Index = %d", PalmaWaveSetToString(detailInfo.waveSet), detailInfo.index);
            break;
        }
        case nn::hid::PalmaOperationType_ReadDataBaseIdentificationVersion:
        {
            auto& detailInfo = info.individual.readDataBaseIdentificationVersion;
            NN_LOG(" , Version = 0x%08X", detailInfo.version);
            break;
        }
        case nn::hid::PalmaOperationType_WriteDataBaseIdentificationVersion:
        {
            auto& detailInfo = info.individual.writeDataBaseIdentificationVersion;
            NN_LOG(" , Version = 0x%08X", detailInfo.version);
            break;
        }
        case nn::hid::PalmaOperationType_SuspendFeature:
        {
            auto& detailInfo = info.individual.suspendPalmaFeature;
            NN_LOG(" , %d%d%d%d",
                detailInfo.suspendFeatureSet.Test<nn::hid::PalmaFeature::MuteSwitch>() ? 1 : 0,
                detailInfo.suspendFeatureSet.Test<nn::hid::PalmaFeature::Step>() ? 1 : 0,
                detailInfo.suspendFeatureSet.Test<nn::hid::PalmaFeature::RumbleFeedback>() ? 1 : 0,
                detailInfo.suspendFeatureSet.Test<nn::hid::PalmaFeature::FrMode>() ? 1 : 0);
            break;
        }
        default:
            return;
    }
} //NOLINT(impl/function_size)

bool IsActivated(int index) NN_NOEXCEPT
{
    return g_PalmaDevices[index].isValid;
}

nn::hid::PalmaFrModeType GetNext(nn::hid::PalmaFrModeType current) NN_NOEXCEPT
{
    uint32_t max = static_cast<uint32_t>(nn::hid::PalmaFrModeType_Downloaded) + 1;
    uint32_t nextFrModeType = (static_cast<uint32_t>(current) + 1) % max;
    return static_cast<nn::hid::PalmaFrModeType>(nextFrModeType);
}

}
