﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadPalma.h>
#include "HidNpadController.h"
#include "PalmaDeviceController.h"
#include "PalmaConnectionManager.h"

namespace {
    nn::hid::NpadHandheldState g_OldNpadHandheldState;
    nn::hid::NpadHandheldState g_CurrentNpadHandheldState;
    nn::hid::NpadFullKeyState  g_OldNpadFullKeyState[NpadIdCountMax];
    nn::hid::NpadFullKeyState  g_CurrentNpadFullKeyState[NpadIdCountMax];
    nn::hid::NpadJoyDualState  g_OldNpadJoyDualState[NpadIdCountMax * 2];
    nn::hid::NpadJoyDualState  g_CurrentNpadJoyDualState[NpadIdCountMax * 2];
    nn::hid::NpadPalmaState    g_OldNpadPalmaState[NpadIdCountMax];
    nn::hid::NpadPalmaState    g_CurrentNpadPalmaState[NpadIdCountMax];

    struct NpadControllerProperty
    {
        bool isConnected;
    };
    NpadControllerProperty g_Property[NpadIdCountMax];

    bool g_TerminateFlag = false;

    // Activity を書き込む対象の Palma、最初に接続した Palma とする
    const int TargetPalma = 0;

    void PrintNpadButtonState(const nn::hid::NpadButtonSet& state) NN_NOEXCEPT;

    void PrintNpadStickState(const nn::hid::AnalogStickState& state) NN_NOEXCEPT;

    template <typename TState>
    void PrintNpadState(const TState& state) NN_NOEXCEPT;

    int UpdateNpadStyleFullKey(int index) NN_NOEXCEPT;

    int UpdateNpadStyleJoyDual(int index) NN_NOEXCEPT;

    int UpdateNpadStyleHandheld(int index) NN_NOEXCEPT;

    int UpdateNpadStylePalma(int index) NN_NOEXCEPT;

    bool IsAnyButtons(const nn::hid::NpadButtonSet& current, const nn::hid::NpadButtonSet& old) NN_NOEXCEPT;

    bool IsPlusAndMinus(const nn::hid::NpadButtonSet& buttons) NN_NOEXCEPT;

    void OnPressed(const nn::hid::NpadButtonSet& buttons) NN_NOEXCEPT;

    void OnAPressed() NN_NOEXCEPT;

    void OnXPressed() NN_NOEXCEPT;

    void OnYPressed() NN_NOEXCEPT;

    void OnLeftPressed() NN_NOEXCEPT;

    void OnRightPressed() NN_NOEXCEPT;

    void OnPlusPressed() NN_NOEXCEPT;

    const char* NpadIdToString(nn::hid::NpadIdType id) NN_NOEXCEPT;
}

void InitializeNpadController() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定する
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStylePalma::Mask);

    // 使用する Npad を設定する
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    for (auto& npad : g_Property)
    {
        npad.isConnected = false;
    }
}

int UpdateNpadController() NN_NOEXCEPT
{
    for(int i = 0; i < NpadIdCountMax; i++)
    {
        bool isActive = false;

        //現在有効な操作形態(NpadStyleSet)を取得する
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        // フルキー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleFullKey>() == true)
        {
            isActive = true;
            if (UpdateNpadStyleFullKey(i) < 0) break;
        }
        // Joy-Con 操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            isActive = true;
            if (UpdateNpadStyleJoyDual(i) < 0) break;
        }
        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            isActive = true;
            if (UpdateNpadStyleHandheld(i) < 0) break;
        }
        // Palma 操作が有効な場合
        if (style.Test<nn::hid::NpadStylePalma>() == true)
        {
            isActive = true;
            ActivatePalmaDevice(i, NpadIds[i]);
            if (UpdateNpadStylePalma(i) < 0) break;
        }

        auto& npad = g_Property[i];
        if (npad.isConnected)
        {
            if (!isActive)
            {
                npad.isConnected = false;
                NN_LOG("Disconnected %s.\n", NpadIdToString(NpadIds[i]));
            }
        }
        else
        {
            if (isActive)
            {
                npad.isConnected = true;
                NN_LOG("Connected %s.\n", NpadIdToString(NpadIds[i]));
            }
        }
    }
    return g_TerminateFlag ? -1 : 0;
}

namespace {

void PrintNpadButtonState(const nn::hid::NpadButtonSet& state) NN_NOEXCEPT
{
    char buttons[38];
    buttons[0]   = (state.Test<nn::hid::NpadJoyButton::A>())         ? 'A' : '_';
    buttons[1]   = (state.Test<nn::hid::NpadJoyButton::B>())         ? 'B' : '_';
    buttons[2]   = (state.Test<nn::hid::NpadJoyButton::X>())         ? 'X' : '_';
    buttons[3]   = (state.Test<nn::hid::NpadJoyButton::Y>())         ? 'Y' : '_';
    buttons[4]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'L' : '_';
    buttons[5]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'S' : '_';
    buttons[6]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'R' : '_';
    buttons[7]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'S' : '_';
    buttons[8]   = (state.Test<nn::hid::NpadJoyButton::L>())         ? 'L' : '_';
    buttons[9]   = (state.Test<nn::hid::NpadJoyButton::R>())         ? 'R' : '_';
    buttons[10]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'Z' : '_';
    buttons[11]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'L' : '_';
    buttons[12]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'Z' : '_';
    buttons[13]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'R' : '_';
    buttons[14]  = (state.Test<nn::hid::NpadJoyButton::Plus>())      ? '+' : '_';
    buttons[15]  = (state.Test<nn::hid::NpadJoyButton::Minus>())     ? '-' : '_';
    buttons[16]  = (state.Test<nn::hid::NpadJoyButton::Left>())      ? '<' : '_';
    buttons[17]  = (state.Test<nn::hid::NpadJoyButton::Up>())        ? '^' : '_';
    buttons[18]  = (state.Test<nn::hid::NpadJoyButton::Right>())     ? '>' : '_';
    buttons[19]  = (state.Test<nn::hid::NpadJoyButton::Down>())      ? 'v' : '_';
    buttons[20]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'S' : '_';
    buttons[21]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'L' : '_';
    buttons[22]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'S' : '_';
    buttons[23]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'R' : '_';
    buttons[24]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'S' : '_';
    buttons[25]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'L' : '_';
    buttons[26]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'S' : '_';
    buttons[27]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'R' : '_';
    buttons[28]  = (state.Test<nn::hid::NpadJoyButton::StickLRight>()) ? '>' : '_';
    buttons[29]  = (state.Test<nn::hid::NpadJoyButton::StickLUp>())    ? '^' : '_';
    buttons[30]  = (state.Test<nn::hid::NpadJoyButton::StickLLeft>())  ? '<' : '_';
    buttons[31]  = (state.Test<nn::hid::NpadJoyButton::StickLDown>())  ? 'v' : '_';
    buttons[32]  = (state.Test<nn::hid::NpadJoyButton::StickRRight>()) ? '>' : '_';
    buttons[33]  = (state.Test<nn::hid::NpadJoyButton::StickRUp>())    ? '^' : '_';
    buttons[34]  = (state.Test<nn::hid::NpadJoyButton::StickRLeft>())  ? '<' : '_';
    buttons[35]  = (state.Test<nn::hid::NpadJoyButton::StickRDown>())  ? 'v' : '_';
    buttons[36]  = (state.Test<nn::hid::NpadPalmaButton::Palma>())  ? 'P' : '_';
    buttons[37]  = '\0';

    NN_LOG("%s", buttons);
}

void PrintNpadStickState(const nn::hid::AnalogStickState& state) NN_NOEXCEPT
{
    NN_LOG("(%6d, %6d)", state.x,
                            state.y);
}

template <typename TState>
void PrintNpadState(const TState& state) NN_NOEXCEPT
{
    NN_LOG("[%6lld] ", state.samplingNumber);
    PrintNpadButtonState(state.buttons);
    NN_LOG(" L");
    PrintNpadStickState(state.analogStickL);
    NN_LOG(" R");
    PrintNpadStickState(state.analogStickR);
    NN_LOG("\n");
}

int UpdateNpadStyleFullKey(int index) NN_NOEXCEPT
{
    g_OldNpadFullKeyState[index] = g_CurrentNpadFullKeyState[index];

    //最新のNpadのステートを取得する
    nn::hid::GetNpadState(&(g_CurrentNpadFullKeyState[index]), NpadIds[index]);

    //ボタンが押された
    if (IsAnyButtons(g_CurrentNpadFullKeyState[index].buttons, g_OldNpadFullKeyState[index].buttons))
    {
        // Npad の入力状態を表示する
        NN_LOG("NpadFullKey (%d)     ", index);
        PrintNpadState(g_CurrentNpadFullKeyState[index]);
    }

    if (IsPlusAndMinus(g_CurrentNpadFullKeyState[index].buttons))
    {
        g_TerminateFlag = true;
        return -1;
    }
    return 0;
}

int UpdateNpadStyleJoyDual(int index) NN_NOEXCEPT
{
    g_OldNpadJoyDualState[index] = g_CurrentNpadJoyDualState[index];

    //最新のNpadのステートを取得する
    nn::hid::GetNpadState(&(g_CurrentNpadJoyDualState[index]), NpadIds[index]);

    //ボタンが押された
    if (IsAnyButtons(g_CurrentNpadJoyDualState[index].buttons, g_OldNpadJoyDualState[index].buttons))
    {
        // Npad の入力状態を表示する
        NN_LOG("NpadJoyDual (%d)     ", index);
        PrintNpadState(g_CurrentNpadJoyDualState[index]);
        OnPressed(g_CurrentNpadJoyDualState[index].buttons);
    }

    if (IsPlusAndMinus(g_CurrentNpadJoyDualState[index].buttons))
    {
        g_TerminateFlag = true;
        return -1;
    }
    return 0;
}

int UpdateNpadStyleHandheld(int index) NN_NOEXCEPT
{
    g_OldNpadHandheldState = g_CurrentNpadHandheldState;

    //最新のNpadのステートを取得する
    nn::hid::GetNpadState(&(g_CurrentNpadHandheldState), NpadIds[index]);

    //ボタンが押された
    if (IsAnyButtons(g_CurrentNpadHandheldState.buttons, g_OldNpadHandheldState.buttons))
    {
        // Npad の入力状態を表示する
        NN_LOG("NpadHandheld         ");
        PrintNpadState(g_CurrentNpadHandheldState);
        OnPressed(g_CurrentNpadHandheldState.buttons);
    }

    if (IsPlusAndMinus(g_CurrentNpadHandheldState.buttons))
    {
        g_TerminateFlag = true;
        return -1;
    }
    return 0;
}

int UpdateNpadStylePalma(int index) NN_NOEXCEPT
{
    g_OldNpadPalmaState[index] = g_CurrentNpadPalmaState[index];

    //最新のNpadのステートを取得する
    nn::hid::GetNpadState(&(g_CurrentNpadPalmaState[index]), NpadIds[index]);

    //ボタンが押された
    if (IsAnyButtons(g_CurrentNpadPalmaState[index].buttons, g_OldNpadPalmaState[index].buttons))
    {
        // Npad の入力状態を表示する
        NN_LOG("NpadPalma         ");
        PrintNpadState(g_CurrentNpadPalmaState[index]);
        // Palma の Activity を再生する
        PlayPalmaActivity(index);
    }
    return 0;
}

bool IsAnyButtons(const nn::hid::NpadButtonSet& current, const nn::hid::NpadButtonSet& old) NN_NOEXCEPT
{
    //ボタンが押された
    if ((current ^ old & current).IsAnyOn())
    {
        return true;
    }
    return false;
}

bool IsPlusAndMinus(const nn::hid::NpadButtonSet& buttons) NN_NOEXCEPT
{
    // ＋ と - ボタンを同時に押された
    if (buttons.Test<nn::hid::NpadButton::Plus>() && buttons.Test<nn::hid::NpadButton::Minus>())
    {
        return true;
    }
    return false;
}

void OnPressed(const nn::hid::NpadButtonSet& buttons) NN_NOEXCEPT
{
    // A が押された
    if (buttons.Test<nn::hid::NpadButton::A>())
    {
        OnAPressed();
    }
    // X が押された
    if (buttons.Test<nn::hid::NpadButton::X>())
    {
        OnXPressed();
    }
    // Y が押された
    if (buttons.Test<nn::hid::NpadButton::Y>())
    {
        OnYPressed();
    }
    // < が押された
    if (buttons.Test<nn::hid::NpadButton::Left>())
    {
        OnLeftPressed();
    }
    // > が押された
    if (buttons.Test<nn::hid::NpadButton::Right>())
    {
        OnRightPressed();
    }
    // ^ が押された
    if (buttons.Test<nn::hid::NpadButton::Up>())
    {
    }
    // v が押された
    if (buttons.Test<nn::hid::NpadButton::Down>())
    {
    }
    // + が押された
    if (buttons.Test<nn::hid::NpadButton::Plus>())
    {
        OnPlusPressed();
    }
}

void OnAPressed() NN_NOEXCEPT
{
    // Palma のスキャン状態を切り替える
    IsRunningPalmaScanner() ? StopPalmaScanner() : StartPalmaScanner();
}

void OnXPressed() NN_NOEXCEPT
{
    // 接続している Palma をすべて切断する
    DetachAllPalma();
}

void OnYPressed() NN_NOEXCEPT
{
    // Palma のプロパティを取得する
    GetPalmaProperties(TargetPalma);
}

void OnLeftPressed() NN_NOEXCEPT
{
    // Palma にパターン 1 の Activity を書き込む
    WritePalmaActivity(TargetPalma, SampleActivityPattern_1);
}

void OnRightPressed() NN_NOEXCEPT
{
    // Palma にパターン 2 の Activity を書き込む
    WritePalmaActivity(TargetPalma, SampleActivityPattern_2);
}

void OnPlusPressed() NN_NOEXCEPT
{
    // Palma をペアリングする
    PairPalmaDevice(TargetPalma);
}

const char* NpadIdToString(nn::hid::NpadIdType id) NN_NOEXCEPT
{
    switch (id)
    {
    case nn::hid::NpadId::No1:
        return "NpadId::No1";
    case nn::hid::NpadId::No2:
        return "NpadId::No2";
    case nn::hid::NpadId::No3:
        return "NpadId::No3";
    case nn::hid::NpadId::No4:
        return "NpadId::No4";
    case nn::hid::NpadId::Handheld:
        return "NpadId::Handheld";
    default:
        return "Unknown";
    }
}
}
