﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
    @examplesource{HidNpadSimple.cpp,PageSampleHidNpadSimple}

    @brief
    Npadを使ってボタン入力を取得するためのサンプルプログラム
*/

/**
    @page PageSampleHidNpadSimple Npadを使ったボタン入力の取得
    @tableofcontents

    @brief
    Npadを使ってボタン入力を取得するためのサンプルプログラムの解説です。

    @section PageSampleHidNpadSimple_SectionBrief 概要
    コントローラの最新のボタン入力を取得する方法について説明します。

    @section PageSampleHidNpadSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidNpadSimple @endlink 以下にあります。

    @section PageSampleHidNpadSimple_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV 環境で動作させる場合は、事前に SDEV とコントローラをペアリングしてください。

    @section PageSampleHidNpadSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    Joy-Con が1本でも無線接続されている場合は、NpadJoyDual の入力としてコンソールにログが出力されます。
    Joy-Con が2本無線接続されている場合(本体にジョイントされていない場合)は、NpadJoyDual 及び NpadFullKey の入力としてログが出力されます。
    Joy-Con がジョイントされている場合は、NpadHandheld の入力としてコンソールにログが出力されます。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。
    Joy-Con (L) だけ、もしくは Joy-Con (R) だけが接続された場合はVisualStudioかNintendo Target Manager経由で終了させてください。

    @section PageSampleHidNpadSimple_SectionPrecaution 注意事項
    本サンプルプログラムでは画面描画は行いません。

    @section PageSampleHidNpadSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidNpadSimple_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - Npad を初期化
    - 現在のボタン入力状態を取得
*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

namespace
{
    bool    waits = true;

    nn::hid::NpadIdType g_NpadIds[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4,
                                       nn::hid::NpadId::Handheld,
                                       };
    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);
    nn::hid::NpadHandheldState oldNpadHandheldState;
    nn::hid::NpadHandheldState currentNpadHandheldState;
    nn::hid::NpadFullKeyState oldNpadFullKeyState[NpadIdCountMax];
    nn::hid::NpadFullKeyState currentNpadFullKeyState[NpadIdCountMax];
    nn::hid::NpadJoyDualState oldNpadJoyDualState[NpadIdCountMax * 2];
    nn::hid::NpadJoyDualState currentNpadJoyDualState[NpadIdCountMax * 2];

    void PrintNpadButtonState(const nn::hid::NpadButtonSet& state)
    {
        char buttons[37];
        buttons[0]   = (state.Test<nn::hid::NpadJoyButton::A>())         ? 'A' : '_';
        buttons[1]   = (state.Test<nn::hid::NpadJoyButton::B>())         ? 'B' : '_';
        buttons[2]   = (state.Test<nn::hid::NpadJoyButton::X>())         ? 'X' : '_';
        buttons[3]   = (state.Test<nn::hid::NpadJoyButton::Y>())         ? 'Y' : '_';
        buttons[4]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'L' : '_';
        buttons[5]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'S' : '_';
        buttons[6]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'R' : '_';
        buttons[7]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'S' : '_';
        buttons[8]   = (state.Test<nn::hid::NpadJoyButton::L>())         ? 'L' : '_';
        buttons[9]   = (state.Test<nn::hid::NpadJoyButton::R>())         ? 'R' : '_';
        buttons[10]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'Z' : '_';
        buttons[11]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'L' : '_';
        buttons[12]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'Z' : '_';
        buttons[13]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'R' : '_';
        buttons[14]  = (state.Test<nn::hid::NpadJoyButton::Plus>())      ? '+' : '_';
        buttons[15]  = (state.Test<nn::hid::NpadJoyButton::Minus>())     ? '-' : '_';
        buttons[16]  = (state.Test<nn::hid::NpadJoyButton::Left>())      ? '<' : '_';
        buttons[17]  = (state.Test<nn::hid::NpadJoyButton::Up>())        ? '^' : '_';
        buttons[18]  = (state.Test<nn::hid::NpadJoyButton::Right>())     ? '>' : '_';
        buttons[19]  = (state.Test<nn::hid::NpadJoyButton::Down>())      ? 'v' : '_';
        buttons[20]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'S' : '_';
        buttons[21]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'L' : '_';
        buttons[22]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'S' : '_';
        buttons[23]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'R' : '_';
        buttons[24]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'S' : '_';
        buttons[25]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'L' : '_';
        buttons[26]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'S' : '_';
        buttons[27]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'R' : '_';
        buttons[28]  = (state.Test<nn::hid::NpadJoyButton::StickLRight>()) ? '>' : '_';
        buttons[29]  = (state.Test<nn::hid::NpadJoyButton::StickLUp>())    ? '^' : '_';
        buttons[30]  = (state.Test<nn::hid::NpadJoyButton::StickLLeft>())  ? '<' : '_';
        buttons[31]  = (state.Test<nn::hid::NpadJoyButton::StickLDown>())  ? 'v' : '_';
        buttons[32]  = (state.Test<nn::hid::NpadJoyButton::StickRRight>()) ? '>' : '_';
        buttons[33]  = (state.Test<nn::hid::NpadJoyButton::StickRUp>())    ? '^' : '_';
        buttons[34]  = (state.Test<nn::hid::NpadJoyButton::StickRLeft>())  ? '<' : '_';
        buttons[35]  = (state.Test<nn::hid::NpadJoyButton::StickRDown>())  ? 'v' : '_';
        buttons[36]  = '\0';

        NN_LOG("%s", buttons);
    }

    void PrintNpadStickState(const nn::hid::AnalogStickState& state)
    {
        NN_LOG("(%6d, %6d)", state.x,
                             state.y);
    }

    template <typename TState>
    void PrintNpadState(const TState& state)
    {
        NN_LOG("[%6lld] ", state.samplingNumber);
        PrintNpadButtonState(state.buttons);
        NN_LOG(" L");
        PrintNpadStickState(state.analogStickL);
        NN_LOG(" R");
        PrintNpadStickState(state.analogStickR);
        NN_LOG("\n");
    }

    void Init()
    {
        nn::hid::InitializeNpad();

        //使用する操作形態を設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

        // 使用する Npad を設定
        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);
    }

    void Update()
    {
        for(int i = 0; i < NpadIdCountMax; i++)
        {
            //現在有効な操作形態(NpadStyleSet)を取得
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);

            // フルキー操作が有効な場合
            if (style.Test<nn::hid::NpadStyleFullKey>() == true)
            {
                oldNpadFullKeyState[i] = currentNpadFullKeyState[i];

                //最新のNpadのステートを取得
                nn::hid::GetNpadState(&(currentNpadFullKeyState[i]), g_NpadIds[i]);

                //ボタンが押された
                if ((currentNpadFullKeyState[i].buttons ^ oldNpadFullKeyState[i].buttons & currentNpadFullKeyState[i].buttons).IsAnyOn())
                {
                    // Npad の入力状態を表示
                    NN_LOG("NpadFullKey (%d)     ", i);
                    PrintNpadState(currentNpadFullKeyState[i]);
                }

                // ＋ と - ボタンを同時に押されたら終了
                if (currentNpadFullKeyState[i].buttons.Test<nn::hid::NpadButton::Plus>() && currentNpadFullKeyState[i].buttons.Test<nn::hid::NpadButton::Minus>())
                {
                    waits = false;
                }
            }
            // Joy-Con 操作が有効な場合
            if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
            {
                oldNpadJoyDualState[i] = currentNpadJoyDualState[i];

                //最新のNpadのステートを取得
                nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), g_NpadIds[i]);

                //ボタンが押された
                if ((currentNpadJoyDualState[i].buttons ^ oldNpadJoyDualState[i].buttons & currentNpadJoyDualState[i].buttons).IsAnyOn())
                {
                    // Npad の入力状態を表示
                    NN_LOG("NpadJoyDual (%d)     ", i);
                    PrintNpadState(currentNpadJoyDualState[i]);
                }

                // ＋ と - ボタンを同時に押されたら終了
                if (currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadButton::Plus>() && currentNpadJoyDualState[i].buttons.Test<nn::hid::NpadButton::Minus>())
                {
                    waits = false;
                }
            }
            // 携帯機コントローラー操作が有効な場合
            if (style.Test<nn::hid::NpadStyleHandheld>() == true)
            {
                oldNpadHandheldState = currentNpadHandheldState;

                //最新のNpadのステートを取得
                nn::hid::GetNpadState(&(currentNpadHandheldState), g_NpadIds[i]);

                //ボタンが押された
                if ((currentNpadHandheldState.buttons ^ oldNpadHandheldState.buttons & currentNpadHandheldState.buttons).IsAnyOn())
                {
                    // Npad の入力状態を表示
                    NN_LOG("NpadHandheld         ");
                    PrintNpadState(currentNpadHandheldState);
                }

                // ＋ と - ボタンを同時に押されたら終了
                if (currentNpadHandheldState.buttons.Test<nn::hid::NpadButton::Plus>() && currentNpadHandheldState.buttons.Test<nn::hid::NpadButton::Minus>())
                {
                    waits = false;
                }
            }
        }
    }
}//namespace

extern "C" void nnMain()
{
    NN_LOG("Npad Sample Start.\n");

    Init();

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");

    while(waits)
    {
        Update();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    NN_LOG("Npad Sample Done.\n");
}
