﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/vi.h>

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <map>
#include <utility>
#include <nn/nn_Windows.h>
#include <nn/os/os_Mutex.h>
#endif

#include "HidKeyboardAndMouse_WindowMessage.h"

namespace {

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
struct WndProcContext
{
    WNDPROC defaultWndProc;
    bool needsToBeActive;
    bool needsToBeClosed;
    bool isActive;
    bool isClosed;
};

nn::os::MutexType& GetMutex() NN_NOEXCEPT
{
    static nn::os::MutexType s_Mutex = NN_OS_MUTEX_INITIALIZER(false);

    return s_Mutex;
}

std::map<HWND, WndProcContext>& GetWndProcContextMap() NN_NOEXCEPT
{
    static std::map<HWND, WndProcContext> s_Map;

    return s_Map;
}

LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    nn::os::LockMutex(&GetMutex());

    WndProcContext& context = GetWndProcContextMap()[hwnd];

    switch(uMsg)
    {
    case WM_CLOSE:
        context.needsToBeClosed = true;
        break;

    case WM_ACTIVATEAPP:
        if(wParam == TRUE)
        {
            context.needsToBeActive = true;
        }
        else
        {
            context.needsToBeActive = false;
        }

        break;

    default:
        break;
    }

    WNDPROC defaultWndProc = context.defaultWndProc;

    nn::os::UnlockMutex(&GetMutex());

    if (uMsg == WM_CLOSE)
    {
        return 0;
    }
    else
    {
        return ::CallWindowProc(defaultWndProc, hwnd, uMsg, wParam, lParam);
    }
}
#endif

} // namespace

void EnableWindowMessage(nn::vi::NativeWindowHandle handle) NN_NOEXCEPT
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    NN_UNUSED(handle);
#else
    HWND hwnd = reinterpret_cast<HWND>(handle);

    nn::os::LockMutex(&GetMutex());

    WndProcContext context =
    {
        reinterpret_cast<WNDPROC>(::GetWindowLongPtr(hwnd, GWLP_WNDPROC)),
    };

    std::map<HWND, WndProcContext>& map = GetWndProcContextMap();

    map.insert(std::pair<HWND, WndProcContext>(hwnd, context));

    ::SetWindowLongPtr(hwnd, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WndProc));

    nn::os::UnlockMutex(&GetMutex());
#endif
}

WindowMessage GetWindowMessage(nn::vi::NativeWindowHandle handle) NN_NOEXCEPT
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    NN_UNUSED(handle);

    return WindowMessage_None;
#else
    HWND hwnd = reinterpret_cast<HWND>(handle);

    WindowMessage message = WindowMessage_None;

    nn::os::LockMutex(&GetMutex());

    WndProcContext& context = GetWndProcContextMap()[hwnd];

    if (context.isActive != context.needsToBeActive)
    {
        if (context.needsToBeActive)
        {
            message = WindowMessage_Active;
        }
        else
        {
            message = WindowMessage_Inactive;
        }

        context.isActive = context.needsToBeActive;
    }

    if (context.isClosed != context.needsToBeClosed)
    {
        if (context.needsToBeClosed)
        {
            message = WindowMessage_Close;
        }

        context.isClosed = context.needsToBeClosed;
    }

    nn::os::UnlockMutex(&GetMutex());

    return message;
#endif
}
