﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx.h>
#include <nn/util/util_BitUtil.h>
#include <nn/vi.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include "HidKeyboardAndMouse_Color.h"
#include "HidKeyboardAndMouse_GraphicsSystem.h"

namespace {

const size_t MemoryPoolSize = 64 * 1024 * 1024;
const size_t MemoryPoolSizeInvisible = 32 * 1024 * 1024;

const size_t CommandBufferCommandMemoryChunkSize = 512 * 1024;
const size_t CommandBufferControlMemoryChunkSize = 64 * 1024;

const int SwapInterval = 1;

} // namespace

GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
    : m_pApplicationHeap(0)
    , m_Width(0)
    , m_Height(0)
    , m_pMemoryPool(0)
    , m_pMemoryPoolOffset(0)
    , m_pMemoryPoolCommandOffset(0)
    , m_pMemoryPoolInvisible(0)
    , m_pMemoryPoolInvisibleOffset(0)
    , m_TextureDescriptorBaseIndex(0)
    , m_SamplerDescriptorBaseIndex(0)
    , m_pDisplay(0)
    , m_pLayer(0)
{
}

nn::gfx::DescriptorPool& GraphicsSystem::GetTextureDescriptorPool() NN_NOEXCEPT
{
    return m_TextureDescriptorPool;
}

int GraphicsSystem::GetTextureDescriptorBaseIndex() const NN_NOEXCEPT
{
    return m_TextureDescriptorBaseIndex;
}

nn::gfx::DescriptorPool& GraphicsSystem::GetSamplerDescriptorPool() NN_NOEXCEPT
{
    return m_SamplerDescriptorPool;
}

int GraphicsSystem::GetSamplerDescriptorBaseIndex() const NN_NOEXCEPT
{
    return m_SamplerDescriptorBaseIndex;
}

int GraphicsSystem::GetWidth() const NN_NOEXCEPT
{
    return m_Width;
}

int GraphicsSystem::GetHeight() const NN_NOEXCEPT
{
    return m_Height;
}

nn::vi::NativeWindowHandle GraphicsSystem::GetNativeWindowHandle(
    ) const NN_NOEXCEPT
{
    return m_NativeWindowHandle;
}

nn::gfx::Device& GraphicsSystem::GetDevice() NN_NOEXCEPT
{
    return m_Device;
}

nn::gfx::CommandBuffer& GraphicsSystem::GetCommandBuffer() NN_NOEXCEPT
{
    return m_CommandBuffer;
}

void GraphicsSystem::Initialize(
    ApplicationHeap* pApplicationHeap, int width, int height) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pApplicationHeap);
    NN_ASSERT_GREATER(width, 0);
    NN_ASSERT_GREATER(height, 0);
    m_pApplicationHeap = pApplicationHeap;
    m_Width = width;
    m_Height = height;
    nn::vi::Initialize();
    this->InitializeDisplay();
    nn::gfx::Initialize();
    this->InitializeDevice();
    this->InitializeMemoryPool();
    this->InitializeMemoryPoolInvisible();
    this->InitializeSwapChain();
    this->InitializeQueue();
    this->InitializeFence();
    this->InitializeColorTargetTexture();
    this->InitializeDepthStencilTexture();
    this->InitializeDepthStencilView();
    this->InitializeTextureDescriptorPool();
    this->InitializeSamplerDescriptorPool();
    this->InitializeCommandBuffer();
    this->InitializeViewportScissorState();
    this->InitializeBlendState();
    this->InitializeDepthStencilState();
    this->InitializeRasterizerState();
}

void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    m_RasterizerState.Finalize(&m_Device);
    m_DepthStencilState.Finalize(&m_Device);
    this->FinalizeBlendState();
    this->FinalizeViewportScissorState();
    this->ResetCommandBuffer();
    m_CommandBuffer.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_DepthStencilView.Finalize(&m_Device);
    m_DepthStencilTexture.Finalize(&m_Device);
    m_ColorTargetTexture.Finalize(&m_Device);
    m_Fence.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    this->FinalizeMemoryPoolInvisible();
    this->FinalizeMemoryPool();
    m_Device.Finalize();
    nn::gfx::Finalize();
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();
}

void GraphicsSystem::BeginDraw() NN_NOEXCEPT
{
    this->ResetCommandBuffer();
    this->AddCommandMemoryToCommandBuffer(&m_CommandBuffer);
    this->AddControlMemoryToCommandBuffer(&m_CommandBuffer);

    m_CommandBuffer.Begin();

    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    nn::gfx::ColorTargetView*
        pColorTargetView = m_SwapChain .AcquireNextScanBufferView();

    m_CommandBuffer.ClearColor(
        pColorTargetView,
        static_cast<float>(Color::Black.v[0]) / Color::White.v[0],
        static_cast<float>(Color::Black.v[1]) / Color::White.v[1],
        static_cast<float>(Color::Black.v[2]) / Color::White.v[2],
        static_cast<float>(Color::Black.v[3]) / Color::White.v[3],
        0);

    m_CommandBuffer.ClearDepthStencil(
        &m_DepthStencilView,
        1.0f,
        0,
        nn::gfx::DepthStencilClearMode_DepthStencil,
        0);

    m_CommandBuffer.SetRenderTargets(1, &pColorTargetView, &m_DepthStencilView);

    m_CommandBuffer.SetViewportScissorState(&m_ViewportScissorState);
    m_CommandBuffer.SetBlendState(&m_BlendState);
    m_CommandBuffer.SetDepthStencilState(&m_DepthStencilState);
    m_CommandBuffer.SetRasterizerState(&m_RasterizerState);
}

void GraphicsSystem::EndDraw() NN_NOEXCEPT
{
    m_CommandBuffer.End();
    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_Fence);
    m_Queue.Flush();
}

void GraphicsSystem::Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT
{
    m_Fence.Sync(timeout);
    m_Queue.Present(&m_SwapChain, SwapInterval);
    m_Queue.Sync();
}

void GraphicsSystem::OutOfCommandMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddCommandMemoryToCommandBuffer(pCommandBuffer);
}

void GraphicsSystem::OutOfControlMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddControlMemoryToCommandBuffer(pCommandBuffer);
}

void GraphicsSystem::InitializeDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::CreateLayer(&m_pLayer, m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::SetLayerScalingMode(
            m_pLayer, nn::vi::ScalingMode_PreserveAspectRatio));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::GetNativeWindow(&m_NativeWindowHandle, m_pLayer));
}

void GraphicsSystem::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);

    m_Device.Initialize(info);
}

void GraphicsSystem::InitializeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached |
                               nn::gfx::MemoryPoolProperty_GpuCached);

    const size_t aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPool = m_pApplicationHeap->Allocate(MemoryPoolSize, aligment);

    info.SetPoolMemory(m_pMemoryPool, MemoryPoolSize);

    m_MemoryPool.Initialize(&m_Device, info);

    m_pMemoryPoolOffset = 0;

    m_pMemoryPoolCommandOffset = 0;
}

void GraphicsSystem::InitializeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
                               nn::gfx::MemoryPoolProperty_GpuCached |
                               nn::gfx::MemoryPoolProperty_Compressible);

    const size_t aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPoolInvisible =
        m_pApplicationHeap->Allocate(MemoryPoolSizeInvisible, aligment);

    info.SetPoolMemory(m_pMemoryPoolInvisible, MemoryPoolSizeInvisible);

    m_MemoryPoolInvisible.Initialize(&m_Device, info);

    m_pMemoryPoolInvisibleOffset = 0;
}


void GraphicsSystem::InitializeSwapChain() NN_NOEXCEPT
{
    NN_ASSERT_GREATER(m_Width, 0);
    NN_ASSERT_GREATER(m_Height, 0);

    nn::gfx::SwapChain::InfoType info;
    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(m_Width);
    info.SetHeight(m_Height);
    info.SetBufferCount(2);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(
        m_pMemoryPoolInvisibleOffset,
        nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));

    size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);

    m_SwapChain.Initialize(&m_Device,
                           info,
                           &m_MemoryPoolInvisible,
                           m_pMemoryPoolInvisibleOffset,
                           size);

    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);

    m_Queue.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();

    m_Fence.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeColorTargetTexture() NN_NOEXCEPT
{
    NN_ASSERT_GREATER(m_Width, 0);
    NN_ASSERT_GREATER(m_Height, 0);

    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_Width);
    info.SetHeight(m_Height);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(
        m_pMemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_ColorTargetTexture.Initialize(&m_Device,
                                    info,
                                    &m_MemoryPoolInvisible,
                                    m_pMemoryPoolInvisibleOffset,
                                    size);

    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeDepthStencilTexture() NN_NOEXCEPT
{
    NN_ASSERT_GREATER(m_Width, 0);
    NN_ASSERT_GREATER(m_Height, 0);

    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_Width);
    info.SetHeight(m_Height);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_pMemoryPoolInvisibleOffset = nn::util::align_up(
        m_pMemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_DepthStencilTexture.Initialize(&m_Device,
                                     info,
                                     &m_MemoryPoolInvisible,
                                     m_pMemoryPoolInvisibleOffset,
                                     size);

    m_pMemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeDepthStencilView() NN_NOEXCEPT
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension(nn::gfx::ImageDimension_2d);
    info.SetTexturePtr(&m_DepthStencilTexture);

    m_DepthStencilView.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& type = nn::gfx::AccessorToData(m_Device);
    nvnDeviceGetInteger(type.pNvnDevice,
                        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS,
                        &m_TextureDescriptorBaseIndex);
#endif
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(m_TextureDescriptorBaseIndex + 1);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
            &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_TextureDescriptorPool.Initialize(&m_Device,
                                       info,
                                       &m_MemoryPool,
                                       m_pMemoryPoolOffset,
                                       size);

    m_pMemoryPoolOffset += size;
}

void GraphicsSystem::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& type = nn::gfx::AccessorToData(m_Device);
    nvnDeviceGetInteger(type.pNvnDevice,
                        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS,
                        &m_SamplerDescriptorBaseIndex);
#endif
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(m_SamplerDescriptorBaseIndex + 1);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
            &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_SamplerDescriptorPool.Initialize(&m_Device,
                                       info,
                                       &m_MemoryPool,
                                       m_pMemoryPoolOffset,
                                       size);

    m_pMemoryPoolOffset += size;
}

void GraphicsSystem::InitializeCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    m_CommandBuffer.Initialize(&m_Device, info);
    m_CommandBuffer.SetUserPtr(this);
    m_CommandBuffer.SetOutOfCommandMemoryEventCallback(
        OutOfCommandMemoryEventCallback);
    m_CommandBuffer.SetOutOfControlMemoryEventCallback(
        OutOfControlMemoryEventCallback);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    m_pMemoryPoolCommandOffset = m_pMemoryPoolOffset;
}

void GraphicsSystem::InitializeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);
    NN_ASSERT_GREATER(m_Width, 0);
    NN_ASSERT_GREATER(m_Height, 0);

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);

    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth(static_cast<float>(m_Width));
    viewportStateInfo.SetHeight(static_cast<float>(m_Height));
    info.SetViewportStateInfoArray(&viewportStateInfo, 1);

    nn::gfx::ScissorStateInfo scissorStateInfo;
    scissorStateInfo.SetDefault();
    scissorStateInfo.SetWidth(m_Width);
    scissorStateInfo.SetHeight(m_Height);
    info.SetScissorStateInfoArray(&scissorStateInfo, 1);

    size_t memorySize =
        nn::gfx::ViewportScissorState::GetRequiredMemorySize(info);
    m_ViewportScissorState.SetMemory(
        m_pApplicationHeap->Allocate(memorySize), memorySize);

    m_ViewportScissorState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red |
                              nn::gfx::ChannelMask_Green |
                              nn::gfx::ChannelMask_Blue);
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetDestinationColorBlendFactor(
        nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
    m_BlendState.SetMemory(m_pApplicationHeap->Allocate(memorySize),
                           memorySize);

    m_BlendState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(true);

    m_DepthStencilState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);

    m_RasterizerState.Initialize(&m_Device, info);
}

void GraphicsSystem::FinalizeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPool.Finalize(&m_Device);

    m_pApplicationHeap->Deallocate(m_pMemoryPool);
}

void GraphicsSystem::FinalizeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPoolInvisible.Finalize(&m_Device);

    m_pApplicationHeap->Deallocate(m_pMemoryPoolInvisible);
}

void GraphicsSystem::FinalizeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_ViewportScissorState.GetMemory();

    m_ViewportScissorState.Finalize(&m_Device);

    m_pApplicationHeap->Deallocate(ptr);
}

void GraphicsSystem::FinalizeBlendState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_BlendState.GetMemory();

    m_BlendState.Finalize(&m_Device);

    m_pApplicationHeap->Deallocate(ptr);
}

void GraphicsSystem::ResetCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_CommandBuffer.Reset();

    m_pMemoryPoolOffset = m_pMemoryPoolCommandOffset;

    const size_t size = m_CommandBufferControlMemoryChunks.size();

    for (size_t i = 0; i < size; ++i)
    {
        m_pApplicationHeap->Deallocate(
            m_CommandBufferControlMemoryChunks[i]);
    }

    m_CommandBufferControlMemoryChunks.clear();
}

void GraphicsSystem::AddCommandMemoryToCommandBuffer(
    nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    m_pMemoryPoolOffset = nn::util::align_up(
        m_pMemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    pCommandBuffer->AddCommandMemory(&m_MemoryPool,
                                     m_pMemoryPoolOffset,
                                     CommandBufferCommandMemoryChunkSize);

    m_pMemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
}

void GraphicsSystem::AddControlMemoryToCommandBuffer(
    nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* pControlMemoryChunk = m_pApplicationHeap->Allocate(
        CommandBufferControlMemoryChunkSize,
        nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device));

    pCommandBuffer->AddControlMemory(pControlMemoryChunk,
                                     CommandBufferControlMemoryChunkSize);

    m_CommandBufferControlMemoryChunks.push_back(pControlMemoryChunk);
}
