﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidControllerSequence_Title.cpp,PageSampleHidControllerSequence_Title}

    @brief
    サンプルアプリケーションのタイトル画面でのコントローラーサポートアプレットの呼び出し方
*/

/**
    @page PageSampleHidControllerSequence_Title タイトル画面でのコントローラーサポートアプレットの呼び出し方
    @tableofcontents

    @brief
    サンプルアプリケーションのタイトル画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Title_SectionBrief 概要
    サンプルアプリケーションのタイトル画面においてのコントローラーサポートアプレットの呼び出し方の解説です

    @section PageSampleHidControllerSequence_Title_SectionFileStructure ファイル構成
    本サンプルプログラムは
    @link ../../../Samples/Sources/Applications/HidControllerSequence
    Samples/Sources/Applications/HidControllerSequence @endlink 以下にあります。

    @section PageSampleHidControllerSequence_Title_SectionNecessaryEnvironment 必要な環境
    事前に実機とコントローラーをペアリングしてください。

    @section PageSampleHidControllerSequence_Title_SectionHowToOperate 操作方法
    マスタープレイヤーのみ操作選択権があります。

    上下の方向ボタンもしくはスティック操作でゲームのプレイ人数の選択する事が出来ます。@n
    Aボタンを押す事により選択されているプレイ人数でゲームを開始します。

    @section PageSampleHidControllerSequence_Title_SectionPrecaution 注意事項
    コントローラーは十分に充電した状態でお使いください。

    @section PageSampleHidControllerSequence_Title_SectionHowToExecute 実行手順
    サンプルアプリケーションをビルドし、実行してください。

    @section PageSampleHidControllerSequence_Title_SectionDetail 解説

    @image html Applications\HidControllerSequence\HidControllerSequence_Title.png

    タイトル画面での処理の流れは以下の通りです
    - コントローラーの接続確認
    - プレイ人数の選択
    - プレイ人数の確定
    - 接続中のコントローラーの個数確認
    - ゲーム画面に移動

    タイトル画面で以下の場合にコントローラーサポートアプレットが呼び出されます
    - マスタープレイヤーのコントローラーが消失
    - プレイ人数文の無線コントローラーが存在しない

    @image html Applications\HidControllerSequence\HidControllerSequence_Single.png
*/

#include "./HidControllerSequence_Title.h"

TitleSceneThread::TitleSceneThread() : ThreadState()
{

}

TitleSceneThread::TitleSceneThread(const char* name, nn::os::ThreadFunction func) : ThreadState(name, func)
{
}

void TitleSceneThread::Initialize()
{
    m_pRenderer     = &g_pGraphicsSystem->GetPrimitiveRenderer();
    m_pTextWriter   = &g_pGraphicsSystem->GetDebugFont();
    m_pCommand      = &g_pGraphicsSystem->GetCommandBuffer();

    m_IsExit                = false;
    m_FrameCount            = 0;
    m_SelectMenuNumber      = 0;        // 選択中の項目のインデックス

    //-------------------------------------------------------
    // メニュー
    //-------------------------------------------------------
    m_MenuText[0] = "1P";
    m_MenuText[1] = "2P";
    m_MenuText[2] = "3P";
    m_MenuText[3] = "4P";
    for (size_t i = 0; i < MenuCount; ++i)
    {
        m_MenuSize[i] = nn::util::MakeFloat2(512.f, 48.f);
        m_MenuTextPos[i] = nn::util::MakeFloat2((1280.f - m_MenuSize[i].x) / 2.f, static_cast<float>(200 + i * (m_MenuSize[i].y + 16.f)));
    }

    //-------------------------------------------------------
    // スタイル
    //-------------------------------------------------------
    m_StyleIconMargin = 32.f;
    m_StyleIconSize = nn::util::MakeFloat2(128.f, 128.f);
    m_StyleIconDrawAreaWidth = (m_StyleIconSize.x * NpadStykeCount) + m_StyleIconMargin * (NpadStykeCount - 1);

    // 操作形態をのアイコンの描画位置
    for (size_t i = 0; i < NpadStykeCount; ++i)
    {
        const float drawAreaWidth    = (m_StyleIconSize.x * NpadStykeCount) + m_StyleIconMargin * (NpadStykeCount - 1);
        m_StyleIconPos[i]           = nn::util::MakeFloat2((m_InfoLinePos - drawAreaWidth) / 2.f + (m_StyleIconSize.x + m_StyleIconMargin) * static_cast<float>(i), 560.f);
    }

    // 上選択
    m_UpButtons = nns::hid::Button::Up::Mask | nns::hid::Button::LsUp::Mask | nns::hid::Button::RsUp::Mask;

    // 下選択
    m_DownButtons = nns::hid::Button::Down::Mask | nns::hid::Button::LsDown::Mask | nns::hid::Button::RsDown::Mask;

    //-------------------------------------------------------
    // ツールの状態表示
    //-------------------------------------------------------
    // 分割線
    m_IsOpenInfo = false;
    m_InfoLinePos = 1280.f;
    m_InfoLineLeftPos = (1280.f / 3.f) * 2.f;

    // ツールの状態を表示するテキスト
    m_InfoText[InfoTextType_PlayMode]              = "< Play Mode >";
    m_InfoText[InfoTextType_PlayerCount]           = "< Player Count >";
    m_InfoText[InfoTextType_MasterPlayerNpadId]    = "< Master Player NpadId >";
    m_InfoText[InfoTextType_EnableNpadStyle]       = "< Enable Npad Style >";

    // 描画位置
    for (size_t i = 0; i < NN_ARRAY_SIZE(m_InfoTextPos); ++i)
    {
        m_InfoTextPos[i] = NN_UTIL_FLOAT_2_INITIALIZER(32.f, 128.f + 96.f * i);
    }

    // タイトルアイコンのサイズ
    m_TitleLogoSize = NN_UTIL_FLOAT_2_INITIALIZER(640, 64);

    m_AnimationCount = 0;
    m_AnimationCountMax = 20;

    nn::os::LockMutex(&g_Mutex);
    {
        g_pControllerManager->Update();
    }
    nn::os::UnlockMutex(&g_Mutex);

}

void TitleSceneThread::Update()
{
    ++m_FrameCount;
    //==============================================
    // コントローラー の処理
    //==============================================
    nn::os::LockMutex(&g_Mutex);
    {
        g_pControllerManager->Update();
    }
    nn::os::UnlockMutex(&g_Mutex);

    // コントローラーの接続確認
    CheckPatternSet check;
    check.Reset();
    check.Set<CheckPattern::Single>();
    check.Set<CheckPattern::CallInterval>();
    check.Set<CheckPattern::Resume>();
    if (CheckConnectController(check))
    {
        // コントローラーサポートアプレットの呼び出しが必要な場合
        CallControllerSupportApplet(nullptr, 1);
    }
    else
    {
        int32_t connectedControllerCount = 0;

        for (
            std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
            it != g_pControllerManager->GetControllerList().end();
            ++it
            )
        {
            if ((*it)->IsConnected() == false)
            {
                continue;
            }
            // コントローラーのIDを取得します
            const nn::hid::NpadIdType id = ApplicationState::NpadIds[it - g_pControllerManager->GetControllerList().begin()];

            // 本体装着コントローラーかNpad::No1以外のコントローラーの入力では操作できないようにします
            if (
                (ApplicationState::IsHandheldMode() && id == nn::hid::NpadId::Handheld) ||
                (!ApplicationState::IsHandheldMode() && id == nn::hid::NpadId::No1)
                )
            {
                if (id != nn::hid::NpadId::Handheld)
                {
                    ++connectedControllerCount;
                }
                if (((*it)->GetButtonsDown() & (nns::hid::Button::Plus::Mask | nns::hid::Button::Minus::Mask)).IsAnyOn())
                {
                    m_IsOpenInfo ^= true;
                }

                // メニューの選択
                nns::hid::ButtonSet pressButton = (*it)->GetButtons();                                  // 押されているボタン
                nns::hid::ButtonSet inputButton = (*it)->GetButtonsDown() | (*it)->GetButtonsRepeat();  // 入力判定として扱うボタン

                if ((pressButton & m_UpButtons).IsAnyOn())
                {
                    const int buttonIndex =
                        pressButton.Test(nns::hid::Button::Up::Index) ? nns::hid::Button::Up::Index :
                        pressButton.Test(nns::hid::Button::LsUp::Index) ? nns::hid::Button::LsUp::Index : nns::hid::Button::RsUp::Index;
                    if (inputButton.Test(buttonIndex))
                    {
                        m_SelectMenuNumber = (m_SelectMenuNumber - 1 + NN_ARRAY_SIZE(m_MenuText)) % NN_ARRAY_SIZE(m_MenuText);
                        m_AnimationCount = 1;
                    }
                }
                else if ((pressButton & m_DownButtons).IsAnyOn())
                {
                    const int buttonIndex =
                        pressButton.Test(nns::hid::Button::Down::Index) ? nns::hid::Button::Down::Index :
                        pressButton.Test(nns::hid::Button::LsDown::Index) ? nns::hid::Button::LsDown::Index : nns::hid::Button::RsDown::Index;
                    if (inputButton.Test(buttonIndex))
                    {
                        m_SelectMenuNumber = (m_SelectMenuNumber + 1) % NN_ARRAY_SIZE(m_MenuText);
                        m_AnimationCount = 1;
                    }
                }
                else if ((*it)->GetButtonsDown().Test(nns::hid::Button::A::Index))
                {
                    if (id == nn::hid::NpadId::Handheld)
                    {
                        connectedControllerCount = 1;       // 本体装着コンで選択された場合、コントローラーの接続台数を1台とみなします
                    }
                    m_IsExit = true;
                }
            }
            else if ((id != nn::hid::NpadId::Handheld) && (static_cast<int32_t>(id) <= m_SelectMenuNumber))
            {
                /*
                 *  本体装着コントローラー以外かつ
                 *  複数人プレイで使用されるコントローラーの場合、接続台数を加算します
                 */
                ++connectedControllerCount;
            }
        }

        if (m_IsExit && (m_SelectMenuNumber + 1 > connectedControllerCount))
        {
            /*
            *  プレイヤー人数に対してコントローラーが足りない場合
            *  コントローラーサポートアプレットを呼び出します。
            */
            nn::hid::ControllerSupportResultInfo info;
            nn::Result result = CallControllerSupportApplet(&info, m_SelectMenuNumber + 1);
            if (result.IsSuccess())
            {
                m_IsExit = true;
                ApplicationState::EnableHandheldMode((info.selectedId == nn::hid::NpadId::Handheld) && (m_SelectMenuNumber == 0));
            }
            else
            {
                m_IsExit = false;
            }
        }

        if (m_IsExit)
        {
            nn::os::LockMutex(&g_Mutex);
            {
                ApplicationState::SetPlayerCount(m_SelectMenuNumber + 1);
                ThreadState* pThreadState = nullptr;
                NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneTitle, &pThreadState));
                g_ThreadManager.SetEndThread(pThreadState->GetThreadType());
                NN_ASSERT(g_ThreadManager.GetThreadStateFromIndex(ThreadManager::ThreadList_SceneGame, &pThreadState));
                g_ThreadManager.SetNextThread(pThreadState->GetThreadType());
            }
            nn::os::UnlockMutex(&g_Mutex);
        }
    }

    // 内部情報とメニューメッセージの更新
    UpdateInfo();

    // 内部情報とテキスト描画
    ShowInfo();

} // NOLINT(impl/function_size)

void TitleSceneThread::Draw()
{
    //==============================================
    // 描画処理
    //==============================================
    nn::os::LockMutex(&g_Mutex);
    {
        g_pGraphicsSystem->BeginDraw();

        //--------------
        // 画面右側
        //--------------
        m_pRenderer->SetColor(nn::util::Color4u8::Gray());
        m_pRenderer->Draw2DLine(m_pCommand, m_InfoLinePos, 32.f, m_InfoLinePos, 720.f - 32.f);

        // 開閉メッセージとアイコン
        if (m_InfoLinePos > m_InfoLineLeftPos + 256)
        {
            nn::util::Color4u8 color = nn::util::Color4u8::Gray();
            color.SetA(static_cast<uint8_t>(255 * std::min(m_InfoLinePos - (m_InfoLineLeftPos + 256.f), 256.f) / 256.f));
            m_pRenderer->SetColor(color);
            m_pRenderer->Draw2DFrame(m_pCommand, m_InfoLinePos - 192, 32, 192, 48);

            m_pRenderer->Draw2DRect(m_pCommand, m_InfoLinePos - 184, 40, 64.f, 32.f, g_TextureDescriptor[ResourceType_ButtonIcon], g_SamplerDescriptor);
        }
        if (m_InfoLinePos < m_InfoLineLeftPos + 256)
        {
            nn::util::Color4u8 color = nn::util::Color4u8::Gray();
            color.SetA(static_cast<uint8_t>(255 * std::min((m_InfoLineLeftPos + 256) - m_InfoLinePos, 256.f) / 256.f));
            m_pRenderer->SetColor(color);
            m_pRenderer->Draw2DFrame(m_pCommand, m_InfoLinePos, 720.f - 80.f, 192, 48);

            m_pRenderer->Draw2DRect(m_pCommand, m_InfoLinePos + 8.f, 720.f - 72.f, 64.f, 32.f, g_TextureDescriptor[ResourceType_ButtonIcon], g_SamplerDescriptor);
        }
        // サポートしている操作形態の表示
        nn::hid::NpadStyleSet supportedStyle = nn::hid::GetSupportedNpadStyleSet();
        if (m_SelectMenuNumber != 0) { supportedStyle &= supportedStyle.Reset<nn::hid::NpadStyleHandheld>(); }
        for (int styleIndex = 0; styleIndex < nn::hid::GetSupportedNpadStyleSet().CountPopulation(); ++styleIndex)
        {
            m_pRenderer->SetColor(supportedStyle.Test(styleIndex) ? nn::util::Color4u8::White() : nn::util::Color4u8::Red());
            m_pRenderer->Draw2DFrame(m_pCommand,
                m_InfoLinePos + m_InfoTextPos[InfoTextType_EnableNpadStyle].x + 64 * styleIndex,
                m_InfoTextPos[InfoTextType_EnableNpadStyle].y + m_pTextWriter->GetLineHeight() * 2.f,
                48.f, 48.f);
            if (!supportedStyle.Test(styleIndex))
            {
                m_pRenderer->Draw2DLine(m_pCommand,
                    m_InfoLinePos + m_InfoTextPos[InfoTextType_EnableNpadStyle].x + 64 * styleIndex + 48.f,
                    m_InfoTextPos[InfoTextType_EnableNpadStyle].y + m_pTextWriter->GetLineHeight() * 2.f,
                    m_InfoLinePos + m_InfoTextPos[InfoTextType_EnableNpadStyle].x + 64 * styleIndex,
                    m_InfoTextPos[InfoTextType_EnableNpadStyle].y + m_pTextWriter->GetLineHeight() * 2.f + 48.f);
            }
            m_pRenderer->SetColor(nn::util::Color4u8::Gray());

            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(m_InfoLinePos + m_InfoTextPos[InfoTextType_EnableNpadStyle].x + 64 * styleIndex, m_InfoTextPos[InfoTextType_EnableNpadStyle].y + m_pTextWriter->GetLineHeight() * 2.f);
            nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(48.f, 48.f);

            DrawStyleIcon(pos, size, styleIndex);
        }

        //--------------
        // 画面中央のメニューの描画
        //--------------
        nn::util::Color4u8 BgColor = nn::util::Color4u8(24, 128, 224, 4);
        for (int32_t i = 0; i < (int32_t)NN_ARRAY_SIZE(m_MenuTextPos); ++i)
        {
            if (i == m_SelectMenuNumber && m_AnimationCount > 0)
            {
                // 選択が変更された事を主張する
                const float baseScale = 1.f - ((float)m_AnimationCount / (float)m_AnimationCountMax);
                float scale = ((nn::util::SinEst(nn::util::DegreeToRadian(static_cast<float>(m_AnimationCount * 36)))) / 20.f) * baseScale;

                m_pRenderer->SetColor(BgColor);
                m_pRenderer->Draw2DRect(m_pCommand,
                    m_MenuTextPos[i].x - (m_MenuSize[i].x * scale) / 2.f,
                    m_MenuTextPos[i].y - (m_MenuSize[i].y * -scale) / 2.f,
                    m_MenuSize[i].x + (m_MenuSize[i].x * scale),
                    m_MenuSize[i].y + m_MenuSize[i].y * -scale);
                m_pRenderer->SetColor(nn::util::Color4u8::Yellow());
                m_pRenderer->Draw2DFrame(m_pCommand,
                    m_MenuTextPos[i].x - (m_MenuSize[i].x * scale) / 2.f,
                    m_MenuTextPos[i].y - (m_MenuSize[i].y * -scale) / 2.f,
                    m_MenuSize[i].x + (m_MenuSize[i].x * scale),
                    m_MenuSize[i].y + m_MenuSize[i].y * -scale);
            }
            else
            {
                if (i == m_SelectMenuNumber)
                {
                    m_pRenderer->SetColor(BgColor);
                    m_pRenderer->Draw2DRect(m_pCommand, m_MenuTextPos[i].x, m_MenuTextPos[i].y, m_MenuSize[i].x, m_MenuSize[i].y);
                    m_pRenderer->SetColor(nn::util::Color4u8::Yellow());
                }
                else
                {
                    m_pRenderer->SetColor(nn::util::Color4u8::White());
                }
                m_pRenderer->Draw2DFrame(m_pCommand, m_MenuTextPos[i].x, m_MenuTextPos[i].y, m_MenuSize[i].x, m_MenuSize[i].y);
            }
        }

        //--------------
        // 画面下部に表示される接続中のコントローラーアイコン
        //--------------

        for (
            std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
            it != g_pControllerManager->GetControllerList().end();
            ++it
            )
        {
            size_t controllerIndex = it - g_pControllerManager->GetControllerList().begin();
            const nn::hid::NpadIdType id = ApplicationState::NpadIds[controllerIndex];
            bool isLeftConnected = false;
            bool isRightConnected = false;
            if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>())
            {
                isLeftConnected = static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsLeftConnected>();
                isRightConnected = static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsRightConnected>();
            }

            if ((*it)->IsConnected() || (isLeftConnected || isRightConnected))
            {
                nn::hid::NpadStyleSet styleSet = nn::hid::GetNpadStyleSet(id);
                /*
                * 携帯モードで無い場合は本体装着コントローラーを表示せず
                * 携帯モードの場合は本体装着コントローラーのみを表示する
                */
                if (!styleSet.Test<nn::hid::NpadStyleHandheld>())
                {
                    if (
                        (styleSet.CountPopulation() == 1) &&
                        (controllerIndex < ApplicationState::MaxPlayerCount))
                    {
                        nn::hid::NpadControllerColor controllerColor;
                        controllerColor.main = nn::util::Color4u8::Gray();
                        controllerColor.sub = nn::util::Color4u8::Black();

                        for (int styleIndex = 0; styleIndex < styleSet.GetCount(); ++styleIndex)
                        {
                            if (styleSet.Test(styleIndex))
                            {
                                m_pRenderer->SetColor(controllerColor.sub);
                                m_pRenderer->Draw2DRect(m_pCommand,
                                    m_StyleIconPos[controllerIndex + 1].x, m_StyleIconPos[controllerIndex + 1].y, m_StyleIconSize.x, m_StyleIconSize.y);
                                m_pRenderer->SetColor(controllerColor.main);

                                DrawStyleIcon(m_StyleIconPos[controllerIndex + 1], m_StyleIconSize, styleIndex);

                                // JoyDualで接続されていない方のコントローラーの色を暗くする
                                if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>())
                                {
                                    nn::util::Color4u8 color = nn::util::Color4u8::Black();
                                    color.SetA(224);
                                    m_pRenderer->SetColor(color);
                                    if (!isLeftConnected)
                                    {
                                        m_pRenderer->Draw2DRect(m_pCommand, m_StyleIconPos[controllerIndex + 1].x, m_StyleIconPos[controllerIndex + 1].y, m_StyleIconSize.x / 2.f, m_StyleIconSize.y);
                                    }
                                    if (!isRightConnected)
                                    {
                                        m_pRenderer->Draw2DRect(m_pCommand, m_StyleIconPos[controllerIndex + 1].x + m_StyleIconSize.x / 2.f, m_StyleIconPos[controllerIndex + 1].y, m_StyleIconSize.x / 2.f, m_StyleIconSize.y);
                                    }
                                }
                                break;
                            }
                        }
                    }
                }
                else
                {
                    // 携帯モード
                    nn::hid::NpadControllerColor controllerColor;
                    controllerColor.main = nn::util::Color4u8::Gray();
                    controllerColor.sub = nn::util::Color4u8::Black();

                    m_pRenderer->SetColor(controllerColor.sub);
                    m_pRenderer->Draw2DRect(m_pCommand,
                        m_StyleIconPos[0].x, m_StyleIconPos[0].y, m_StyleIconSize.x, m_StyleIconSize.y);
                    m_pRenderer->SetColor(controllerColor.main);
                    DrawStyleIcon( m_StyleIconPos[0], m_StyleIconSize, nn::hid::NpadStyleHandheld::Index);
                }

            }
        }

        //--------------
        // 画面下部に表示される接続中のコントローラーアイコンの枠とプレイヤーランプ
        //--------------
        for (size_t i = 0; i < ApplicationState::MaxPlayerCount + 1; ++i)
        {
            m_pRenderer->SetColor(
                (i < 2) ?
                (ApplicationState::IsHandheldMode() && i == 0) || (!ApplicationState::IsHandheldMode() && i == 1) ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::White() :
                (i <= static_cast<size_t>(m_SelectMenuNumber + 1)) ?
                nn::util::Color4u8::White() : nn::util::Color4u8::Gray());
            m_pRenderer->Draw2DFrame(m_pCommand, m_StyleIconPos[i].x, m_StyleIconPos[i].y, m_StyleIconSize.x, m_StyleIconSize.y);
            if (i > 0)
            {
                nn::Bit8 pattern = nn::hid::GetPlayerLedPattern(ApplicationState::NpadIds[i - 1]);
                for (int led = 0; led < 4; ++led)
                {
                    m_pRenderer->SetColor((pattern & (0x01 << led)) != 0 ? nn::util::Color4u8::Green() : nn::util::Color4u8::Gray());
                    m_pRenderer->Draw2DRect(m_pCommand,
                        m_StyleIconPos[i].x + static_cast<float>(led * 8.f), m_StyleIconPos[i].y + m_StyleIconSize.y + 8.f, 6.f, 6.f);
                }
            }
        }

        m_pTextWriter->Draw(m_pCommand);

        //--------------
        // 画面上部に表示される小さなタイトルアイコン
        //--------------
        m_pRenderer->SetColor(nn::util::Color4u8::Gray());
        m_pRenderer->Draw2DRect(m_pCommand, 0, 0, m_TitleLogoSize.x / 3.5f, m_TitleLogoSize.y / 3.5f, g_TextureDescriptor[ResourceType_Title], g_SamplerDescriptor);

        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }
    if (m_AnimationCount > 0) { m_AnimationCount = (m_AnimationCount + 1) % m_AnimationCountMax; }
    nn::os::UnlockMutex(&g_Mutex);

} // // NOLINT(impl/function_size)

void TitleSceneThread::Finalize()
{

}

void TitleSceneThread::UpdateInfo()
{
    if (m_IsOpenInfo)
    {
        float dist = m_InfoLinePos - m_InfoLineLeftPos;
        const float MaxDist = 1280.f - m_InfoLineLeftPos;
        float speed = (std::min(1.f, std::max((dist / MaxDist) * (dist / MaxDist), 0.f)) * 0.9f + 0.1f) * 32.f;

        m_InfoLinePos = m_InfoLinePos > (1280.f / 3.f) * 2.f ? m_InfoLinePos -= speed : m_InfoLineLeftPos;
    }
    else
    {
        float dist = 1280.f - m_InfoLinePos;
        const float MaxDist = 1280.f - m_InfoLineLeftPos;
        float speed = (std::min(1.f, std::max((dist / MaxDist) * (dist / MaxDist), 0.f) * 0.9f + 0.1f)) * 32.f;

        m_InfoLinePos = m_InfoLinePos < 1280.f ? m_InfoLinePos += speed : 1280.f;
    }
    for (size_t i = 0; i < NN_ARRAY_SIZE(m_MenuTextPos); ++i)
    {
        m_MenuTextPos[i] = nn::util::MakeFloat2((m_InfoLinePos - m_MenuSize[i].x) / 2.f, m_MenuTextPos[i].y);
    }
    for (size_t i = 0; i < NN_ARRAY_SIZE(m_StyleIconPos); ++i)
    {
        m_StyleIconPos[i] = nn::util::MakeFloat2(
            (m_InfoLinePos - m_StyleIconDrawAreaWidth) / 2.f + (m_StyleIconSize.x + m_StyleIconMargin) * static_cast<float>(i),
            m_StyleIconPos[i].y);
    }
}

void TitleSceneThread::ShowInfo()
{
    nn::os::LockMutex(&g_Mutex);
    {
        // プレイヤー人数
        for (size_t i = 0; i < MenuCount; ++i)
        {
            const float baseScale = 1.f - ((float)m_AnimationCount / (float)m_AnimationCountMax);
            float fontScale = ((nn::util::SinEst(nn::util::DegreeToRadian(static_cast<float>(m_AnimationCount * 36)))) / static_cast<float>(m_AnimationCountMax)) * baseScale;
            static_cast<int32_t>(i) == m_SelectMenuNumber && m_AnimationCount > 0 ? m_pTextWriter->SetScale(1.f + fontScale, 1.05f - fontScale) : m_pTextWriter->SetScale(1.f, 1.f);

            m_pTextWriter->SetCursorY(m_MenuTextPos[i].y + (m_MenuSize[i].y - m_pTextWriter->CalculateStringHeight(m_MenuText[i].c_str())) / 2.f);
            m_pTextWriter->SetCursorX(m_MenuTextPos[i].x + (m_MenuSize[i].x - m_pTextWriter->CalculateStringWidth(m_MenuText[i].c_str())) / 2.f);

            m_pTextWriter->SetTextColor(static_cast<int32_t>(i) == m_SelectMenuNumber ? nn::util::Color4u8::Yellow() :
                ApplicationState::IsHandheldMode() ? nn::util::Color4u8::Gray() : nn::util::Color4u8::White());
            m_pTextWriter->Print(m_MenuText[i].c_str());
        }
        // info
        m_pTextWriter->SetScale(1.f, 1.f);
        if (m_InfoLinePos > m_InfoLineLeftPos + 256)
        {
            nn::util::Color4u8 color = nn::util::Color4u8::Gray();
            color.SetA(static_cast<uint8_t>(255 * std::min(m_InfoLinePos - (m_InfoLineLeftPos + 256.f), 256.f) / 256.f));
            m_pTextWriter->SetTextColor(color);
            m_pTextWriter->SetCursor(m_InfoLinePos - 112.f, 44.f);
            m_pTextWriter->Print("ShowInfo");
        }
        if (m_InfoLinePos < m_InfoLineLeftPos + 256)
        {
            nn::util::Color4u8 color = nn::util::Color4u8::Gray();
            color.SetA(static_cast<uint8_t>(255 * std::min((m_InfoLineLeftPos + 256) - m_InfoLinePos, 256.f) / 256.f));
            m_pTextWriter->SetTextColor(color);
            m_pTextWriter->SetCursor(m_InfoLinePos + 80.f, 720.f - 68.f);
            m_pTextWriter->Print("CloseInfo");
        }
        for (size_t i = 0; i < InfoTextType_Num; ++i)
        {
            m_pTextWriter->SetTextColor(nn::util::Color4u8::Gray());
            m_pTextWriter->SetCursor(m_InfoLinePos + m_InfoTextPos[i].x, m_InfoTextPos[i].y);
            m_pTextWriter->Print("%s\n\n", m_InfoText[i].c_str());
            switch (static_cast<InfoTextType>(i))
            {
            case InfoTextType_PlayMode:
                m_pTextWriter->SetTextColor(m_SelectMenuNumber == 0 ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::Gray());
                m_pTextWriter->Print("Single");
                m_pTextWriter->SetCursorX(m_InfoLinePos + m_InfoTextPos[i].x + (1280.f - m_InfoLineLeftPos) / 2.f);
                m_pTextWriter->SetTextColor(m_SelectMenuNumber != 0 ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::Gray());

                m_pTextWriter->Print("Multi");
                break;
            case InfoTextType_PlayerCount:
                for (size_t cnt = 1; cnt < NN_ARRAY_SIZE(ApplicationState::NpadIds); ++cnt)
                {
                    m_pTextWriter->SetTextColor(static_cast<size_t>(m_SelectMenuNumber + 1) == cnt ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::Gray());
                    m_pTextWriter->SetCursorX(m_InfoLinePos + m_InfoTextPos[i].x + ((1280.f - m_InfoLineLeftPos - m_InfoTextPos[i].x) / static_cast<float>(NN_ARRAY_SIZE(ApplicationState::NpadIds) - 1)) * (cnt - 1));
                    m_pTextWriter->Print("%d", cnt);
                }
                break;
            case InfoTextType_MasterPlayerNpadId:
                m_pTextWriter->SetTextColor(!ApplicationState::IsHandheldMode() ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::Gray());
                m_pTextWriter->Print("No1");
                m_pTextWriter->SetCursorX(m_InfoLinePos + m_InfoTextPos[i].x + (1280.f - m_InfoLineLeftPos) / 2.f);
                m_pTextWriter->SetTextColor(ApplicationState::IsHandheldMode() ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::Gray());
                m_pTextWriter->Print("Handheld");
                break;
            case InfoTextType_EnableNpadStyle:
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    nn::os::UnlockMutex(&g_Mutex);
}
