﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "Butterworth.h"
#include "VibrationDecoder.h"
#include <cstdio>


namespace nns
{


/**
 * @brief Real-time vibration encoder
 *
 * Real time version of the `VibrationConverterConsole` tool. The principles in boths encoders are
 * similar but the process is not exactly the same. In particular, the PC tool performs a normalization
 * pass after encoding. Due to its on-the-fly nature, the present encoder cannot perform such a pass,
 * which means that you will likely notice intensity differences between the outputs of the two encoders.
 *
 * Input to the encoder is mono audio data whose sampling rate must be a multiple of 200 Hz and
 * of the encoder's internal sampling rate (see details below). Data is processed in chunks of
 * 5 ms, each chunk produces a vibration value that can then be sent to the controllers using the
 * `nn::hid::SendVibrationValue()` function.
 *
 * If you intend to encode several streams, you should use one encoder instance per stream.
 *
 * Bands
 * =====
 *
 * The encoder works by splitting the audio signal into two differents bands. By default, the two
 * bands are 128-200 Hz and 256-400 Hz, but these values can be set to anything you like in the
 * [10; 2500] Hz range.
 *
 * However, considering the controllers' characteristics, values outside the [40; 800] Hz range
 * are unlikely to be useful.
 *
 * Performance
 * ===========
 *
 * With its default settings, the encoder can process a 48 kHz stream using only 0.2% of a single
 * CPU core. Therefore, encoding eight such streams can be done with less than 2% of a CPU core.
 *
 * The easiest way to improve performance is to feed the encoder with data at lower sampling rates.
 * For instance, 32 kHz data is 20% faster than 48 kHz data.
 *
 * You may also reduce the encoder's internal sampling rate. This sampling rate must be a multiple
 * of 200 Hz and be such that the input sampling rate is a multiple of it. Please note that it is
 * not possible to extract frequencies above half the internal sampling rate (Shannon-Nyquist theorem).
 */
class VibrationEncoder
{
public:
    static const uint32_t MinSamplingRate       =  4000; ///< Lowest supported sampling rate, in Hz
    static const uint32_t MaxSamplingRate       = 48000; ///< Highest supported sampling rate, in Hz
    static const uint32_t DefaultResamplingRate =  8000; ///< Default resampling rate, in Hz
    static const uint32_t ChunkDurationMs       = 5;     ///< Chunk duration in ms
private:
    static const size_t MaxChunkLength = (MaxSamplingRate * ChunkDurationMs) / 1000;

private:
    typedef Butterworth Filter;

public:
    struct Band
    {
        Filter lowPass;
        Filter highPass;
        size_t lastCrossing;
        int    prevSample;
        float  prevAmplitude;
        float  prevPitch;
    };

public:
    nn::Result Init(uint32_t samplingRate, uint32_t resamplingrate=DefaultResamplingRate) NN_NOEXCEPT;
    nn::Result Init(uint32_t samplingRate, uint32_t resamplingrate, FILE* trace) NN_NOEXCEPT;
    nn::Result Init(uint32_t samplingRate, float lowFreq1, float highFreq1, float lowFreq2, float highFreq2, uint32_t resamplingRate=DefaultResamplingRate) NN_NOEXCEPT;
    nn::Result Init(uint32_t samplingRate, float lowFreq1, float highFreq1, float lowFreq2, float highFreq2, uint32_t resamplingRate, FILE* trace) NN_NOEXCEPT;

    void ProcessChunk(const int16_t* samples, nn::hid::VibrationValue* result) NN_NOEXCEPT;
    void ProcessChunk(const int16_t* samples, nn::hid::VibrationValue* result, FILE* trace) NN_NOEXCEPT;

private:
    void ExtractBand(const int16_t* samples, uint32_t bandNumber, float& amplitude, float& pitch) NN_NOEXCEPT;

private:
    Band             m_bands[2];             ///< Current state of each band
    VibrationDecoder m_decoder;              ///< The decoder used to compute attenuation
    uint32_t         m_samplingRate;         ///< The original data's sampling rate
    uint32_t         m_resamplingRate;       ///< The encoder's internal sampling rate
    uint32_t         m_downsamplingFactor;   ///< The downsampling factor
    int32_t          m_downsamplingFilter;   ///< Number of the band whose low-pass filter to use for downsampling. Negative when none.
    uint32_t         m_chunkLength;          ///< Length of a resampled chunk, in samples
    size_t           m_samplesProcessed;     ///< How many samples have been processed so far
    int16_t          m_tmp1[MaxChunkLength]; ///< Temporary work buffer #1
    int16_t          m_tmp2[MaxChunkLength]; ///< Temporary work buffer #2
};


} // namespace nns
