﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <new>
#include <string>
#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>
#include <nn/hid/hid_VibrationTarget.h>
#include <nn/hid/hid_VibrationWriter.h>
#include "VibrationEncoder.h"


#define ENABLE_TRACES  0 // NOLINT(preprocessor/const)


namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int SeCount = 4;

const char Title[] = "HdRumbleAudioEncoder";


const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};


int16_t clamp(int32_t x)
{
    const int32_t max = static_cast<int32_t>(std::numeric_limits<int16_t>::max());
    const int32_t min = static_cast<int32_t>(std::numeric_limits<int16_t>::min());
    x = x < max ? x : max;
    x = x > min ? x : min;
    return static_cast<int16_t>(x);
}

nn::audio::I3dl2ReverbParameterSet i3dl2ReverbParameterUserSet =
{
    -1000.f,    //roomGain;
    -1000.f,    //roomHfGain
    1.0f,       //lateReverbDecayTime
    0.3f,       //lateReverbHfDecayRatio
    -500.f,     //reflectionsGain;
    0.01f,      //reflectionsDelayTime;
    -310.f,     //reverbGain;
    0.04f,      //reverbDelayTime;
    80.f,       //reverbDiffusion;
    60.f,       //reverbDensity;
    5000.f,     //hfReference
    0.2f,       //dryGain;
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             StartSound                   (SampleSe0)\n");
    NNS_LOG("[B]             StartSound                   (SampleSe1)\n");
    NNS_LOG("[X]             StartSound                   (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound                   (SampleSe3)\n");
    NNS_LOG("[R]             Toggle HD Rumble Audio Encoding\n");
    NNS_LOG("[L]             Update i3dl2 Parameters\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

int UpdateI3dl2ReverbParameter(nn::audio::I3dl2ReverbType* pReverb, int updateMode, std::string& message)
{
    // Reverb のパラメータを更新します。
    nn::audio::I3dl2ReverbParameterSet parameterSet;
    parameterSet = nn::audio::GetI3dl2ReverbParameters(pReverb); //use the original parameter set when re-enable

    switch (updateMode)
    {
    case nn::audio::I3dl2ReverbType::Preset_SmallRoom:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_SmallRoom);
        message = "SmallRoom";
        break;
    case nn::audio::I3dl2ReverbType::Preset_LargeRoom:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_LargeRoom);
        message = "LargeRoom";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Hall:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_Hall);
        message = "Hall";
        break;
    case nn::audio::I3dl2ReverbType::Preset_CavernousCathedral:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_CavernousCathedral);
        message = "Cathedral";
        break;
    case nn::audio::I3dl2ReverbType::Preset_MetalCorridor:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_MetalCorridor);
        message = "Corridor";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count:
        parameterSet = i3dl2ReverbParameterUserSet;
        message = "UserDefine ";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count + 1:
        // Reverb エフェクトの無効化します。
        nn::audio::SetI3dl2ReverbEnabled(pReverb, false);
        message = "disabled";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count + 2:
        // 有効にする際に、Reverb エフェクトの内部状態はリセットされます。
        // よって前回無効化したタイミングで鳴っていた残響音はすべてクリアされます。
        nn::audio::SetI3dl2ReverbEnabled(pReverb, true);
        message = "re-enabled";
        break;
    default:
        break;
    }

    nn::audio::SetI3dl2ReverbParameters(pReverb, &parameterSet);
    return (updateMode + 1) % 8;
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    const nn::hid::NpadIdType npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld
    };
    const size_t padIdCount = sizeof(npadIds) / sizeof(npadIds[0]);
    nn::hid::SetSupportedNpadIdType(npadIds, padIdCount);

    // Retrieve HID device handles
    const size_t maxDeviceCount = 10;
    nn::hid::VibrationDeviceHandle handles[maxDeviceCount] = {};
    int deviceCount = 0;
    NNS_LOG("Initialize Rumble Pads\n");
    for (size_t i=0; i<padIdCount; ++i)
    {
        nn::hid::NpadStyleSet mask = (npadIds[i]==nn::hid::NpadId::Handheld) ? nn::hid::NpadStyleHandheld::Mask : nn::hid::NpadStyleFullKey::Mask;
        deviceCount += nn::hid::GetVibrationDeviceHandles(handles + deviceCount, maxDeviceCount - deviceCount, npadIds[i], mask);
    }

    // Print HID device info
    for (int i = 0; i < deviceCount; ++i)
    {
        nn::hid::InitializeVibrationDevice(handles[i]);

        nn::hid::VibrationDeviceInfo deviceInfo;
        nn::hid::GetVibrationDeviceInfo(&deviceInfo, handles[i]);
    }

#if ENABLE_TRACES
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
    FILE* trace = fopen("C:\\Temp\\VibrationEncoder.txt", "w");
#else
    FILE* trace = NULL;
#endif

    int i3dl2ReverbParameterIndex = 0;
    std::string i3dl2ReverbState = "default";
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 4;
    parameter.voiceCount = 4;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 0;
    auto hdRumbleSampleRate = 8000;
    nns::VibrationEncoder encoder;
    NN_ABORT_UNLESS_RESULT_SUCCESS(encoder.Init(parameter.sampleRate, hdRumbleSampleRate, trace));


    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;
    int8_t subBusA[2];
    subBusA[nn::audio::ChannelMapping_FrontLeft] = 2;
    subBusA[nn::audio::ChannelMapping_FrontRight] = 3;
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    NNS_LOG("Open AudioRenderer\n");
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 4);

    nn::audio::DeviceSinkType deviceSink;

    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);

    rt = RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    //i3dl2
    nn::audio::I3dl2ReverbType i3dl2Reverb;
    size_t i3dl2ReverbSize = nn::audio::GetRequiredBufferSizeForI3dl2Reverb(parameter.sampleRate, sizeof(subBusA));
    void* i3dl2ReverbBuffer = g_EffectBufferAllocator.Allocate(i3dl2ReverbSize);
    NN_ABORT_UNLESS(i3dl2ReverbBuffer != nullptr);

    // Reverb を FinalMix に追加します。
    result = nn::audio::AddI3dl2Reverb(&config, &i3dl2Reverb, i3dl2ReverbBuffer, i3dl2ReverbSize, &finalMix, sizeof(subBusA));
    NN_ABORT_UNLESS(result.IsSuccess());

    // Reverb
    nn::audio::SetI3dl2ReverbInputOutput(&i3dl2Reverb, subBusA, subBusA, sizeof(subBusA));

    nn::audio::I3dl2ReverbParameterSet i3dl2Parameter;
    nn::audio::LoadI3dl2ReverbPreset(&i3dl2Parameter, nn::audio::I3dl2ReverbType::Preset_SmallRoom);
    nn::audio::SetI3dl2ReverbParameters(&i3dl2Reverb, &i3dl2Parameter);


    // Aux buffer for HD Rumble encoding
    nn::audio::AuxType aux;

    const int AuxBufferFrameCount = 4;
    size_t auxBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, AuxBufferFrameCount, sizeof(subBusA));
    void* sendBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(sendBuffer);
    void* returnBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(returnBuffer);

    result = nn::audio::AddAux(&config, &aux, &finalMix, sendBuffer, returnBuffer, auxBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    int auxInputChannelCount = sizeof(subBusA);
    nn::audio::SetAuxInputOutput(&aux, subBusA, subBusA, auxInputChannelCount);

    int32_t* readBuffer = static_cast<int32_t*>(g_Allocator.Allocate(auxBufferSize));
    int16_t* reducedDataLengthBuffer = static_cast<int16_t*>(g_Allocator.Allocate(auxBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    NN_ABORT_UNLESS_NOT_NULL(reducedDataLengthBuffer);
    memset(readBuffer, 0, auxBufferSize);
    memset(reducedDataLengthBuffer, 0, auxBufferSize);

    nn::audio::BufferMixerType mixer;

    result = nn::audio::AddBufferMixer(&config, &mixer, &finalMix);
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::audio::SetBufferMixerInputOutput(&mixer, subBusA, mainBus, channelCount);

    nn::audio::SetBufferMixerVolume(&mixer, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixer, 1, 1.0f);

    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());

    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        if(i != 3)
            nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
         nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 1.0f, 0, subBusA[0]);
         nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 1.0f, 0, subBusA[1]);
    }

    PrintUsage();
    for (;;)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        for (int i = 0; i < SeCount; ++i)
        {
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                    NNS_LOG("Playing Sound: %s\n", g_SeFileNames[i]);
                }
                if((nn::audio::GetVoicePlayState(&voiceSe[i]) == nn::audio::VoiceType::PlayState_Stop) || (nn::audio::GetVoicePlayState(&voiceSe[i]) == nn::audio::VoiceType::PlayState_Pause))
                {
                    nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
                    NNS_LOG("Playing Sound: %s\n", g_SeFileNames[i]);
                }
            }
        }
        if ( npadButtonDown.Test<::nn::hid::NpadButton::R>())
        {
            if(nn::audio::IsAuxEnabled(&aux))
            {
                nn::audio::SetAuxEnabled(&aux, false);
                //clear out vibration
                nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make(0.0f, 0.0f, 0.0f, 0.0f);
                for(int j = 0; j < deviceCount; ++j)
                {
                    nn::hid::SendVibrationValue(handles[j], vib);
                }
                NNS_LOG("HD Rumble Encoding Off\n");
            }
            else
            {
                nn::audio::SetAuxEnabled(&aux, true);
                NNS_LOG("HD Rumble Encoding On\n");
            }
        }
        if(npadButtonDown.Test<::nn::hid::NpadButton::L>())
        {
            i3dl2ReverbParameterIndex = UpdateI3dl2ReverbParameter(&i3dl2Reverb, i3dl2ReverbParameterIndex, i3dl2ReverbState);
            NNS_LOG("I3dl2Reverb:%-15s\n", i3dl2ReverbState.c_str());
        }
        // GetAuxBuffer with audio data
        if(nn::audio::IsAuxEnabled(&aux))
        {
            int readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, nn::audio::GetAuxSampleCount(&aux));
            const int chunkLengthOfAudio = (parameter.sampleRate * nns::VibrationEncoder::ChunkDurationMs) / 1000;

            for (auto i = 0; i < readCount; ++i)
            {
                reducedDataLengthBuffer[i] = clamp(readBuffer[i]);
            }

            for(auto i = 0; i < readCount; i += chunkLengthOfAudio)
            {
                nn::hid::VibrationValue vib;
                encoder.ProcessChunk((reducedDataLengthBuffer + i), &vib, trace);

                for(int j = 0; j < deviceCount; ++j)
                {
                    nn::hid::SendVibrationValue(handles[j], vib);
                }
            }
            nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        }
        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS(result.IsSuccess());

    }

    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
    if(i3dl2ReverbBuffer)
    {
        g_EffectBufferAllocator.Free(i3dl2ReverbBuffer);
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
