﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// This is skeleton code for gfx0 client process.
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nns/gfx0.h>
#include <nns/display0.h>

NN_ALIGNAS(4096) uint32_t   g_DeviceMemory[4096 / sizeof(uint32_t)];

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( nn::os::MemoryBlockUnitSize * 4 );
}

//-----------------------------------------------------------------------------
// Create TransferMemory and provide it to server.
//
void PrepareTranfermemory(void* address, size_t size)
{
    // Create TransferMemory and provide this to server.
    nn::os::TransferMemoryType  transferMemory;
    auto result = nn::os::CreateTransferMemory(
                                            &transferMemory,
                                            address,
                                            size,
                                            nn::os::MemoryPermission_ReadWrite);
    NN_ASSERT(result.IsSuccess());

    // Send transfer memory via gfx0 driver.
    nns::gfx0::AttachTransferMemory(&transferMemory, size);

    nn::os::DestroyTransferMemory(&transferMemory);
}

//-----------------------------------------------------------------------------
// Prepare SharedMemory
//
char* PrepareSharedMemory(nn::os::SharedMemoryType* pSharedMemory)
{
    // Query TransferMemory to server.
    nns::gfx0::QuerySharedMemory( pSharedMemory );

    // Mapping to virtual space
    void* address = nn::os::MapSharedMemory(pSharedMemory, nn::os::MemoryPermission_ReadWrite);
    NN_ASSERT(address != NULL);
    return static_cast<char*>( address );
}

//-----------------------------------------------------------------------------
// Execute Demo
//
void ExecuteDemo(char* address)
{
    nn::os::TimerEvent  timer(nn::os::EventClearMode_AutoClear);

    timer.StartPeriodic( nn::TimeSpan::FromMilliSeconds(100),
                         nn::TimeSpan::FromMilliSeconds(100) );

    for (int i=0; i<32; ++i)
    {
        timer.Wait();
        nns::gfx0::AdvanceDemo();

        NN_LOG("shmem_value=%s", address);

        nns::gfx0::AttachDeviceMemory( reinterpret_cast<uintptr_t>(g_DeviceMemory), sizeof(g_DeviceMemory));
        auto value = g_DeviceMemory[0];
        NN_LOG("   das_value=0x%08x\n", value);
    }
}

//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    // Try to connect gfx0 driver server
    nns::gfx0::InitializeGfx0Driver();

    // Get memory block from heap.
    uintptr_t   address;
    auto result = nn::os::AllocateMemoryBlock(&address, nn::os::MemoryBlockUnitSize);
    NN_ASSERT(result.IsSuccess());

    // Transfer Memory
    PrepareTranfermemory(reinterpret_cast<void*>(address), nn::os::MemoryBlockUnitSize);

    // Shared Memory
    nn::os::SharedMemoryType    sharedMemory;
    char* mappedAddress = PrepareSharedMemory(&sharedMemory);

    // System Event
    nn::os::SystemEventType     systemEvent;
    nns::gfx0::QuerySystemEvent(&systemEvent);

    // Start Display0 service
    nns::display0::StartDisplay0(&systemEvent);

    // Execute Demo
    ExecuteDemo( reinterpret_cast<char*>(mappedAddress) );

    // Stop Display0 service
    nns::display0::StopDisplay0();

    // Disconnect from gfx0 server
    nns::gfx0::FinalizeGfx0Driver();
}

//-----------------------------------------------------------------------------

