﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SkeletalAnimation.cpp,PageSampleG3dSkeletalAnimation}
 *
 * @brief
 * スケルタルアニメーション機能を使用するサンプルプログラム
 */

/**
 * @page PageSampleG3dSkeletalAnimation G3dDemo SkeletalAnimation
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\skeletalanimation.png
 *
 * @brief
 * サンプルプログラム SkeletalAnimation の解説です。
 *
 * @section PageSampleG3dSkeletalAnimation_SectionBrief 概要
 * g3d のスケルタルアニメーション機能を使用したサンプルプログラムです。
 *
 * nn::g3d::SkeletalAnimObj の初期化時、再生し得るアニメーションリソースを再生できるだけのメモリを nn::g3d::SkeletalAnimObj に確保し、
 * 描画ループ内で skeletalAnimObj::SetResource() を使用して再生するアニメーションリソースを切り替えます。
 * メニューからアニメーションを選択、再生することでスケルタルアニメーションの挙動を確認できます。
 * Pierrot と Robot の 2 つのモデルに対して、Pierrot 向けに作成したスケルタルアニメーションを適用します。
 *
 * 本サンプルプログラムでは、以下のスケルタルアニメーションの機能を使用します。
 * - アニメーションブレンド
 *
 * nn::g3d::SkeletalAnimBlender を利用して複数のスケルタルアニメーションをブレンド再生します。
 *
 * - リターゲティング
 *
 * 2つのスケールが異なるモデル Pierrot と Robot に対して、Pierrot 向けのスケルタルアニメーションを適用します。
 * Robot にアニメーションを通常適用した場合、スケールの違いを無視して再生されるため、アニメーション挙動が Pierrot と同じ見た目になってしまいます。
 * リターゲティングを有効にして Robot にアニメーションを適用した場合、モデルのスケールの差異を補正された上で再生されます。
 * リターゲティングによって 1 つのアニメーションリソースを他のモデルに適用できるため、アニメーション作成コストを削減することが可能です。
 *
 * また、以下のアニメーションの機能を使用します。
 * - アニメーションベイク (g_BakeAnimEnabled = true で有効化)
 * - PreBind と BindFast 機能
 *
 * @section PageSampleG3dSkeletalAnimation_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dSkeletalAnimation_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dSkeletalAnimation_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左スティックまたはマウス左ボタン押し </td><td> カメラ回転。マウスの場合、画面の中心点からクリック位置への方向、距離によって回転します。 </td></tr>
 * <tr><td> 右スティックまたはマウス右ボタン押し </td><td> カメラ移動。マウスの場合、画面の中心点からクリック位置への方向、距離によって移動します。 </td></tr>
 * </table>
 * </p>
 *
 * Pierrot と Robot モデルに対して、各アニメーションの項目を選択して以下の挙動を確認することが可能です。
 * <p>
 * <table>
 * <tr><th> 項目 </th><th> 動作 </th></tr>
 * <tr><td> 十字ボタン上下またはカーソルキー上下 </td><td> エディット項目の選択を切り替えます。 </td></tr>
 * <tr><td> 十字ボタン左右またはカーソルキー左右 </td><td> 選択されているエディット項目の値を切り替えます。 </td></tr>
 * <tr><td> AnimationPreset </td><td> 再生するアニメーションプリセットを左右で選択します。 選択したアニメーションが自動で再生されます。</td></tr>
 * <tr><td> Retargetng </td><td> Robot モデルに対して Pierrot モデルへのリターゲティング機能を適用します。</td></tr>
 * <tr><td> ViewSkeleton </td><td> モデルのスケルトン構造を可視化します。 </td></tr>
 * <tr><td> AnimStep </td><td> アニメーションの再生速度を変更します </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dSkeletalAnimation_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleG3dSkeletalAnimation_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから SkeletalAnimation を選択し、実行してください。
 *
 * @section PageSampleG3dSkeletalAnimation_SectionDetail 解説
 * このサンプルプログラムは、g3d のスケルタルアニメーション機能を利用してモデルのアニメーションを行います。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャー初期化
 * - サンプラーをディスクリプタプールに登録
 * - nns::g3d を使用した 2 つのモデルのセットアップ
 * - スロット数分の SkeletalAnimObj の作成
 * - ループ開始
 * - 前フレームで作成したコマンドリスト実行
 * - メニューで選択されたアニメーションリソースを SkeletalAnimObj に割り当て
 * - 各行列更新
 * - モデル更新
 * - コマンドリスト生成
 * - ループ開始に戻る
 *
 */

#include "SkeletalAnimation.h"
#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"

namespace g3ddemo = nn::g3d::demo;

namespace nn { namespace g3d { namespace demo { namespace skeletalanim {

#define RESOURCE_PATH "Resource:/"
const char* g_BfshaPath = RESOURCE_PATH "shader/demo.bfsha";
const char* g_EnvBfresPath = RESOURCE_PATH "env.bfres";
const char* g_ModelBfresPath = RESOURCE_PATH "SkeletalAnimDemo.bfres";

nns::g3d::RenderModelObj*    g_pRenderModelObj[ModelCount];
nn::util::Matrix4x3fType     g_BaseMtx[ModelCount];
nn::g3d::SkeletalAnimObj*    g_pSkeletalAnimObjArray[ModelCount][AnimIndexCount];
nn::g3d::SkeletalAnimBlender g_SkeletalAnimBlender[ModelCount];
nn::util::Color4u8           g_MeterColor[ModelCount];

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraUp;
nn::util::Vector3fType g_CameraTarget;

nns::g3d::RenderView    g_RenderView;
g3ddemo::ResourceHolder g_Holder;

const int g_BufferingCount = 2;
const int g_InvalidIndex = -1;

// アニメーションをベイクするかどうかのフラグ
const bool g_BakeAnimEnabled = false;

void ResetAnim()
{
    for (int animIndex = 0; animIndex < AnimIndexCount; ++animIndex)
    {
        nn::g3d::ResSkeletalAnim* pNewResAnim = NULL;
        for (int holderAnimIndex = 0; holderAnimIndex < g_Holder.modelAnims.GetCount(); holderAnimIndex++)
        {
            nn::g3d::ResSkeletalAnim* pTmpAnim = reinterpret_cast<nn::g3d::ResSkeletalAnim*>(g_Holder.modelAnims[holderAnimIndex]);
            if (strcmp(GetSelectedAnimationPreset()->name[animIndex], pTmpAnim->GetName()) == 0)
            {
                pNewResAnim = pTmpAnim;
                break;
            }
        }

        if (pNewResAnim)
        {
            for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
            {
                // アニメーションを再生するモデルのスケルトンリソースをバインド
                nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = g_pSkeletalAnimObjArray[modelIndex][animIndex];
                pSkeletalAnimObj->SetResource(pNewResAnim);
                nn::g3d::SkeletalAnimObj::BindArgument bindArg;
                bindArg.SetResource(g_pRenderModelObj[modelIndex]->GetRenderModel()->GetResModel()->GetSkeleton());
                switch (modelIndex)
                {
                case Model_Pierrot:
                    {
                        // Pierrot モデルは PreBind しているので高速な BindFast が可能
                        pSkeletalAnimObj->BindFast(bindArg);
                    }
                    break;
                case Model_Robot:
                    {
                        if (IsRetargetingEnabled())
                        {
                            // リターゲティング有効の場合は host に Pierrot リソースを指定する
                            bindArg.SetHostResource(g_pRenderModelObj[Model_Pierrot]->GetRenderModel()->GetResModel()->GetSkeleton());
                            bindArg.SetRetargetingEnabled();
                        }
                        pSkeletalAnimObj->Bind(bindArg);
                    }
                    break;
                default:
                    break;
                }
                pSkeletalAnimObj->GetFrameCtrl().SetStep(GetAnimStep());
                pSkeletalAnimObj->GetFrameCtrl().SetFrame(0.f);
            }
        }
        else
        {
            for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
            {
                nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = g_pSkeletalAnimObjArray[modelIndex][animIndex];
                g_pRenderModelObj[modelIndex]->GetModelObj()->GetSkeleton()->ClearLocalMtx();
                pSkeletalAnimObj->ResetResource();
                pSkeletalAnimObj->GetFrameCtrl().SetFrame(0.f);
            }
        }
    }
}

void CalculateAnim(int modelIndex)
{
    nn::g3d::ModelObj* pModelObj = g_pRenderModelObj[modelIndex]->GetModelObj();

    // アニメーション更新
    int singleAnimIndex = g_InvalidIndex;
    int skeletalAnimCount = 0;
    float weightSum = 0.0f;

    // 有効アニメーション数と weight 値の計算
    for (int animIndex = 0; animIndex < AnimIndexCount; ++animIndex)
    {
        if (g_pSkeletalAnimObjArray[modelIndex][animIndex]->IsTargetBound())
        {
            ++skeletalAnimCount;
            weightSum += GetSelectedAnimationPreset()->weight[animIndex];
            singleAnimIndex = animIndex;
        }
    }
    if (skeletalAnimCount == 1)
    {
        // 単一スロットのアニメーションが有効な場合は単独再生
        nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = g_pSkeletalAnimObjArray[modelIndex][singleAnimIndex];
        pSkeletalAnimObj->Calculate();
        pSkeletalAnimObj->ApplyTo(pModelObj);
        pSkeletalAnimObj->GetFrameCtrl().UpdateFrame();
    }
    else if (skeletalAnimCount > 1)
    {
        // 複数スロットのアニメーションが有効な場合はブレンド再生を行う
        // 合計 weight 値が 1 になっていない場合 nn::g3d::SkeletalAnimBlender::ApplyTo() 内で正規化されるが
        // 正規化の処理コストが高くつくため、アプリ側で weight 値の合計を 1 にしておく
        nn::g3d::SkeletalAnimBlender& blender = g_SkeletalAnimBlender[modelIndex];
        float invWeightSum = nn::util::Rcp(weightSum);
        blender.ClearResult();
        for (int animIndex = 0; animIndex < AnimIndexCount; ++animIndex)
        {
            nn::g3d::SkeletalAnimObj* pSkeletalAnimObj = g_pSkeletalAnimObjArray[modelIndex][animIndex];
            if (pSkeletalAnimObj->IsTargetBound())
            {
                blender.Blend(pSkeletalAnimObj, GetSelectedAnimationPreset()->weight[animIndex] * invWeightSum);
                pSkeletalAnimObj->GetFrameCtrl().UpdateFrame();
            }
        }
        blender.ApplyTo(pModelObj->GetSkeleton());
    }
}

void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData)
{
    NN_UNUSED(pGfxFramework);
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);
    g3ddemo::Pad& pad = g3ddemo::GetPad();

    // メニューの更新 & メニューで選択されたアニメーションの切り替え
    switch (CalculateMenu(pad))
    {
    case MenuResult_ChangeSelectedAnim:
        {
            ResetAnim();
        }
        break;
    case MenuResult_ChangeAnimStep:
        {
            for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
            {
                for (int animIndex = 0; animIndex < AnimIndexCount; ++animIndex)
                {
                    g_pSkeletalAnimObjArray[modelIndex][animIndex]->GetFrameCtrl().SetStep(GetAnimStep());
                }
            }
        }
        break;
    default:
        break;
    }

    // カメラコントロール
    ControlCamera(&g_CameraPosition, &g_CameraUp, &g_CameraTarget, &g_RenderView.GetViewMtx(0), &pad);
    nn::util::Matrix4x3fType viewMatrix;
    MatrixLookAtRightHanded(&viewMatrix,
        g_CameraPosition,
        g_CameraTarget,
        g_CameraUp);
    g_RenderView.SetViewMtx(0, viewMatrix);

    // 投影行列
    nn::util::Matrix4x4fType projectionMatrix;
    MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
    g_RenderView.SetProjectionMtx(0, projectionMatrix);

    // ビューの更新処理
    g_RenderView.Calculate(uniformBufferIndex);

    // モデルの更新処理
    for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
    {
        NN_PERF_SET_COLOR(g_MeterColor[modelIndex]);
        NN_PERF_AUTO_MEASURE();
        nn::g3d::ModelObj* pModelObj = g_pRenderModelObj[modelIndex]->GetModelObj();

        // アニメーション更新
        CalculateAnim(modelIndex);

        // ワールド座標更新
        pModelObj->CalculateWorld(g_BaseMtx[modelIndex]);

        // ユニフォームブロック更新
        pModelObj->CalculateView(0, g_RenderView.GetViewMtx(0), uniformBufferIndex);
        pModelObj->CalculateMaterial(uniformBufferIndex);
        pModelObj->CalculateSkeleton(uniformBufferIndex);
        pModelObj->CalculateShape(uniformBufferIndex);
    }
}

void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData)
{
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // コマンド生成
    pGfxFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        // カラーターゲットをクリア
        nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
        pCommandBuffer->ClearColor(pColor, 0.3f, 0.3f, 0.3f, 1.0f, nullptr);

        // 深度ステンシルをクリア
        nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
        pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        // レンダーターゲットをセット
        pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

        // ビューポートシザーステートをセット
        pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

        // モデル表示 コマンド生成
        for (int modelIndex = 0; modelIndex < g_Holder.renderModelObjs.GetCount(); ++modelIndex)
        {
            NN_PERF_SET_COLOR_GPU(g_MeterColor[modelIndex]);
            NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);
            g_Holder.renderModelObjs[modelIndex]->Draw(pCommandBuffer, 0, uniformBufferIndex);
            NN_PERF_END_MEASURE_GPU(pCommandBuffer);
        }

        // スケルトンデバッグ描画
        if (IsViewSkeletonEnabled())
        {
            for (int index = 0; index < g_Holder.renderModelObjs.GetCount(); ++index)
            {
                nn::g3d::SkeletonObj* skeletonObj = g_Holder.renderModelObjs[index]->GetModelObj()->GetSkeleton();
                DrawSkeleton(pCommandBuffer, skeletonObj);
            }
        }

        // テキスト表示 コマンド生成
        DrawMenu(pCommandBuffer);
    }
    pGfxFramework->EndFrame(bufferIndex);
}

nns::g3d::RenderModelObj* CreateRenderModelObj(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, nn::g3d::ResShaderArchive* pDemoShaderArchive)
{
    nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
    pRenderModel->CreateDrawPass(pDevice, pDemoShaderArchive);
    g_Holder.renderModels.PushBack(pRenderModel);

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.ShapeBufferingCount(g_BufferingCount);
    builder.SkeletonBufferingCount(g_BufferingCount);
    builder.MaterialBufferingCount(g_BufferingCount);
    nn::g3d::ModelObj* pModelObj = nns::g3d::CreateModelObj(pDevice, builder);
    g_Holder.modelObjs.PushBack(pModelObj);

    nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelObj, pRenderModel);
    g_Holder.renderModelObjs.PushBack(pRenderModelObj);
    g3ddemo::WriteSkinningOption(pRenderModelObj);
    pRenderModelObj->UpdateShader(pDevice);
    return pRenderModelObj;
}

// モデル * スロット数分の SkeletalAnimObj 、モデル数分の SkeletalAnimBlender を作成する。
// 本サンプルでは、SkeletalAnimObj の初期化時に、再生し得るアニメーションリソースを再生できるメモリを確保しておき
// 描画ループ内で skeletalAnimObj が再生するアニメーションを切り替える方法を取る
void InitializeSkeletalAnimObj(nn::g3d::ResFile* pResFile)
{
    nn::g3d::SkeletalAnimObj::Builder builderBase;
    for (int animIndex = 0; animIndex < pResFile->GetSkeletalAnimCount(); ++animIndex)
    {
        nn::g3d::ResSkeletalAnim* pAnim = pResFile->GetSkeletalAnim(animIndex);

        // アニメーションのベイク
        if (g_BakeAnimEnabled)
        {
            size_t bakeSize = pAnim->GetBakedSize();
            void* pBakeBuffer = nns::g3d::Allocate(bakeSize);
            bool isSucceed = pAnim->BakeCurve(pBakeBuffer, bakeSize);
            NN_ASSERT(isSucceed);
        }

        // 全てのアニメーションリソースをあらかじめ SkeletalAnimObj の builer に Reserve しておく
        // そうすることで SkeletalAnimObj を作り直すことなく、SkeletalAnimObj::SetResource() で再生するアニメーションリソースの切り替えが可能
        builderBase.Reserve(pAnim);

        // 高速 Bind のためアニメーションリソースに Pierrot モデルを PreBind しておく
        pAnim->PreBind(g_pRenderModelObj[Model_Pierrot]->GetRenderModel()->GetResModel()->GetSkeleton());
    }

    // モデル * スロット数分の SkeletalAnimObj を作成
    for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = g_pRenderModelObj[modelIndex]->GetRenderModel()->GetResModel();
        nn::g3d::SkeletalAnimObj::Builder builder = builderBase;
        builder.Reserve(pResModel);
        // Robot はリターゲティングを行う設定にしておく
        if (modelIndex == Model_Robot)
        {
            builder.SetRetargetingEnabled();
        }
        for (int animIndex = 0; animIndex < AnimIndexCount; ++animIndex)
        {
            g_pSkeletalAnimObjArray[modelIndex][animIndex] = nns::g3d::CreateAnimObj<nn::g3d::SkeletalAnimObj>(builder);
        }
        // SkeletalAnimBlender の設定
        nn::g3d::SkeletalAnimBlender::Builder skeletalAnimBlenderBuilder;
        skeletalAnimBlenderBuilder.Reserve(pResModel);
        skeletalAnimBlenderBuilder.CalculateMemorySize();
        size_t bufferSize = skeletalAnimBlenderBuilder.GetWorkMemorySize();
        void* pBlendBuffer = nns::g3d::Allocate(bufferSize, nn::g3d::SkeletalAnimBlender::Alignment_Buffer);
        skeletalAnimBlenderBuilder.Build(&g_SkeletalAnimBlender[modelIndex], pBlendBuffer, bufferSize);
        g_SkeletalAnimBlender[modelIndex].SetBoneCount(pResModel->GetSkeleton()->GetBoneCount());
    }
}

void Initialize()
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();
    pGfxFramework->InitializePerf();
#if defined( NN_BUILD_CONFIG_OS_WIN )
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(g3ddemo::GetFixedCoreNumber);
#endif
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    g_Holder.Initialize();
    InitializeMenu(pDevice);

    // シェーダーアーカイブの初期化
    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_BfshaPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
        pDemoShaderArchive = pFile->GetResShaderArchive();
    }

    // 環境モデルリソースの初期化
    nn::g3d::ResFile* pEnvModelFile;
    {
        pEnvModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_EnvBfresPath);
        pEnvModelFile->Setup(pDevice);
        g_Holder.files.PushBack(pEnvModelFile);
    }

    // モデルリソースの初期化
    nn::g3d::ResFile* pPierrotModelFile;
    {
        pPierrotModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_ModelBfresPath);
        pPierrotModelFile->Setup(pDevice);
        g3ddemo::SetupTexture(pDevice, pPierrotModelFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pPierrotModelFile);
        g_Holder.files.PushBack(pPierrotModelFile);
    }

    // モデルの初期化
    g_pRenderModelObj[Model_Pierrot] = CreateRenderModelObj(pDevice, pPierrotModelFile->FindModel("Pierrot"), pDemoShaderArchive);
    g_pRenderModelObj[Model_Robot] = CreateRenderModelObj(pDevice, pPierrotModelFile->FindModel("Robot"), pDemoShaderArchive);

    // SkeletalAnimObj の初期化
    InitializeSkeletalAnimObj(pPierrotModelFile);

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&g_Holder);

    // アニメーションの初期化
    ResetAnim();

    // Env モデルの初期化
    nn::g3d::ModelObj* pEnvModelObj;
    {
        nn::g3d::ResModel* pResModel = pEnvModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pDemoShaderArchive->FindShadingModel("env"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        pEnvModelObj->CalculateMaterial(0);
        pEnvModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pEnvModelObj);
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);
        buffers[1] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(1);
        g_pRenderModelObj[Model_Pierrot]->BindUniformBlock("env", buffers, g_BufferingCount);
        g_pRenderModelObj[Model_Robot]->BindUniformBlock("env", buffers, g_BufferingCount);
    }

    // ビュー用のユニフォームブロックを設定
    g_RenderView.Initialize(pDevice, 1, g_BufferingCount);
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(0, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(0, 1);
        g_pRenderModelObj[Model_Pierrot]->BindUniformBlock("view", buffers, g_BufferingCount);
        g_pRenderModelObj[Model_Robot]->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // カメラ初期化
    {
        VectorSet(&g_CameraPosition, 0, 110.f, 350.0f);
        VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
        VectorSet(&g_CameraTarget, 0.0f, 110.f, 0.0f);

        nn::util::Matrix4x3fType matrix;
        MatrixLookAtRightHanded(&matrix,
            g_CameraPosition,
            g_CameraTarget,
            g_CameraUp);
        g_RenderView.SetViewMtx(0, matrix);
    }

    // モデルの位置を設定
    for (int modelIndex = 0; modelIndex < ModelCount; ++modelIndex)
    {
        nn::util::Vector3fType translate;
        VectorSet(&translate, -80.f + (modelIndex * 160.f), 0, 0.0f);
        MatrixIdentity(&g_BaseMtx[modelIndex]);
        nn::util::MatrixSetTranslate(&g_BaseMtx[modelIndex], translate);
    }

    // モデルのパフォーマンスメーターの色を設定
    g_MeterColor[Model_Pierrot] = nn::util::Color4u8::Red();
    g_MeterColor[Model_Robot] = nn::util::Color4u8::Blue();

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);
}

// メインループ
void MainLoop()
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();

    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        uniformBufferIndex ^= 1;
    }
    pGfxFramework->QueueFinish();
}

void Finalize()
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // ベイクしたバッファーの破棄
    if (g_BakeAnimEnabled)
    {
        for (int animIndex = 0; animIndex < g_Holder.modelAnims.GetCount(); animIndex++)
        {
            void* ptr = g_Holder.modelAnims[animIndex];
            nn::g3d::ResSkeletalAnim* pAnim = reinterpret_cast<nn::g3d::ResSkeletalAnim*>(ptr);
            void* pBakeBuffer = pAnim->ResetCurve();
            nns::g3d::Free(pBakeBuffer);
        }
    }

    // 終了処理
    g3ddemo::DestroyAll(pDevice, &g_Holder);
    g_RenderView.Finalize(pDevice);
    FinalizeMenu(pDevice);
    pGfxFramework->FinalizePerf();
    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);
}

}}}} // namespace nn::g3d::demo::skeletalanim

//--------------------------------------------------------------------------------------------------
int SkeletalAnimationMain()
{
    g3ddemo::skeletalanim::Initialize();
    g3ddemo::skeletalanim::MainLoop();
    g3ddemo::skeletalanim::Finalize();
    return EXIT_SUCCESS;
}
