﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Simple.cpp,PageSampleG3dSimple}
 *
 * @brief
 * シンプルなモデル表示のサンプルプログラム
 */

/**
 * @page PageSampleG3dSimple G3dDemo Simple
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\simple.png
 *
 * @brief
 * サンプルプログラム Simple の解説です。g3d を使用したモデル描画のコード例を示します。
 *
 * @section PageSampleG3dSimple_SectionBrief 概要
 * g3d を使用したモデル描画のコード例を示すサンプルです。
 *
 * @section PageSampleG3dSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dSimple_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dSimple_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード による入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dSimple_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleG3dSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから Simple を選択し、実行してください。
 *
 * @section PageSampleG3dSimple_SectionDetail 解説
 * このサンプルプログラムは、モデルリソースを読み込み、アニメーションを適用し、
 * モデルを表示します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ブレンドステート、深度ステンシルステート、ラスタライザステートを初期化
 * - シェーダーアーカイブ初期化
 * - モデルリソース初期化およびテクスチャ初期化、テクスチャ、サンプラーをディスクリプタプールに登録
 * - 頂点ステートを初期化
 * - マテリアルのシェーダーパラメーター初期化
 * - モデルインスタンス作成
 * - シェーダーセレクター作成
 * - スケルタルアニメーションインスタンス作成
 * - g3dで管理されない描画時に必要となるユニフォームブロック初期化
 * - モデル表示用コマンドリスト作成
 * - ループ開始
 * - ビュー計算
 * - アニメーション更新
 * - モデル更新
 * - ビュー更新
 * - コマンドリスト実行
 * - スキャンバッファをスワップ
 * - ループ開始に戻る
 *
 */

#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#endif

namespace g3ddemo = nn::g3d::demo;

namespace {

#define RESOURCE_PATH  "Resource:/"

const char* g_ModelPath  = RESOURCE_PATH "human.bfres";
const char* g_ShaderPath = RESOURCE_PATH "shader/demo.bfsha";

const char* g_ModelName        = "human";
const char* g_SkeletalAnimName = "human_walk";

nn::gfx::BlendState        g_BlendState;
nn::gfx::DepthStencilState g_DepthStencilState;
nn::gfx::RasterizerState   g_RasterizerState;
nn::gfx::Buffer g_ViewBlock;
nn::gfx::Buffer g_EnvBlock;
ptrdiff_t g_ViewBlockMemoryPoolOffset = -1;
ptrdiff_t g_EnvBlockMemoryPoolOffset = -1;

// 外部メモリープールを使用するかどうかのフラグ。
// バイナリーは、バイナリー内のメモリープールもしくはアプリで用意した外部メモリープールを使用して初期化できます。
// バイナリー内のメモリープールを使用する場合は、メモリープールの管理が不要なので取り回しが楽になります。
// 一方、特定のプラットフォームでは、メモリープールをまとめて確保して扱うとパフォーマンスが高くなる場合があります。
// そのようなプラットフォームでは、アプリがまとめて確保した外部メモリープールを使用してバイナリーを初期化するといった選択が考えられます。
const bool g_IsUseExternalMemoryPool = true;
ptrdiff_t g_ModelMemoryPoolOffset = -1;
ptrdiff_t g_ShaderMemoryPoolOffset = -1;

//--------------------------------------------------------------------------------------------------

struct ViewBlock
{
    nn::util::FloatColumnMajor4x4 projMtx;
    nn::util::FloatColumnMajor4x3 cameraMtx;
};

struct EnvBlock
{
    nn::util::Float4 skyColor;
    nn::util::Float4 groundColor;
    nn::util::Float4 dirLightDir;
    nn::util::Float4 dirLightColor;
};

EnvBlock g_EnvBlockValue =
{
    NN_UTIL_FLOAT_4_INITIALIZER(0.7f, 0.85f, 1.0f, 0.0f), // skyColor
    NN_UTIL_FLOAT_4_INITIALIZER(0.3f, 0.07f, 0.02f, 0.0f), // groundColor
    NN_UTIL_FLOAT_4_INITIALIZER(-1.0f, -1.0f, -1.0f, 0.0f), // dirLightDir
    NN_UTIL_FLOAT_4_INITIALIZER(0.8f, 0.8f, 0.8f, 0.0f) // dirLightColor
};

// ブレンドステート初期化
void InitializeBlendState(nn::gfx::Device* pDevice, nn::gfx::BlendState& blendState)
{
    nn::gfx::BlendStateInfo blendStateInfo;
    blendStateInfo.SetDefault();
    nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
    {
        blendTargetStateInfo.SetDefault();
    };
    blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);
    void* pMemory = g3ddemo::AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
    NN_ASSERT_NOT_NULL(pMemory);
    blendState.SetMemory(pMemory, size);
    blendState.Initialize(pDevice, blendStateInfo);
}

// ブレンドステート破棄
void FinalizeBlendState(nn::gfx::Device* pDevice, nn::gfx::BlendState& blendState)
{
    void* pMemory = blendState.GetMemory();
    blendState.Finalize(pDevice);
    if (pMemory)
    {
        g3ddemo::FreeMemory(pMemory);
    }
}

// デプスステンシルステート初期化
void InitializeDepthStencilState(nn::gfx::Device* pDevice, nn::gfx::DepthStencilState& depthStencilState)
{
    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthTestEnabled(true);
    depthStencilStateInfo.SetDepthWriteEnabled(true);
    depthStencilState.Initialize(pDevice, depthStencilStateInfo);
}

// デプスステンシルステート破棄
void FinalizeDepthStencilState(nn::gfx::Device* pDevice, nn::gfx::DepthStencilState& depthStencilState)
{
    depthStencilState.Finalize(pDevice);
}

// ラスタライザステート初期化
void InitializeRasterizerState(nn::gfx::Device* pDevice, nn::gfx::RasterizerState& rasterizerState)
{
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    rasterizerStateInfo.SetDefault();
    rasterizerStateInfo.SetScissorEnabled(true);
    rasterizerState.Initialize(pDevice, rasterizerStateInfo);
}

// ラスタライザステート破棄
void FinalizeRasterizerState(nn::gfx::Device* pDevice, nn::gfx::RasterizerState& rasterizerState)
{
    rasterizerState.Finalize(pDevice);
}

// 頂点ステート初期化
void InitializeVertexState(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, nn::g3d::ResShaderArchive* pResShaderArchive)
{
    int shapeCount = pResModel->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ResShape* pResShape = pResModel->GetShape(shapeIndex);
        const nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(pResShape->GetMaterialIndex());
        const nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
        const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(pResShaderAssign->GetShadingModelName());
        const nn::g3d::ResVertex* pResVertex = pResShape->GetVertex();

        int shaderVertexAttrCount = pResShadingModel->GetAttrCount();
        size_t attrInfoSize = sizeof(nn::gfx::VertexAttributeStateInfo) * shaderVertexAttrCount;
        nn::gfx::VertexAttributeStateInfo* pVertexAttributeStateInfo;
        pVertexAttributeStateInfo = g3ddemo::AllocateMemory <nn::gfx::VertexAttributeStateInfo>(attrInfoSize, 8);

        int vertexBufferCount = pResVertex->GetVertexBufferCount();
        size_t bufferInfoSize = sizeof(nn::gfx::VertexBufferStateInfo) * vertexBufferCount;
        nn::gfx::VertexBufferStateInfo* pVertexBufferStateInfo;
        pVertexBufferStateInfo = g3ddemo::AllocateMemory<nn::gfx::VertexBufferStateInfo>(bufferInfoSize, 8);

        // 頂点ステートを構築
        nn::gfx::VertexStateInfo* pVertexStateInfo = g3ddemo::AllocateMemory<nn::gfx::VertexStateInfo>(sizeof(nn::gfx::VertexStateInfo), 8);
        pVertexStateInfo->SetDefault();
        int validIndex = 0;
        for (int shaderVertexAttrIndex = 0; shaderVertexAttrIndex < shaderVertexAttrCount; ++shaderVertexAttrIndex)
        {
            // シェーダーの頂点属性のidを取得
            const char* id = pResShadingModel->GetAttrName(shaderVertexAttrIndex);
            // 頂点属性のidでモデルにおける頂点属性名を取得
            const char* name = pResShaderAssign->FindAttrAssign(id);
            if (name == nullptr)
            {
                // ツールで頂点属性が関連付けられていない
                continue;
            }

            // モデルの頂点属性情報を取得
            const nn::g3d::ResVertexAttr* pResVertexAttr = pResVertex->FindVertexAttr(name);
            if (pResVertexAttr == nullptr)
            {
                continue; // 同一マテリアルの異なるシェイプには存在しない場合があります。
            }

            // シェーダーの頂点属性情報を取得
            const nn::g3d::ResAttrVar* pResAttrVar = pResShadingModel->GetAttr(shaderVertexAttrIndex);
            // 頂点属性情報設定
            pVertexAttributeStateInfo[validIndex].SetDefault();
            pVertexAttributeStateInfo[validIndex].SetBufferIndex(pResVertexAttr->GetBufferIndex());
            pVertexAttributeStateInfo[validIndex].SetFormat(pResVertexAttr->GetFormat());
            pVertexAttributeStateInfo[validIndex].SetOffset(pResVertexAttr->GetOffset());
            pVertexAttributeStateInfo[validIndex].SetShaderSlot(pResAttrVar->GetLocation());
            pVertexAttributeStateInfo[validIndex].SetNamePtr(nullptr);
            ++validIndex;
        }
        pVertexStateInfo->SetVertexAttributeStateInfoArray(pVertexAttributeStateInfo, validIndex);

        // ResVertex中のすべてのバッファを設定する
        for (int index = 0; index < vertexBufferCount; ++index)
        {
            pVertexBufferStateInfo[index].SetDefault();
            pVertexBufferStateInfo[index].SetStride(pResVertex->GetVertexBufferStride(index));
        }
        pVertexStateInfo->SetVertexBufferStateInfoArray(pVertexBufferStateInfo, vertexBufferCount);

        size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize(*pVertexStateInfo);
        void* pVertexStateMemory = g3ddemo::AllocateMemory(memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        nn::gfx::VertexState* pVertexState = g3ddemo::AllocateMemory<nn::gfx::VertexState>(sizeof(nn::gfx::VertexState), 8);
        pVertexState->SetMemory(pVertexStateMemory, memorySize);
        pVertexState->Initialize(pDevice, *pVertexStateInfo, nullptr);

        pResShape->SetUserPtr(pVertexState);

        g3ddemo::FreeMemory(pVertexStateInfo);
        g3ddemo::FreeMemory(pVertexAttributeStateInfo);
        g3ddemo::FreeMemory(pVertexBufferStateInfo);
    }
}

// 頂点ステート破棄
void FinalizeVertexState(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel)
{
    int shapeCount = pResModel->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ResShape* pResShape = pResModel->GetShape(shapeIndex);
        nn::gfx::VertexState* pVertexState = pResShape->GetUserPtr<nn::gfx::VertexState>();
        void* pBuffer = pVertexState->GetMemory();
        pVertexState->Finalize(pDevice);
        if (pBuffer)
        {
            g3ddemo::FreeMemory(pBuffer);
        }
        g3ddemo::FreeMemory(pVertexState);
    }
}

// 頂点設定
void LoadVertex(nn::gfx::CommandBuffer* pCommandBuffer, const nn::g3d::ShapeObj* pShapeObj)
{
    // 頂点ステートを設定
    const nn::gfx::VertexState* pVertexState = pShapeObj->GetResource()->GetUserPtr<const nn::gfx::VertexState>();
    pCommandBuffer->SetVertexState(pVertexState);

    // 頂点バッファを設定
    const nn::g3d::ResVertex* pResVertex = pShapeObj->GetResVertex();
    int vertexBufferCount = pResVertex->GetVertexBufferCount();
    // ResVertex中のすべてのバッファを設定する
    for (int bufferIndex = 0; bufferIndex < vertexBufferCount; ++bufferIndex)
    {
        nn::gfx::GpuAddress gpuAddress;
        pResVertex->GetVertexBuffer(bufferIndex)->GetGpuAddress(&gpuAddress);
        pCommandBuffer->SetVertexBuffer(bufferIndex,
                                        gpuAddress,
                                        pResVertex->GetVertexBufferStride(bufferIndex),
                                        pResVertex->GetVertexBufferInfo(bufferIndex)->GetSize());
    }
}

// 各シェーダーステージにサンプラーとテクスチャを設定
void LoadTextureSampler(nn::gfx::CommandBuffer* pCommandBuffer,
                        const nn::gfx::DescriptorSlot& textureDescriptor,
                        const nn::gfx::DescriptorSlot& samplerDescriptor,
                        const nn::g3d::ResShaderProgram* pShaderProgram,
                        int samplerIndex)
{
    NN_ASSERT_NOT_NULL(pShaderProgram);

    int locationVS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationVS, nn::gfx::ShaderStage_Vertex, textureDescriptor, samplerDescriptor);
    }
    int locationGS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationGS, nn::gfx::ShaderStage_Geometry, textureDescriptor, samplerDescriptor);
    }
    int locationFS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationFS, nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor);
    }
    int locationCS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationCS, nn::gfx::ShaderStage_Compute, textureDescriptor, samplerDescriptor);
    }
}

// テクスチャ、サンプラーのロード
void LoadTextureSampler(nn::gfx::CommandBuffer*          pCommandBuffer,
                        const nn::g3d::ResShaderProgram* pResShaderProgram,
                        const nn::g3d::MaterialObj*      pMaterialObj,
                        const nn::g3d::ShadingModelObj*  pShadingModelObj)
{
    const nn::g3d::ResShadingModel* pResShadingModel = pShadingModelObj->GetResource();
    int shaderSamplerCount = pResShadingModel->GetSamplerCount();
    const nn::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();
    const nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
    for (int shaderSamplerIndex = 0; shaderSamplerIndex < shaderSamplerCount; ++shaderSamplerIndex)
    {
        // 都度名前引きを行わず、調べたサンプラーインデックスを記録しておき、それを使用するほうが高速です
        // ここでは分かりやすさのため、都度名前引きをしています
        const char* id = pResShadingModel->GetSamplerName(shaderSamplerIndex);
        // シェーダーのサンプラーidから対応するモデルのサンプラー名を見つける
        const char* name = pResShaderAssign->FindSamplerAssign(id);
        if (name)
        {
            int modelSamplerIndex = pResMaterial->FindSamplerIndex(name);
            nn::g3d::SamplerRef samplerRef = pMaterialObj->GetSampler(modelSamplerIndex);
            nn::g3d::TextureRef textureRef = pMaterialObj->GetTexture(modelSamplerIndex);
            LoadTextureSampler(pCommandBuffer, textureRef.GetDescriptorSlot(), samplerRef.GetDescriptorSlot(), pResShaderProgram, shaderSamplerIndex);
        }
    }
}

// 各シェーダーステージにユニフォームブロックを設定
void LoadUniformBlock(nn::gfx::CommandBuffer* pCommandBuffer,
                      const nn::gfx::Buffer* pBuffer,
                      const nn::g3d::ResShaderProgram* pShaderProgram,
                      int blockIndex,
                      size_t size)
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pShaderProgram);

    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

// ユニフォームブロックのロード
void LoadUniformBlock(nn::gfx::CommandBuffer*          pCommandBuffer,
                      const nn::g3d::ResShaderProgram* pResShaderProgram,
                      const nn::g3d::ShadingModelObj*  pShadingModelObj,
                      const nn::g3d::SkeletonObj*      pSkeletonObj,
                      const nn::g3d::ShapeObj*         pShapeObj,
                      const nn::g3d::MaterialObj*      pMaterialObj)
{
    const nn::g3d::ResShadingModel* pResShadingModel = pShadingModelObj->GetResource();

    // オプション
    int optionUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Option);
    const nn::gfx::Buffer* pOptionUniformBlock = pShadingModelObj->GetOptionBlock();
    if (optionUniformBlockIndex >= 0 && pShadingModelObj->IsBlockBufferValid())
    {
        size_t size = pShadingModelObj->GetOptionBlockSize();
        LoadUniformBlock(pCommandBuffer, pOptionUniformBlock, pResShaderProgram, optionUniformBlockIndex, size);
    }

    // マテリアル
    int materialUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Material);
    const nn::gfx::Buffer* pMaterialUniformBlock = pMaterialObj->GetMaterialBlock(0);
    if (materialUniformBlockIndex >= 0 && pMaterialObj->IsBlockBufferValid())
    {
        size_t size = pMaterialObj->GetMaterialBlockSize();
        LoadUniformBlock(pCommandBuffer, pMaterialUniformBlock, pResShaderProgram, materialUniformBlockIndex, size);
    }

    // スケルトン
    int skeletonUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Skeleton);
    const nn::gfx::Buffer* pSkeletonUniformBlock = pSkeletonObj->GetMtxBlock(0);
    if (skeletonUniformBlockIndex >= 0 && pSkeletonObj->IsBlockBufferValid())
    {
        size_t size = pSkeletonObj->GetMtxBlockSize();
        LoadUniformBlock(pCommandBuffer, pSkeletonUniformBlock, pResShaderProgram, skeletonUniformBlockIndex, size);
    }

    // シェイプ
    int shapeUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Shape);
    const nn::gfx::Buffer* pShapeUniformBlock = pShapeObj->GetShapeBlock(0, 0);
    if (shapeUniformBlockIndex >= 0 && pShapeObj->IsBlockBufferValid())
    {
        size_t size = sizeof(nn::g3d::ShapeBlock);
        LoadUniformBlock(pCommandBuffer, pShapeUniformBlock, pResShaderProgram, shapeUniformBlockIndex, size);
    }

    // カメラ行列
    int viewUniformBlockIndex = pResShadingModel->FindUniformBlockIndex("view");
    if (viewUniformBlockIndex >= 0)
    {
        LoadUniformBlock(pCommandBuffer, &g_ViewBlock, pResShaderProgram, viewUniformBlockIndex, sizeof(ViewBlock));
    }

    // 半球ライティングおよびディレクショナルライト
    int envUniformBlockIndex = pResShadingModel->FindUniformBlockIndex("env");
    if (envUniformBlockIndex >= 0)
    {
        LoadUniformBlock(pCommandBuffer, &g_EnvBlock, pResShaderProgram, envUniformBlockIndex, sizeof(EnvBlock));
    }
}

// 各シェーダーステージにシェーダーストレージブロックを設定
void LoadShaderStorageBlock(nn::gfx::CommandBuffer* pCommandBuffer,
                            const nn::gfx::Buffer* pBuffer,
                            const nn::g3d::ResShaderProgram* pShaderProgram,
                            int blockIndex,
                            size_t size)
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pShaderProgram);

    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

// シェーダーストレージブロックのロード
void LoadShaderStorageBlock(nn::gfx::CommandBuffer* pCommandBuffer,
                            const nn::g3d::ResShaderProgram* pResShaderProgram,
                            const nn::g3d::ShadingModelObj* pShadingModelObj,
                            const nn::g3d::SkeletonObj* pSkeletonObj)
{
    const nn::g3d::ResShadingModel* pResShadingModel = pShadingModelObj->GetResource();

    // スケルトンのみシェーダーストレージを使用することが可能
    int skeletonShaderStorageBlockIndex = pResShadingModel->GetSystemShaderStorageBlockIndex(nn::g3d::ResShaderStorageBlockVar::Type_Skeleton);
    const nn::gfx::Buffer* pSkeletonShaderStorageBlock = pSkeletonObj->GetMtxBlock(0);
    if (skeletonShaderStorageBlockIndex >= 0 && pSkeletonObj->IsBlockBufferValid())
    {
        size_t size = pSkeletonObj->GetMtxBlockSize();
        LoadShaderStorageBlock(pCommandBuffer, pSkeletonShaderStorageBlock, pResShaderProgram, skeletonShaderStorageBlockIndex, size);
    }
}

// モデル描画
void DrawModel(nn::gfx::CommandBuffer* pCommandBuffer, nn::g3d::ModelObj* pModelObj)
{
    // レンダーステートを設定
    pCommandBuffer->SetBlendState(&g_BlendState);
    pCommandBuffer->SetDepthStencilState(&g_DepthStencilState);
    pCommandBuffer->SetRasterizerState(&g_RasterizerState);

    // ここではソートやビューボリュームによるカリングなどは行っていません
    // アプリケーションの仕様にとって最適な方法で描画処理を行うことを推奨します
    int shapeCount = pModelObj->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        const nn::g3d::SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();
        const nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        int materialIndex = pShapeObj->GetMaterialIndex();
        const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);

        // シェーダーのロード
        const nn::g3d::ShaderSelector* pShaderSelector = pShapeObj->GetUserPtr<nn::g3d::ShaderSelector>();
        const nn::g3d::ShadingModelObj* pShadingModelObj = pShaderSelector->GetShadingModel();
        const nn::g3d::ResShaderProgram* pResShaderProgram = pShaderSelector->GetProgram();
        if (pResShaderProgram == nullptr)
        {
            pResShaderProgram = pShaderSelector->GetDefaultProgram();
        }
        NN_ASSERT_NOT_NULL(pResShaderProgram);

        pResShaderProgram->Load(pCommandBuffer);

        // 頂点設定
        LoadVertex(pCommandBuffer, pShapeObj);

        // テクスチャ、サンプラーのロード
        LoadTextureSampler(pCommandBuffer, pResShaderProgram, pMaterialObj, pShadingModelObj);

        // ユニフォームブロックをロード
        LoadUniformBlock(pCommandBuffer, pResShaderProgram, pShadingModelObj, pSkeletonObj, pShapeObj, pMaterialObj);
        // シェーダーストレージブロックをロード
        LoadShaderStorageBlock(pCommandBuffer, pResShaderProgram, pShadingModelObj, pSkeletonObj);

        // 描画コール
        pShapeObj->GetResMesh()->Draw(pCommandBuffer, 1);
    }
}

// テクスチャ関連付けコールバック
// テクスチャリソースからnameのテクスチャを探し、そのテクスチャのテクスチャビューとディスクリプタスロットを返します
nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData)
{
    nn::g3d::TextureRef textureRef;
    const nn::gfx::ResTextureFile* pTextureFile = reinterpret_cast<const nn::gfx::ResTextureFile*>(pUserData);
    const nn::util::ResDic* pDic = pTextureFile->GetTextureDic();
    int index = pDic->FindIndex(name);
    if (index == nn::util::ResDic::Npos)
    {
        return textureRef;
    }
    const nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(index);
    nn::gfx::DescriptorSlot descriptorSlot;
    pResTexture->GetUserDescriptorSlot(&descriptorSlot);
    textureRef.SetTextureView(pResTexture->GetTextureView());
    textureRef.SetDescriptorSlot(descriptorSlot);
    return textureRef;
}

// モデルリソース初期化
nn::g3d::ResFile* InitializeModelFile(nn::gfx::Device* pDevice, const char* path)
{
    nn::g3d::ResFile* pResFile = NULL;
    size_t alignment = g3ddemo::GetFileAlignment(path);
    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPool))
    {
        // CpuCached_GpuCached のメモリープール領域を確保
        nns::gfx::GraphicsFramework* pGfxFramework = nn::g3d::demo::GetGfxFramework();
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);
        size_t modelMemoryPoolSize = g3ddemo::GetFileSize(path) + nn::g3d::demo::DEFAULT_ALIGNMENT;
        g_ModelMemoryPoolOffset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, modelMemoryPoolSize, g3ddemo::GetFileAlignment(path));

        // ResFile に変換し、アプリのメモリープール上にリソースをロード
        void* pModelFile = g3ddemo::LoadFile(path, pMemoryPool, g_ModelMemoryPoolOffset, modelMemoryPoolSize);
        NN_ASSERT(nn::g3d::ResFile::IsValid(pModelFile));
        pResFile = nn::g3d::ResFile::ResCast(pModelFile);
        pResFile->Setup(pDevice, pMemoryPool, g_ModelMemoryPoolOffset, modelMemoryPoolSize);
    }
    else
    {
        // ファイルをロード
        void* pModelFile = g3ddemo::LoadFile(path, nullptr, alignment);
        NN_ASSERT(nn::g3d::ResFile::IsValid(pModelFile));

        // ResFile に変換し、モデルリソース内のメモリープールを使用して初期化
        pResFile = nn::g3d::ResFile::ResCast(pModelFile);
        pResFile->Setup(pDevice);
    }

    // テクスチャファイルを取り出し
    nn::g3d::ResExternalFile* pExternalFile = pResFile->GetExternalFile(0);
    nn::gfx::ResTextureFile* pTextureFile = nn::gfx::ResTextureFile::ResCast(pExternalFile->GetData());

    // テクスチャの初期化
    pTextureFile->Initialize(pDevice);
    int textureCount = pTextureFile->GetTextureDic()->GetCount();
    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(textureIndex);
        pResTexture->Initialize(pDevice);
    }

    // テクスチャをディスクリプタプールに登録
    g3ddemo::RegisterTextureFileToDescriptorPool(pTextureFile);
    // テクスチャをモデルにバインド
    pResFile->BindTexture(TextureBindCallback, pTextureFile);

    // サンプラーをディスクリプタプールに登録
    for (int modelIndex = 0; modelIndex < pResFile->GetModelCount(); ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        for (int materialIndex = 0; materialIndex < pResModel->GetMaterialCount(); ++materialIndex)
        {
            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
            for (int samplerIndex = 0; samplerIndex < pResMaterial->GetSamplerCount(); ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pResMaterial->GetSampler(samplerIndex);
                nn::gfx::DescriptorSlot descriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);
                pResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
            }
        }
    }

    return pResFile;
}

// モデルリソース破棄
void FinalizeModelFile(nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile)
{
    // テクスチャファイルを取り出し
    nn::g3d::ResExternalFile* pExternalFile = pResFile->GetExternalFile(0);
    nn::gfx::ResTextureFile* pTextureFile = nn::gfx::ResTextureFile::ResCast(pExternalFile->GetData());

    // テクスチャの登録をディスクリプタプールから取り消し
    g3ddemo::UnregisterTextureFileFromDescriptorPool(pTextureFile);

    // テクスチャを破棄
    int textureCount = pTextureFile->GetTextureDic()->GetCount();
    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(textureIndex);
        pResTexture->Finalize(pDevice);
    }
    pTextureFile->Finalize(pDevice);

    // サンプラーの登録をディスクリプタプールから取り消し
    for (int modelIndex = 0; modelIndex < pResFile->GetModelCount(); ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        for (int materialIndex = 0; materialIndex < pResModel->GetMaterialCount(); ++materialIndex)
        {
            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
            for (int samplerIndex = 0; samplerIndex < pResMaterial->GetSamplerCount(); ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pResMaterial->GetSampler(samplerIndex);
                g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
            }
        }
    }

    pResFile->Cleanup(pDevice);

    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPool))
    {
        // アプリ管理下のメモリープール上にリソースをロードした場合は、メモリープールを破棄する。
        g3ddemo::GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_ModelMemoryPoolOffset);
    }
    else
    {
        g3ddemo::FreeMemory(pResFile);
    }
}

// シェーダーファイル初期化
nn::g3d::ResShaderFile* InitializeShaderFile(nn::gfx::Device* pDevice, const char* path)
{
    nn::g3d::ResShaderFile* pResShaderFile = NULL;
    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPool))
    {
        // CpuCached_GpuCached_ShaderCode のメモリープール領域を確保
        nns::gfx::GraphicsFramework* pGfxFramework = nn::g3d::demo::GetGfxFramework();
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Shader);
        size_t shaderMemoryPoolSize = g3ddemo::GetFileSize(path) + nn::g3d::demo::DEFAULT_ALIGNMENT;
        size_t alignment = g3ddemo::GetFileAlignment(path);
        g_ShaderMemoryPoolOffset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, shaderMemoryPoolSize, alignment);

        // ResShaderFileに変換し、アプリ側のメモリープール上にリソースをロード
        void* pShaderFile = g3ddemo::LoadFile(path, pMemoryPool, g_ShaderMemoryPoolOffset, shaderMemoryPoolSize);
        NN_ASSERT(nn::g3d::ResShaderFile::IsValid(pShaderFile));
        pResShaderFile = nn::g3d::ResShaderFile::ResCast(pShaderFile);

        // アプリ側のメモリープールを使用してシェーダーファイルを初期化
        pResShaderFile->Setup(pDevice, pMemoryPool, g_ShaderMemoryPoolOffset, shaderMemoryPoolSize);
    }
    else
    {
        // シェーダーファイルをロード
        size_t alignment = g3ddemo::GetFileAlignment(path);
        void* pShaderFile = g3ddemo::LoadFile(path, nullptr, alignment);

        NN_ASSERT(nn::g3d::ResShaderFile::IsValid(pShaderFile));

        // ResShaderFileに変換し、シェーダーリソース内のメモリープールを使用して初期化
        pResShaderFile = nn::g3d::ResShaderFile::ResCast(pShaderFile);
        pResShaderFile->Setup(pDevice);
    }
    return pResShaderFile;
}

// シェーダーファイル破棄
void FinalizeShaderFile(nn::gfx::Device* pDevice, nn::g3d::ResShaderFile* pResShaderFile)
{
    pResShaderFile->GetResShaderArchive()->Cleanup(pDevice);

    if (NN_STATIC_CONDITION(g_IsUseExternalMemoryPool))
    {
        // アプリ管理下のメモリープール上にシェーダーをロードした場合は、メモリープールを破棄する。
        g3ddemo::GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, g_ShaderMemoryPoolOffset);
    }
    else
    {
        g3ddemo::FreeMemory(pResShaderFile);
    }
}

// マテリアルのシェーダーパラメータ初期化
void InitializeShaderParam(nn::g3d::ResModel* pResModel, nn::g3d::ResShaderArchive* pResShaderArchive)
{
    int materialCount = pResModel->GetMaterialCount();
    for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
    {
        nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
        const nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
        const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(pResShaderAssign->GetShadingModelName());
        nn::g3d::ShaderUtility::BindShaderParam(pResMaterial, pResShadingModel);
    }
}

// モデルインスタンス初期化
void InitializeModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj, nn::g3d::ResModel* pResModel)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    nn::g3d::ModelObj::Builder builder(pResModel);
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    void* pBuffer = g3ddemo::AllocateMemory(bufferSize, nn::g3d::ModelObj::Alignment_Buffer);
    NN_ASSERT_NOT_NULL(pBuffer);
    bool success = builder.Build(pModelObj, pBuffer, bufferSize);
    NN_ASSERT(success);
    NN_UNUSED(success);

    // ユニフォームブロックのバッファは所定のアライメントが必要
    size_t blockBufferSize = pModelObj->CalculateBlockBufferSize(pDevice);
    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    size_t alignment = pModelObj->GetBlockBufferAlignment(pDevice);
    ptrdiff_t blockBufferOffset = pGfxFramework->AllocatePoolMemory(type, blockBufferSize, alignment);
    success = pModelObj->SetupBlockBuffer(pDevice, pGfxFramework->GetMemoryPool(type), blockBufferOffset, blockBufferSize);
    NN_ASSERT(success);
    NN_UNUSED(success);
}

// モデルインスタンス破棄
void FinalizeModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj)
{
    void* pBuffer = pModelObj->GetBufferPtr();
    ptrdiff_t offset = pModelObj->GetMemoryPoolOffset();
    pModelObj->CleanupBlockBuffer(pDevice);

    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, offset);
    g3ddemo::FreeMemory(pBuffer);
}

// シェーダーセレクター初期化
void InitializeShaderSelector(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj, nn::g3d::ResShaderArchive* pResShaderArchive)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    int shapeCount = pModelObj->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        int materialIndex = pModelObj->GetShape(shapeIndex)->GetMaterialIndex();
        const nn::g3d::ResShaderAssign* pResShaderAssign = pModelObj->GetMaterial(materialIndex)->GetResource()->GetShaderAssign();
        const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(pResShaderAssign->GetShadingModelName());
        {
            // nn::g3d::ShadingModelObjを初期化
            nn::g3d::ShadingModelObj* pShadingModelObj = g3ddemo::AllocateMemory<nn::g3d::ShadingModelObj>(sizeof(nn::g3d::ShadingModelObj), 8);
            {
                nn::g3d::ShadingModelObj::Builder builder(pResShadingModel);
                builder.CalculateMemorySize();
                size_t bufferSize = builder.GetWorkMemorySize();
                void* pBuffer = g3ddemo::AllocateMemory(bufferSize, nn::g3d::ShadingModelObj::Alignment_Buffer);
                bool success = builder.Build(pShadingModelObj, pBuffer, bufferSize);
                NN_ASSERT(success == true);
                NN_UNUSED(success);

                size_t blockBufferSize = pShadingModelObj->CalculateBlockBufferSize(pDevice);
                if (blockBufferSize > 0)
                {
                    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
                    size_t alignment = pShadingModelObj->GetBlockBufferAlignment(pDevice);
                    ptrdiff_t blockBufferOffset = pGfxFramework->AllocatePoolMemory(type, blockBufferSize, alignment);
                    success = pShadingModelObj->SetupBlockBuffer(pDevice, pGfxFramework->GetMemoryPool(type), blockBufferOffset, blockBufferSize);
                    NN_ASSERT(success == true);
                    NN_UNUSED(success);
                }
            }

            // シェーダーキーを初期化
            nn::g3d::ShaderUtility::InitializeShaderKey(pShadingModelObj, pResShaderAssign, true);
            // シェーダープログラムの選択範囲を絞る
            pShadingModelObj->UpdateShaderRange();
            // オプションユニフォームブロックを更新
            pShadingModelObj->CalculateOptionBlock();

            // nng3d::ShaderSelectorを初期化
            nn::g3d::ShaderSelector* pShaderSelector = g3ddemo::AllocateMemory<nn::g3d::ShaderSelector>(sizeof(nn::g3d::ShaderSelector), 8);
            {
                nn::g3d::ShaderSelector::Builder builder(pShadingModelObj);
                builder.CalculateMemorySize();
                size_t bufferSize = builder.GetWorkMemorySize();
                void* pBuffer = g3ddemo::AllocateMemory(bufferSize, nn::g3d::ShaderSelector::Alignment_Buffer);
                bool success = builder.Build(pShaderSelector, pBuffer, bufferSize);
                NN_ASSERT(success == true);
                NN_UNUSED(success);
                // シェーダープログラムを選択
                pShaderSelector->UpdateVariation(pDevice);
            }

            nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
            pShapeObj->SetUserPtr(pShaderSelector);
        }
    }
}

// シェーダーセレクター破棄
void FinalizeShaderSelector(nn::g3d::ModelObj* pModelObj)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    int shapeCount = pModelObj->GetShapeCount();
    for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
    {
        nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        nn::g3d::ShaderSelector* pShaderSelector = pShapeObj->GetUserPtr<nn::g3d::ShaderSelector>();
        nn::g3d::ShadingModelObj* pShadingModelObj = pShaderSelector->GetShadingModel();
        // 割り当てたメモリを解放
        {
            void* pBuffer = pShaderSelector->GetBufferPtr();
            if (pBuffer)
            {
                g3ddemo::FreeMemory(pBuffer);
            }
            g3ddemo::FreeMemory(pShaderSelector);
        }
        // 割り当てたメモリプール、メモリを解放
        {
            if (pShadingModelObj->IsBlockBufferValid())
            {
                ptrdiff_t offset = pShadingModelObj->GetMemoryPoolOffset();
                pShadingModelObj->CleanupBlockBuffer(pGfxFramework->GetDevice());
                pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, offset);
            }
            void* pBuffer = pShadingModelObj->GetBufferPtr();
            if (pBuffer)
            {
                g3ddemo::FreeMemory(pBuffer);
            }
            g3ddemo::FreeMemory(pShadingModelObj);
        }
    }
}

// スケルタルアニメーションインスタンス初期化
void InitializeSkeletalAnim(nn::g3d::ModelObj* pModelObj, nn::g3d::SkeletalAnimObj* pSkeletalAnimObj, nn::g3d::ResSkeletalAnim* pResSkeletalAnim)
{
    nn::g3d::SkeletalAnimObj::Builder builder;
    builder.Reserve(pModelObj->GetResource());
    builder.Reserve(pResSkeletalAnim);
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();
    void* pBuffer = g3ddemo::AllocateMemory(bufferSize, nn::g3d::SkeletalAnimObj::Alignment_Buffer);
    NN_ASSERT_NOT_NULL(pBuffer);
    bool success = builder.Build(pSkeletalAnimObj, pBuffer, bufferSize);
    NN_ASSERT(success);
    NN_UNUSED(success);

    // リソースの設定を行います。再設定可能です
    pSkeletalAnimObj->SetResource(pResSkeletalAnim);
    // モデルへの関連付けを行います
    pSkeletalAnimObj->Bind(pModelObj);
}

// スケルタルアニメーションインスタンス破棄
void FinalizeSkeletalAnim(nn::g3d::SkeletalAnimObj* pSkeletalAnimObj)
{
    void* pMemory = pSkeletalAnimObj->GetBufferPtr();
    g3ddemo::FreeMemory(pMemory);
}

// カメラ行列用ユニフォームブロック初期化
void InitializeViewBlock(nn::gfx::Device* pDevice, nn::gfx::Buffer* pViewBlock)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    nn::gfx::BufferInfo bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize(sizeof(ViewBlock));
    bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, bufferInfo);
    g_ViewBlockMemoryPoolOffset = pGfxFramework->AllocatePoolMemory(type, bufferInfo.GetSize(), alignment);
    pViewBlock->Initialize(pDevice, bufferInfo, pGfxFramework->GetMemoryPool(type), g_ViewBlockMemoryPoolOffset, bufferInfo.GetSize());
}

// カメラ行列用ユニフォームブロック破棄
void FinalizeViewBlock(nn::gfx::Device* pDevice, nn::gfx::Buffer* pViewBlock)
{
    pViewBlock->Finalize(pDevice);

    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_ViewBlockMemoryPoolOffset);
}

// 半球ライティングおよびディレクショナルライト用ユニフォームブロック初期化
void InitializeEnvBlock(nn::gfx::Device* pDevice, nn::gfx::Buffer* pEnvBlock)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    nn::gfx::BufferInfo bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetSize(sizeof(EnvBlock));
    bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, bufferInfo);
    g_EnvBlockMemoryPoolOffset = pGfxFramework->AllocatePoolMemory(type, bufferInfo.GetSize(), alignment);
    pEnvBlock->Initialize(pDevice, bufferInfo, pGfxFramework->GetMemoryPool(type), g_EnvBlockMemoryPoolOffset, bufferInfo.GetSize());

    EnvBlock* pMappedEnvBlock = pEnvBlock->Map<EnvBlock>();
    memcpy(pMappedEnvBlock, &g_EnvBlockValue, sizeof(EnvBlock));
    pEnvBlock->Unmap();
}

// 半球ライティングおよびディレクショナルライト用ユニフォームブロック破棄
void FinalizeEnvBlock(nn::gfx::Device* pDevice, nn::gfx::Buffer* pEnvBlock)
{
    pEnvBlock->Finalize(pDevice);

    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_EnvBlockMemoryPoolOffset);
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

int SimpleMain()
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    g3ddemo::ScreenInfo screenInfo;
    screenInfo.Initialize(pDevice);
    screenInfo.SetWindowPosition(16, 16);
    screenInfo.SetFontSize(20);

    // ブレンドステート初期化
    InitializeBlendState(pDevice, g_BlendState);

    // 深度ステンシルステート初期化
    InitializeDepthStencilState(pDevice, g_DepthStencilState);

    // ラスタライザステート初期化
    InitializeRasterizerState(pDevice, g_RasterizerState);

    // シェーダーファイル初期化
    nn::g3d::ResShaderFile* pResShaderFile = InitializeShaderFile(pDevice, g_ShaderPath);
    nn::g3d::ResShaderArchive* pResShaderArchive = pResShaderFile->GetResShaderArchive();

    // モデルリソース初期化
    nn::g3d::ResFile* pResFile = InitializeModelFile(pDevice, g_ModelPath);

    nn::g3d::ResModel* pResModel = pResFile->FindModel(g_ModelName);

    // 頂点ステート初期化
    InitializeVertexState(pDevice, pResModel, pResShaderArchive);

    // マテリアルのシェーダーパラメータ初期化
    InitializeShaderParam(pResModel, pResShaderArchive);

    // モデルインスタンス作成
    nn::g3d::ModelObj modelObj;
    InitializeModelObj(pDevice, &modelObj, pResModel);

    // シェーダーセレクター作成
    InitializeShaderSelector(pDevice, &modelObj, pResShaderArchive);

    // スケルタルアニメーション作成
    nn::g3d::ResSkeletalAnim* pResSkeletalAnim = pResFile->FindSkeletalAnim(g_SkeletalAnimName);
    nn::g3d::SkeletalAnimObj skeletalAnimObj;
    InitializeSkeletalAnim(&modelObj, &skeletalAnimObj, pResSkeletalAnim);

    // g3dで管理されない描画時に必要となるユニフォームブロック初期化
    InitializeViewBlock(pDevice, &g_ViewBlock);
    InitializeEnvBlock(pDevice, &g_EnvBlock);

    pGfxFramework->ResetFrame();

    // メインループ
    g3ddemo::Pad& pad = g3ddemo::GetPad();
    int counter = 0;
    int bufferIndex = 0;
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // テクスチャ取得と vsync 待ち
        pGfxFramework->AcquireTexture(bufferIndex);
        pGfxFramework->WaitDisplaySync(bufferIndex, nn::TimeSpan::FromSeconds(2));

        // 計算
        {
            // 投影行列計算
            nn::util::Matrix4x4fType projMtx;
            MatrixPerspectiveFieldOfViewRightHanded(&projMtx, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 1.0f, 1000.0f);

            // カメラ行列計算
            nn::util::Vector3fType pos = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 82.5f, 247.5f);
            nn::util::Vector3fType target = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 82.5f, 0.0f);
            nn::util::Vector3fType up = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 1.0f, 0.0f);

            nn::util::Matrix4x3fType cameraMtx;
            MatrixLookAtRightHanded(&cameraMtx, pos, target, up);

            // スケルタルアニメーション更新
            if (skeletalAnimObj.GetResource())
            {
                skeletalAnimObj.Calculate();
                skeletalAnimObj.ApplyTo(&modelObj);
                skeletalAnimObj.GetFrameCtrl().UpdateFrame();
            }

            // ベース行列計算
            nn::util::Matrix4x3fType baseMtx;
            nn::util::Vector3fType rotate = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, counter++ * 3.14f * 2 / 256, 0.0f);
            MatrixSetRotateXyz(&baseMtx, rotate);
            nn::util::Vector3fType translate = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);
            MatrixSetAxisW(&baseMtx, translate);

            modelObj.CalculateWorld(baseMtx);

            // GPU 待ちが必要な計算
            modelObj.CalculateSkeleton(0);
            modelObj.CalculateMaterial(0);
            modelObj.CalculateShape(0);
            modelObj.CalculateView(0, cameraMtx, 0);

            // ビュー用ユニフォームブロック更新
            ViewBlock* pMappedViewBlock = g_ViewBlock.Map<ViewBlock>();
            MatrixStore(&pMappedViewBlock->projMtx, projMtx);
            MatrixStore(&pMappedViewBlock->cameraMtx, cameraMtx);
            g_ViewBlock.FlushMappedRange(0, sizeof(ViewBlock));
            g_ViewBlock.Unmap();
        }

        // モデル・テキスト描画 および スキャンバッファへのコピー コマンド生成
        pGfxFramework->BeginFrame(bufferIndex);
        {
            nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

            // カラーターゲットをクリアする
            nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
            pCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

            // 深度ステンシルをクリアする
            nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
            pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

            // レンダーターゲットをセットする
            pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

            // ビューポートシザーステートをセットする
            pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

            // モデル描画コマンド
            DrawModel(pCommandBuffer, &modelObj);

            // テキスト描画コマンド
            screenInfo.StartPrint();
            screenInfo.Print("[Simple]\n");
            screenInfo.EndPrint();
            screenInfo.AdjustWindowSize();
            screenInfo.Draw(pCommandBuffer);
        }
        pGfxFramework->EndFrame(bufferIndex);

        // コマンドの実行と描画
        pGfxFramework->ExecuteCommand(bufferIndex);
        pGfxFramework->QueuePresentTexture(1);

        // GPU の完了待ち
        pGfxFramework->WaitGpuSync(bufferIndex, nn::TimeSpan::FromSeconds(2));

        bufferIndex = 1 - bufferIndex;
    }

    // スケルタルアニメーション破棄
    FinalizeSkeletalAnim(&skeletalAnimObj);

    // シェーダーセレクター破棄
    FinalizeShaderSelector(&modelObj);

    // モデルインスタンス破棄
    FinalizeModelObj(pDevice, &modelObj);

    // 頂点ステート破棄
    FinalizeVertexState(pDevice, pResModel);

    // モデルリソース破棄
    FinalizeModelFile(pDevice, pResFile);

    // g3dで管理されないユニフォームブロック破棄
    FinalizeViewBlock(pDevice, &g_ViewBlock);
    FinalizeEnvBlock(pDevice, &g_EnvBlock);

    // シェーダーファイル破棄
    FinalizeShaderFile(pDevice, pResShaderFile);

    // ラスタライザステート破棄
    FinalizeRasterizerState(pDevice, g_RasterizerState);

    // ブレンドステート破棄
    FinalizeBlendState(pDevice, g_BlendState);

    // デプスステンシルステート破棄
    FinalizeDepthStencilState(pDevice, g_DepthStencilState);

    // スクリーン情報表示破棄
    {
        screenInfo.Cleanup(pDevice);
    }

    return EXIT_SUCCESS;
} // NOLINT
