﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ShapeAnimation.cpp,PageSampleG3dShapeAnimation}
 *
 * @brief
 * シェイプアニメーションを使用したモデル表示のサンプルプログラム
 */

/**
 * @page PageSampleG3dShapeAnimation G3dDemo ShapeAnimation
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\shapeanimation.png
 *
 * @brief
 * サンプルプログラム ShapeAnimation の解説です。シェイプアニメーションを使用したモデル表示をします。
 *
 * @section PageSampleG3dShapeAnimation_SectionBrief 概要
 * シェイプアニメーションを使用したモデル表示を行うサンプルです。
 *
 * @section PageSampleShapeAnimation_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dShapeAnimation_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dShapeAnimation_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左右キー </td><td> シェイプの頂点計算を GPU で行うか CPU で行うか切り替えます。 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleShapeAnimation_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleShapeAnimation_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから ShapeAnimation を選択し、実行してください。
 *
 * @section PageSampleShapeAnimation_SectionDetail 解説
 * このサンプルプログラムは、シェーダーアーカイブ、モデルリソースを読み込み、シェイプアニメーションを適用し、
 * モデルを表示します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャー初期化
 * - サンプラーをディスクリプタプールに登録
 * - nns::g3d を使用したモデルのセットアップ
 * - コンピュートシェーダーの初期化
 * - ループ開始
 * - 前フレームで作成したコマンドリスト実行
 * - 各行列計算
 * - コマンドリスト生成
 * - シェイプアニメーションで頂点ブレンド用のウェイト値を更新
 * - GPU または CPU による頂点ブレンドを実行してモデルの形状を更新
 * - ループ開始に戻る
 *
 */

#include <nn/g3d.h>
#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"

namespace g3ddemo = nn::g3d::demo;

namespace {

#define RESOURCE_PATH "Resource:/"
const char* g_DemoBfshaPath = RESOURCE_PATH "shader/demo.bfsha";
const char* g_ComputeBfshaPath = RESOURCE_PATH "shader/shape_anim_compute.bfsha";
const char* g_EnvBfresPath = RESOURCE_PATH "env.bfres";
const char* g_ModelBfresPath = RESOURCE_PATH "shape_anim.bfres";

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraUp;
nn::util::Vector3fType g_CameraTarget;

g3ddemo::ResourceHolder g_Holder;
g3ddemo::ScreenInfo g_ScreenInfo;
nns::g3d::RenderView g_RenderView;
int g_ShapeAnimId = nns::g3d::ModelAnimObj::InvalidAnimId;

// 計算モード
enum CalcMode
{
    CalcMode_GPU = 0,
    CalcMode_CPU,
};
int g_CalcMode;

// ブレンドする頂点属性
enum BlendAttribute
{
    BlendAttribute_Position,
    BlendAttribute_Normal,
    BlendAttribute_CountMax
};

// キーシェイプ数
const int g_KeyShapeCount = 5;

// 頂点ブレンドを行うシェイプのインデクス
const int g_BlendShapeIndex = 0;

// バッファーリング数
const int g_BufferingCount = 2;

// GPU バッファー
nn::gfx::Buffer g_ExportBuffer[BlendAttribute_CountMax];
nn::gfx::Buffer g_WeightBuffer;
nn::gfx::Buffer g_KeyShapeBuffers[BlendAttribute_CountMax];
nn::gfx::Buffer g_ShapeAnimParamBuffer;
struct shapeAnimParamBuffer
{
    int vertexCount;
    int keyShapeCount;
};

// メモリープールオフセット
ptrdiff_t g_ExportMemoryPoolOffset[BlendAttribute_CountMax];
ptrdiff_t g_WeightMemoryPoolOffset;
ptrdiff_t g_KeyShapeMemoryPoolOffset[BlendAttribute_CountMax];
ptrdiff_t g_ShapeAnimParamMemoryPoolOffset;

// 頂点ブレンドに使用するシェイプの情報
uint32_t g_ShapeVertexCount;
size_t g_KeyShapeBuffersSize;
size_t g_BlendWeightBufferSize;
size_t g_ExportBufferSize;
size_t g_ShapeAnimParamBufferSize;
float g_ShapeBlendWeight[g_KeyShapeCount];

// コンピュートシェーダーで使用する GPU バッファーの初期化
void InitializeGpuBuffer(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj)
{
    nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(g_BlendShapeIndex);

    // ブレンドするシェイプの頂点数を取得
    const nn::g3d::ResVertex* pResVertex = pShapeObj->GetResource()->GetVertex();
    g_ShapeVertexCount = pShapeObj->GetResVertex()->GetCount();

    // モデルのキーシェイプ数がサンプル想定数であることをチェック
    NN_ASSERT(g_KeyShapeCount == pModelObj->GetResource()->GetShape(g_BlendShapeIndex)->GetKeyShapeCount());

    // 各バッファーのサイズを計算
    g_ExportBufferSize = sizeof(nn::util::Float4) * g_ShapeVertexCount;
    g_BlendWeightBufferSize =  sizeof(float) * g_KeyShapeCount;
    g_KeyShapeBuffersSize = sizeof(nn::util::Float4) * g_ShapeVertexCount * g_KeyShapeCount;
    g_ShapeAnimParamBufferSize = sizeof(shapeAnimParamBuffer);

    nns::gfx::GraphicsFramework* pFramework = g3ddemo::GetGfxFramework();
    nn::gfx::MemoryPool* pMemoryPool = pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);

    // コンピュートシェーダー用のキーシェイプの頂点属性を保持するバッファーの初期化
    // g_KeyShapeBuffers は 頂点数×sizeof(nn::util::Float4)×キーシェイプ数のサイズを持ち、全てのキーシェイプの頂点属性の値を順番に保持する。
    const char* srcBlendAttributeName[BlendAttribute_CountMax][g_KeyShapeCount] =
    {
        { "_p0", "_p1", "_p2", "_p3", "_p4" },
        { "_n0", "_n1", "_n2", "_n3", "_n4" }
    };
    for (int blendAttributeIndex = 0; blendAttributeIndex < BlendAttribute_CountMax; ++blendAttributeIndex)
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_UnorderedAccessBuffer);
        info.SetSize(g_KeyShapeBuffersSize);

        g_KeyShapeMemoryPoolOffset[blendAttributeIndex] = g3ddemo::GetGfxFramework()->AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            info.GetSize(),
            nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
        g_KeyShapeBuffers[blendAttributeIndex].Initialize(pDevice,
            info, pMemoryPool, g_KeyShapeMemoryPoolOffset[blendAttributeIndex], info.GetSize());

        // 頂点ブレンドに使用する頂点属性を頂点バッファーから取得
        nn::util::Float4* pMapped = g_KeyShapeBuffers[blendAttributeIndex].Map<nn::util::Float4>();
        for (int shapeIndex = 0; shapeIndex < g_KeyShapeCount; ++shapeIndex)
        {
            const nn::g3d::ResVertexAttr* pAttr = pShapeObj->FindResVertexAttr(srcBlendAttributeName[blendAttributeIndex][shapeIndex]);
            const int bufferIndex = pAttr->GetBufferIndex();
            ptrdiff_t stride = pResVertex->GetVertexBufferStride(bufferIndex);
            const nn::gfx::Buffer* pBuffer = pShapeObj->GetVertexBuffer(bufferIndex);
            nn::util::BytePtr bytePtr(pBuffer->Map());
            bytePtr.Advance(pAttr->GetOffset());
            for (uint32_t vertexIndex = 0; vertexIndex < g_ShapeVertexCount; ++vertexIndex)
            {
                nn::util::Float4* pAttrValue = bytePtr.Get<nn::util::Float4>();
                pMapped->x = pAttrValue->x;
                pMapped->y = pAttrValue->y;
                pMapped->z = pAttrValue->z;
                pMapped->w = 0.0f;
                pMapped++;

                bytePtr.Advance(stride);
            }
        }
        g_KeyShapeBuffers[blendAttributeIndex].Unmap();
    }

    // コンピュートシェーダー用の各キーシェイプの weight 値を保持するバッファーの初期化
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_UnorderedAccessBuffer);
        info.SetSize(g_BlendWeightBufferSize);

        g_WeightMemoryPoolOffset = pFramework->AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            info.GetSize(),
            nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
        g_WeightBuffer.Initialize(pDevice, info, pMemoryPool, g_WeightMemoryPoolOffset, info.GetSize());
    }

    // コンピュートシェーダー用の演算結果を格納するバッファーの初期化
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_UnorderedAccessBuffer);
        info.SetSize(g_ExportBufferSize);

        for (int blendAttributeIndex = 0; blendAttributeIndex < BlendAttribute_CountMax; ++blendAttributeIndex)
        {
            g_ExportMemoryPoolOffset[blendAttributeIndex] = g3ddemo::GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                info.GetSize(), nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
            g_ExportBuffer[blendAttributeIndex].Initialize(pDevice, info, pMemoryPool, g_ExportMemoryPoolOffset[blendAttributeIndex], info.GetSize());
        }
    }

    // ユニフォームブロック用バッファー
    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        info.SetSize(g_ShapeAnimParamBufferSize);

        g_ShapeAnimParamMemoryPoolOffset = g3ddemo::GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            info.GetSize(), nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
        g_ShapeAnimParamBuffer.Initialize(pDevice, info, pMemoryPool, g_ShapeAnimParamMemoryPoolOffset, info.GetSize());

        shapeAnimParamBuffer* pMapped = g_ShapeAnimParamBuffer.Map<shapeAnimParamBuffer>();
        pMapped->vertexCount = g_ShapeVertexCount;
        pMapped->keyShapeCount = g_KeyShapeCount;
        g_ShapeAnimParamBuffer.Unmap();
    }
}

// コンピュートシェーダーの初期化
void InitializeComputeShader(nn::gfx::Device* pDevice)
{
    nn::g3d::ResShaderArchive* pResShaderArchive = g_Holder.shaderFiles[1]->GetResShaderArchive();
    NN_ASSERT(pResShaderArchive);
    nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel("shape_anim_compute");
    NN_ASSERT(pResShadingModel);
    nn::g3d::ResShaderProgram* pResShaderProgram = pResShadingModel->GetShaderProgram(0);
    NN_ASSERT(pResShaderProgram);
    pResShaderProgram->Update(pDevice);
}

// コンピュートシェーダーの破棄
void FinalizeGpuBuffer(nn::gfx::Device* pDevice)
{
    nns::gfx::GraphicsFramework* pFramework = g3ddemo::GetGfxFramework();

    // コンピュートシェーダー用のユニフォームブロックのバッファーの破棄
    g_ShapeAnimParamBuffer.Finalize(pDevice);
    pFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_ShapeAnimParamMemoryPoolOffset);

    // コンピュートシェーダー用の演算結果を格納するバッファーの破棄
    for (int blendAttributeIndex = 0; blendAttributeIndex < BlendAttribute_CountMax; ++blendAttributeIndex)
    {
        g_ExportBuffer[blendAttributeIndex].Finalize(pDevice);
        pFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_ExportMemoryPoolOffset[blendAttributeIndex]);
    }

    // コンピュートシェーダー用の各キーシェイプの weight 値を保持するバッファーの破棄
    g_WeightBuffer.Finalize(pDevice);
    pFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_WeightMemoryPoolOffset);

    // コンピュートシェーダー用のキーシェイプの頂点属性を保持するバッファーの破棄
    for (int blendAttributeIndex = 0; blendAttributeIndex < BlendAttribute_CountMax; ++blendAttributeIndex)
    {
        g_KeyShapeBuffers[blendAttributeIndex].Finalize(pDevice);
        pFramework->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_KeyShapeMemoryPoolOffset[blendAttributeIndex]);
    }
}

void MakeComputeShaderCommand(nn::gfx::CommandBuffer* pCommandBuffer)
{
    nn::g3d::ResShaderArchive* pResShaderArchive= g_Holder.shaderFiles[1]->GetResShaderArchive();
    nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel("shape_anim_compute");
    nn::g3d::ResShaderProgram* pComputeShaderProgram = pResShadingModel->GetShaderProgram(0);

    const nn::gfx::Shader* pShader = pComputeShaderProgram->GetShader();
    pCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_Compute);

    nn::gfx::GpuAddress gpuAddress;
    nn::g3d::Stage stage = nn::g3d::Stage_Compute;
    nn::gfx::ShaderStage gfxStageBit = nn::gfx::ShaderStage_Compute;
    int slotIndex = -1;

    // ユニフォームブロックの値をセット
    g_ShapeAnimParamBuffer.GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetUniformBlockLocation(pResShadingModel->FindUniformBlockIndex("shape_anim_param_buf"), stage);
    pCommandBuffer->SetConstantBuffer(slotIndex, gfxStageBit, gpuAddress, g_ShapeAnimParamBufferSize);

    // 各キーシェイプの位置属性をセット
    g_KeyShapeBuffers[BlendAttribute_Position].GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetShaderStorageBlockLocation(pResShadingModel->FindShaderStorageBlockIndex("pos_buf"), stage);
    pCommandBuffer->SetUnorderedAccessBuffer(slotIndex, gfxStageBit, gpuAddress, g_KeyShapeBuffersSize);

    // 各キーシェイプの法線属性をセット
    g_KeyShapeBuffers[BlendAttribute_Normal].GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetShaderStorageBlockLocation(pResShadingModel->FindShaderStorageBlockIndex("norm_buf"), stage);
    pCommandBuffer->SetUnorderedAccessBuffer(slotIndex, gfxStageBit, gpuAddress, g_KeyShapeBuffersSize);

    // 各キーシェイプの weight 値をセット
    g_WeightBuffer.GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetShaderStorageBlockLocation(pResShadingModel->FindShaderStorageBlockIndex("weight_buf"), stage);
    pCommandBuffer->SetUnorderedAccessBuffer(slotIndex, gfxStageBit, gpuAddress, g_BlendWeightBufferSize);

    // 位置属性のブレンド結果が格納されるバッファーをセット
    g_ExportBuffer[BlendAttribute_Position].GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetShaderStorageBlockLocation(pResShadingModel->FindShaderStorageBlockIndex("export_pos_buf"), stage);
    pCommandBuffer->SetUnorderedAccessBuffer(slotIndex, gfxStageBit, gpuAddress, g_ExportBufferSize);

    // 法線属性のブレンド結果が格納されるバッファーをセット
    g_ExportBuffer[BlendAttribute_Normal].GetGpuAddress(&gpuAddress);
    slotIndex = pComputeShaderProgram->GetShaderStorageBlockLocation(pResShadingModel->FindShaderStorageBlockIndex("export_norm_buf"), stage);
    pCommandBuffer->SetUnorderedAccessBuffer(slotIndex, gfxStageBit, gpuAddress, g_ExportBufferSize);

    // 演算コマンドを追加
    int workGroupX;
    int workGroupY;
    int workGroupZ;
    pShader->GetWorkGroupSize(&workGroupX, &workGroupY, &workGroupZ);
    int groupCount = (g_ShapeVertexCount - 1 + workGroupX) / workGroupX;
    pCommandBuffer->Dispatch(groupCount, 1, 1);
    pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_UnorderedAccessBuffer);
}

void CalcMenu(g3ddemo::Pad& pad)
{
    // 計算モードの切り替え
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT) || pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
    {
        g_CalcMode ^= 1;
        g_Holder.modelAnimObjs[0]->GetAnimObj(g_ShapeAnimId)->GetFrameCtrl().SetFrame(0.0f);
    }
}

void MakeMenuCommand(nn::gfx::CommandBuffer* pGfxCommandBuffer)
{
    // テキスト表示 コマンド生成
    g_ScreenInfo.StartPrint();
    g_ScreenInfo.Print("[ShapeAnimation]\n");

    if (g_CalcMode == CalcMode_GPU)
    {
        g_ScreenInfo.Print("Calculation mode:  GPU\n");
    }
    else
    {
        g_ScreenInfo.Print("Calculation mode:  CPU\n");
    }

    g_ScreenInfo.EndPrint();
    g_ScreenInfo.AdjustWindowSize();
    g_ScreenInfo.Draw(pGfxCommandBuffer);
}

// ・現フレームのコンピュートシェーダーの計算結果の参照
// ・頂点バッファーの書き換え
// ・次フレームのコンピュートシェーダーで使用する weight バッファーの更新
void UpdateVertexBuffer()
{
    nn::g3d::ModelObj* pModelObj = g_Holder.renderModelObjs[0]->GetModelObj();
    nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(g_BlendShapeIndex);

    // 頂点ブレンドした結果を書き換える先の頂点属性名
    const char* destBlendAttributeName[BlendAttribute_CountMax] = {"_p0", "_n0"};
    for (int blendAttributeIndex = 0; blendAttributeIndex < BlendAttribute_CountMax; ++blendAttributeIndex)
    {
        const nn::g3d::ResVertexAttr* pAttr = pShapeObj->FindResVertexAttr(destBlendAttributeName[blendAttributeIndex]);
        const int attrBufferIndex = pAttr->GetBufferIndex();
        const nn::g3d::ResVertex* pResVertex = pShapeObj->GetResource()->GetVertex();
        ptrdiff_t stride = pResVertex->GetVertexBufferStride(attrBufferIndex);
        const nn::gfx::Buffer* pBuffer = pShapeObj->GetVertexBuffer(attrBufferIndex);
        uint8_t* pMapped = pBuffer->Map<uint8_t>();

        // GPU による頂点計算を行うモード、既にコンピュートシェーダーによって計算されたブレンド結果を頂点バッファーに書き込む
        if (g_CalcMode == CalcMode_GPU)
        {
            nn::util::Float4* pResult = g_ExportBuffer[blendAttributeIndex].Map<nn::util::Float4>();
            g_ExportBuffer[blendAttributeIndex].InvalidateMappedRange(0, g_ExportBufferSize);
            for (uint32_t vertexIndex = 0; vertexIndex < g_ShapeVertexCount; ++vertexIndex)
            {
                nn::util::Float3* pDst = reinterpret_cast<nn::util::Float3*>(pMapped + pAttr->GetOffset());
                pDst->x = pResult[vertexIndex].x;
                pDst->y = pResult[vertexIndex].y;
                pDst->z = pResult[vertexIndex].z;
                pMapped += stride;
            }
            g_ExportBuffer[blendAttributeIndex].Unmap();
        }
        // CPU による頂点計算を行うモード、各キーシェイプの値とweight値からブレンド計算を行い、頂点バッファーに書き込む
        else if (g_CalcMode == CalcMode_CPU)
        {
            nn::util::Float4* pKeyShapeBuffers = g_KeyShapeBuffers[blendAttributeIndex].Map<nn::util::Float4>();
            for (uint32_t vertexIndex = 0; vertexIndex < g_ShapeVertexCount; ++vertexIndex)
            {
                nn::util::Vector3fType attributeValue[g_KeyShapeCount];
                nn::util::Vector3fType result;
                nn::util::VectorZero(&result);
                for (int shapeIndex = 0; shapeIndex < g_KeyShapeCount; ++shapeIndex)
                {
                    nn::util::VectorSet(&attributeValue[shapeIndex],
                        pKeyShapeBuffers[shapeIndex * g_ShapeVertexCount + vertexIndex].x,
                        pKeyShapeBuffers[shapeIndex * g_ShapeVertexCount + vertexIndex].y,
                        pKeyShapeBuffers[shapeIndex * g_ShapeVertexCount + vertexIndex].z);

                    nn::util::Vector3fType tmpResult;
                    nn::util::VectorMultiply(&tmpResult, attributeValue[shapeIndex], g_ShapeBlendWeight[shapeIndex]);
                    nn::util::VectorAdd(&result, result, tmpResult);
                }
                // 法線は正規化する
                if (blendAttributeIndex == BlendAttribute_Normal)
                {
                    nn::util::VectorNormalize(&result, result);
                }

                nn::util::Float3* pDst = reinterpret_cast<nn::util::Float3*>(pMapped + pAttr->GetOffset());
                nn::util::VectorStore(pDst, result);
                pMapped += stride;
            }
            g_KeyShapeBuffers[blendAttributeIndex].Unmap();
        }
        pBuffer->Unmap();
    }

    // ブレンディングウェイトを更新
    for (int keyShapeIndex = 0, destIndex = 0; keyShapeIndex < g_KeyShapeCount; ++keyShapeIndex)
    {
        if (pShapeObj->IsBlendWeightValid(keyShapeIndex))
        {
            g_ShapeBlendWeight[destIndex++] = pShapeObj->GetBlendWeight(keyShapeIndex);
        }
    }

    // GPU による頂点計算を行う場合は GPU バッファーのウェイト値を更新
    if (g_CalcMode == CalcMode_GPU)
    {
        float* pMapped = g_WeightBuffer.Map<float>();
        for (int i = 0; i < g_KeyShapeCount; i++)
        {
            *pMapped = g_ShapeBlendWeight[i];
            pMapped++;
        }
        g_WeightBuffer.Unmap();
    }
}

void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData)
{
    NN_UNUSED(pGfxFramework);
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // 計算
    {
        // カメラ設定
        g3ddemo::Pad& pad = g3ddemo::GetPad();
        ControlCamera(&g_CameraPosition, &g_CameraUp, &g_CameraTarget, &g_RenderView.GetViewMtx(0), &pad);
        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix,
                g_CameraPosition, g_CameraTarget, g_CameraUp);
            g_RenderView.SetViewMtx(0, matrix);
        }

        {
            // 投影行列
            nn::util::Matrix4x4fType matrix;
            MatrixPerspectiveFieldOfViewRightHanded(&matrix,
                nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
            g_RenderView.SetProjectionMtx(0, matrix);
        }

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);

        // モデルの更新処理
        for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
        {
            pModelAnimObj->CalculateView(0, uniformBufferIndex, g_RenderView.GetViewMtx(0));
            pModelAnimObj->Calculate();
            pModelAnimObj->CalculateUniformBlock(uniformBufferIndex);
        }
    }
}

void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData)
{
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // コマンド生成
    pGfxFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pGfxCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        // カラーターゲットをクリア
        nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
        pGfxCommandBuffer->ClearColor(pColor, 0.3f, 0.3f, 0.3f, 1.0f, nullptr);

        // 深度ステンシルをクリア
        nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
        pGfxCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        // レンダーターゲットをセット
        pGfxCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

        // ビューポートシザーステートをセット
        pGfxCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

        // モデル表示 コマンド生成
        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            pRenderModelObj->Draw(pGfxCommandBuffer, 0, uniformBufferIndex);
        }

        // テキスト表示 コマンド生成
        MakeMenuCommand(pGfxCommandBuffer);

        // コンピュートシェーダーによる座標の演算コマンドを生成
        if (g_CalcMode == CalcMode_GPU)
        {
            NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
            NN_PERF_BEGIN_MEASURE_GPU(pGfxCommandBuffer);

            NN_PERF_SET_COLOR(nn::util::Color4u8::Red());
            NN_PERF_BEGIN_MEASURE();
            MakeComputeShaderCommand(pGfxCommandBuffer);
            NN_PERF_END_MEASURE();

            NN_PERF_END_MEASURE_GPU(pGfxCommandBuffer);
        }
    }
    pGfxFramework->EndFrame(bufferIndex);

    // 残っているコマンドの完了を待つ
    pGfxFramework->QueueFinish();

    // 頂点バッファーの書き換え
    NN_PERF_SET_COLOR(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE();
    UpdateVertexBuffer();
    NN_PERF_END_MEASURE();
}

// スクリーン情報表示を初期化
void InitializeScreenInfo(nn::gfx::Device* pDevice)
{
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);
}

// リソースの初期化
void InitializeResource(nn::gfx::Device* pDevice)
{
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_DemoBfshaPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
        pDemoShaderArchive = pFile->GetResShaderArchive();
    }
    nn::g3d::ResShaderArchive* pComputeShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_ComputeBfshaPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
        pComputeShaderArchive = pFile->GetResShaderArchive();
    }

    // モデルリソースの初期化
    nn::g3d::ResFile* pEnvModelFile;
    {
        pEnvModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_EnvBfresPath);
        pEnvModelFile->Setup(pDevice);
        g_Holder.files.PushBack(pEnvModelFile);
    }
    nn::g3d::ResFile* pShapeModelFile;
    {
        pShapeModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_ModelBfresPath);
        pShapeModelFile->Setup(pDevice);
        g3ddemo::SetupTexture(pDevice, pShapeModelFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pShapeModelFile);
        g_Holder.files.PushBack(pShapeModelFile);
    }

    // モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するRenderModelを作成
    nns::g3d::RenderModel* pDemoRenderModel;
    {
        nn::g3d::ResModel* pResModel = pShapeModelFile->FindModel("Cat");
        pDemoRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
        pDemoRenderModel->CreateDrawPass(pDevice, pDemoShaderArchive);
        g_Holder.renderModels.PushBack(pDemoRenderModel);
    }

    // モデルおよびアニメーションの計算処理を簡略化するModelAnimObjを作成
    nns::g3d::ModelAnimObj* pShapeModelAnimObj;
    {
        nn::g3d::ModelObj::Builder builder(pShapeModelFile->GetModel(0));
        // CPUとGPUで並列実行出来るようにバッファー数に2を指定
        builder.ShapeBufferingCount(g_BufferingCount);
        builder.SkeletonBufferingCount(g_BufferingCount);
        builder.MaterialBufferingCount(g_BufferingCount);
        pShapeModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        g_Holder.modelAnimObjs.PushBack(pShapeModelAnimObj);
        // アニメーション追加
        g_ShapeAnimId = pShapeModelAnimObj->CreateAnim(pShapeModelFile->FindShapeAnim("Cat"));
    }

    // RenderModelとModelObjを組み合わせ、描画を行うRenderModelObjを作成
    nns::g3d::RenderModelObj* pShapeRenderModelObj;
    {
        pShapeRenderModelObj = nns::g3d::CreateRenderModelObj(pShapeModelAnimObj->GetModelObj(), pDemoRenderModel);
        g_Holder.renderModelObjs.PushBack(pShapeRenderModelObj);
        // このデモではシェーダーを動的に切り替えないため、ここでシェーダーを選択
        g3ddemo::WriteSkinningOption(pShapeRenderModelObj);
        pShapeRenderModelObj->UpdateShader(pDevice);
    }

    // 描画時に必要となるg3dの管理外のユニフォームブロックを作成し、RenderModelObjに設定
    // ユニフォームブロックの管理を簡単にするため、別モデルのマテリアルユニフォームブロックとして扱っています
    nn::g3d::ModelObj* pEnvModelObj;
    {
        nn::g3d::ResModel* pResModel = pEnvModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pDemoShaderArchive->FindShadingModel("env"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        // このデモでは毎フレーム更新が必要でないため、ここで1度だけ計算を行う
        pEnvModelObj->CalculateMaterial(0);
        pEnvModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pEnvModelObj);
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);
        buffers[1] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(1);
        pShapeRenderModelObj->BindUniformBlock("env", buffers, g_BufferingCount);
    }

    // ビュー用のユニフォームブロックを設定
    g_RenderView.Initialize(pDevice, 1, g_BufferingCount);
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(0, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(0, 1);
        pShapeRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int ShapeAnimationMain()
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    InitializeScreenInfo(pDevice);

    // 負荷計測機能の初期化
    pGfxFramework->InitializePerf();

#if defined( NN_BUILD_CONFIG_OS_WIN )
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(g3ddemo::GetFixedCoreNumber);
#endif

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    // リソースの初期化
    InitializeResource(pDevice);

    // コンピュートシェーダーの初期化
    InitializeComputeShader(pDevice);

    // コンピュートシェーダーで使用する GPU バッファーの初期化
    InitializeGpuBuffer(pDevice, g_Holder.renderModelObjs[0]->GetModelObj());

    // 座標の計算モード
    g_CalcMode = CalcMode_GPU;

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    VectorSet(&g_CameraPosition, 0.0f, 0.0f, 80.f);
    VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
    VectorSet(&g_CameraTarget, 0.0f, 0.0f, 0.0f);

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        CalcMenu(pad);

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        uniformBufferIndex ^= 1;
    }
    pGfxFramework->QueueFinish();

    // 終了処理
    g3ddemo::DestroyAll(pDevice, &g_Holder);

    // ビュー用ユニフォームブロックの破棄
    g_RenderView.Finalize(pDevice);

    // 負荷計測メーターの破棄
    pGfxFramework->FinalizePerf();

    // スクリーン情報表示の破棄
    g_ScreenInfo.Cleanup(pDevice);

    // コンピュートシェーダーで使用する GPU バッファーの破棄
    FinalizeGpuBuffer(pDevice);

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    return EXIT_SUCCESS;
} // NOLINT
