﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define ENABLE_CUBE_ARRAY
//#define ENABLE_SKELETON_SSBO
#include <common.glsl>
#include <skeleton_block.glsl>
#include <shape_block.glsl>
#include <view_block.glsl>
#include <town_env_block.glsl>
#include <town_context_block.glsl>

#define c_Local ( 0 )
#define c_World ( c_Local + 1 )

#define ENABLE_VERTEX_COLOR        ( 0 )       // @@ id="enable_vertex_color"
#define ENABLE_ALBEDO_TEX          ( 1 )       // @@ id="enable_albedo"
#define ENABLE_BAKE_TEX            ( 0 )       // @@ id="enable_bake_texture"
#define ENABLE_NORMAL_MAP          ( 0 )       // @@ id="enable_normal_map"
#define ENABLE_SPEC_MASK           ( 0 )       // @@ id="enable_specular_mask"
#define ENABLE_SSAO_TEX            ( 0 )       // @@ id="enable_ssao"
#define ENABLE_INDIRECT_TEXTURE    ( 0 )       // @@ id="enable_indirect"
#define ENABLE_COLOR_BUFFER_ALBEDO ( 0 )       // @@ id="enable_color_buffer_albedo"
#define RECEIVE_DYNAMIC_SHADOW     ( 0 )       // @@ id="receive_dynamic_shadow"
#define ENABLE_TRANS_TEX           ( 0 )       // @@ id="enable_trans_texture"
#define ENABLE_WATER_BUFFER_ALBEDO ( 0 )       // @@ id="enable_water_buffer_albedo"
#define ENABLE_REFLECT             ( 0 )       // @@ id="enable_reflect"
#define ENABLE_ALPHA_TEST          ( 0 )       // @@ id="enable_alpha_test"
#define ENABLE_LIGHTING            ( 1 )       // @@ id="enable_lighting"
#define SKINNING                   ( -1)       // @@ id="skinning" default="-1" choice="[-1, 8]" type="dynamic"
#define WATER_WORLD                ( 0 )       // @@ id="water_world" type="dynamic"
#define ICE_WORLD                  ( 0 )       // @@ id="ice_world" type="dynamic"
#define LOD_LEVEL                  ( 0 )       // @@ id="lod_level" type="dynamic"
#define COORDINATE                 ( c_Local ) // @@ id="coord" choice="c_Local, c_World" type="dynamic"
#define NUM_LIGHT                  ( 2 )
#define MIP_LEVEL                  ( 0 )       // @@ id="mip_level" type="dynamic"
#define ENABLE_MIX_TEX             ( 0 ) // @@ id="enable_mix_tex" label="チャンネル合成の有効化" visible="false"

layout(std140) uniform Option    // @@ id="opt" type="option"
{
    int enableVtxColor;          // @@ id="enable_vertex_color"
    int enableAlbedoTex;         // @@ id="enable_albedo"
    int enableBakeTex;           // @@ id="enable_bake_texture"
    int enableNormMap;           // @@ id="enable_normal_map"
    int enableSpecMask;          // @@ id="enable_specular_mask"
    int enableAOTex;             // @@ id="enable_ssao"
    int enableIndirectTex;       // @@ id="enable_indirect"
    int enableColorBufferAlbedo; // @@ id="enable_color_buffer_albedo"
    int receiveDynamicShadow;    // @@ id="receive_dynamic_shadow"
    int enableTransTexture;      // @@ id="enable_trans_texture"
    int enableWaterBufferAlbedo; // @@ id="enable_water_buffer_albedo"
    int enableReflect;           // @@ id="enable_reflect"
    int enableAlphaTest;         // @@ id="enable_alpha_test"
    int enableLighting;          // @@ id="enable_lighting"
    int enableWaterWorld;        // @@ id="water_world"
    int enableIceWorld;          // @@ id="ice_world"
    int enableLodLevel;          // @@ id="lod_level"
    int enableMipLevel;          // @@ id="mip_level"
};

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 4)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 5)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 6)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
layout(location = 7)  in vec4   i_Color;        // @@ id="_c0"   hint="color0"
layout(location = 8)  in vec2   i_UV0;          // @@ id="_u0"   hint="uv0"
layout(location = 9)  in vec2   i_UV1;          // @@ id="_u1"   hint="uv1"
layout(location = 10) in vec2   i_UV2;          // @@ id="_u2"   hint="uv2"
layout(location = 11) in vec2   i_UV3;          // @@ id="_u3"   hint="uv3"
layout(location = 12) in vec4   i_Binormal;     // @@ id="_b0"   hint="binormal0"

/* @@
    attrib_id="_p0"   label="位置座標"                    editable="false"
    attrib_id="_n0"   label="法線"                        editable="false"
    attrib_id="_t0"   label="接線"                        editable="false"
    attrib_id="_i0"   label="ブレンドインデックス1-4"     editable="false"
    attrib_id="_i1"   label="ブレンドインデックス5-8"     editable="false"
    attrib_id="_w0"   label="ブレンドウェイト1-4"         editable="false"
    attrib_id="_w1"   label="ブレンドウェイト5-8"         editable="false"
    attrib_id="_c0"   label="ディフューズカラー"
    attrib_id="_u0"   label="アルベドテクスチャ用UV"
    attrib_id="_u1"   label="ベイクテクスチャ用UV"
    attrib_id="_u2"   label="法線マップ用UV"
    attrib_id="_u3"   label="スペキュラマスク用UV"
    interleave="_p0 _n0 _t0 _i0 _i1 _w0 _w1 _c0 _u0 _u1 _u2 _u3"
*/

#endif // NN_G3D_VERTEX_SHADER

NN_G3D_RASTER
{
    vec3 pos_w;
    vec3 pos_v;
    vec3 nrm_w;
    vec4 tan_w;
    vec4 color;
    vec4 uv0_uv1;
    vec4 projection;
    vec4 bake_uv;
    flat float lodLevel;
} nn_g3d_raster;

#ifdef NN_G3D_FRAGMENT_SHADER

layout(location = 0)    out vec4 o_Color;

#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

uniform sampler2D albedoTex0;    // @@ id="_a0" hint="albedo0"
uniform sampler2D normalTex;     // @@ id="_n0" hint="normal0"
uniform sampler2D specularTex;   // @@ id="_s0" hint="specular0"
uniform sampler2D extensionTex;  // @@ id="_x0" hint="extension0"
uniform sampler2D bakeTex0;      // @@ id="_b0" hint="bake0"
uniform sampler2DArray bakeTex1; // @@ id="_b1" hint="bake1"
uniform sampler2D indirectTex;   // @@ id="_a3" hint="albedo3"
uniform sampler2D transTex;      // @@ id="_t0" hint="transmit0"
uniform sampler2D mixTex;        // @@ id="_m0" hint="mix0"

layout(std140) uniform Material // @@ id="mat" type="material"
{
    vec4 texsrt0[2];            // @@ id="texsrt0" type="texsrt" hint="albedo0"
    vec4 texsrt1[2];            // @@ id="texsrt1" type="texsrt" hint="bake" visible="false"
    vec4 texsrt2[2];            // @@ id="texsrt2" type="texsrt" hint="indirect"
    vec3 diffColor;             // @@ id="diffuse" default="1 1 1" item="color" hint="diffuse"
    float specInt;              // @@ id="spec_int" min="0" max="1" default="0.2"
    float specPower;            // @@ id="spec_power" min="0" default="50"
    vec3 albedoColor;           // @@ id="albedo_color" default="1 1 1" item="color" hint="albedo"
    vec4 bake0_st;              // @@ id="bake0_st" default="1 1 0 0" hint="bake0_st"
    vec4 bake1_st;              // @@ id="bake1_st" default="1 1 0 0" hint="bake1_st"
    vec2 indirectScale;         // @@ id="ind_scale" default="1 1" hint="ind_scale"
    float reflectScale;         // @@ id="reflect_scale" default="0.02" hint="reflect_scale"
    float refractScale;         // @@ id="refract_scale" default="0.5" hint="refract_scale"
    float waterScale;           // @@ id="water_scale" default="0.5" hint="water_scale"
    float fresnelCoeff;         // @@ id="fresnel_coeff" default="0.1" hint="fresnel_coeff"
    float bakeLightScale;       // @@ id="bake_light_scale" default="1"
    float alphaTest;            // @@ id="alpha_test" default="0.0" min="0" max="1"
};

// @@ group="diffuse_group" label="ディフューズ" order="0" condition="$enable_vertex_color == 0"
/* @@ diffuse_group:
    option_id="enable_vertex_color" order="1"  label="頂点カラー"
    uniform_id="diffuse"            order="2"  label="ディフューズカラー"
*/

// @@ group="albedo_group" label="アルベド" order="1"
/* @@ albedo_group:
    option_id="enable_albedo"              order="1"  label="アルベドテクスチャ"
    sampler_id="_a0"                       order="2"  label="アルベドテクスチャ0"
    uniform_id="texsrt0"                   order="3"  label="アルベド用SRT"
    uniform_id="albedo_color"              order="4"  label="アルベドカラー"
    option_id="enable_color_buffer_albedo" order="5"  label="カラーバッファをアルベドとする"
    option_id="enable_water_buffer_albedo" order="6"  label="水面バッファをアルベドとする"
    uniform_id="refract_scale"             order="7"  label="屈折スケール"
    uniform_id="water_scale"               order="8"  label="水面スケール"
    uniform_id="fresnel_coeff"             order="9"  label="フレネル係数"
    option_id="enable_lighting"            order="11" label="ライティングを行う"
*/

// @@ group="specular_group" label="スペキュラ" order="2"
/* @@ specular_group:
    uniform_id="spec_int"            order="1"   label="スペキュラ強度"
    uniform_id="spec_power"          order="2"   label="スペキュラ絞り"
    option_id="enable_specular_mask" order="3"   label="スペキュラマスク"
    sampler_id="_s0"                 order="4"   label="スペキュラマスク"
*/

// @@ group="bake_group" label="ベイク" order="3"
/* @@ bake_group:
    option_id="enable_bake_texture" order="1"   label="ベイクテクスチャ"
    sampler_id="_b0"                order="2"   label="ベイクテクスチャ0"
    uniform_id="bake0_st"           order="3"   label="ベイク0ST (xy:scale zw:translate)"
    sampler_id="_b1"                order="4"   label="ベイクテクスチャ1"
    uniform_id="bake1_st"           order="5"   label="ベイク1ST (xy:scale zw:translate)"
    uniform_id="bake_light_scale"   order="6"   label="ベイクライトスケール"
*/

// @@ group="normal_map_group" label="法線" order="4"
/* @@ normal_map_group:
    option_id="enable_normal_map"   order="1"   label="法線マップの有効化"
    sampler_id="_n0"                order="2"   label="法線マップ"
*/

// @@ group="ao_group" label="アンビエントオクルージョン" order="5"
/* @@ ao_group:
    option_id="enable_ssao" order="1"  label="AOテクスチャ"
    sampler_id="_x0"        order="2"  label="AOテクスチャ"
*/

// @@ group="indirect_group" label="インダイレクト" order="6"
/* @@ indirect_group:
    option_id="enable_indirect" order="1"  label="インダイレクトテクスチャの有効化"
    sampler_id="_a3"            order="2"  label="インダイレクトテクスチャ"
    uniform_id="ind_scale"      order="3"  label="インダイレクトスケール"
    uniform_id="texsrt2"        order="4"   label="インダイレクト用SRT"
*/

// @@ group="trans_group" label="伝搬" order="7"
/* @@ trans_group:
    option_id="enable_trans_texture" order="1"  label="伝搬テクスチャの有効化"
    sampler_id="_t0"                 order="2"  label="伝搬テクスチャ"
*/

// @@ group="reflect_group" label="反射" order="8"
/* @@ reflect_group:
    option_id="enable_reflect" order="1" label="環境マップによる反射"
    uniform_id="reflect_scale" order="2" label="反射スケール"
*/

// @@ group="dynamic_shadow_group" label="影の生成" order="10"
/* @@ dynamic_shadow_group:
    option_id="receive_dynamic_shadow"     order="1" label="影を受け取る"
    renderinfo="dynamic_shadow" type="int" order="2" default="0" choice="[0,1]" optional="false" label="影を書き込む"
*/

// @@ group="ice_group" label="アイス" order="11"
/* @@ ice_group:
    renderinfo="ice" type="int" order="1" default="0" choice="[0,1]" optional="false" label="アイスを受け取る"
*/

// @@ group="blend_group" label="ブレンド" order="12"
/* @@ blend_group:
    renderinfo="blend" type="int" default="0"
    choice="0: 不透明,
            1: 半透明ブレンド color = src * srcA + dest * (1 - srcA) : alpha = srcA,
            2: 半透明加算 color = src * srcA + dest : alpha = srcA,
            3: カスタム1 color = src + dest : alpha = srcA + destA,
            4: カスタム2 color = src : alpha = srcA"
    optional="false"
    order="1"
    label="ブレンドモード"
*/

// @@ group="alpha_test_group" label="アルファテスト" order="13"
/* @@ alpha_test_group:
    option_id="enable_alpha_test" order="0" label="アルファテストを行う"
    uniform_id="alpha_test"       order="1"  label="アルファテスト値"
*/

// @@ group="culling_group" label="カリング" order="14"
/* @@ culling_group:
    renderinfo="culling" type="int" default="0"
    choice="0: 表面を表示,
            1: 裏面を表示,
            2: 両面を表示"
    optional="false"
    order="1"
    label="カリングモード"
 */

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void main()
{
#if COORDINATE == c_Local
    vec4 pos_w = vec4(0, 0, 0, 1);
    vec3 nrm_w = vec3(0);
    vec3 tan_w = vec3(0);

    int skinning = SKINNING != -1 ? SKINNING : vtxSkinCount;
    if (skinning == 0)
    {
        // rigid body
        pos_w.xyz = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
        if(!NN_G3D_TO_BOOL(WATER_WORLD))
        {
        	tan_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Tangent);
        }
    }
    else if (skinning == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        if(!NN_G3D_TO_BOOL(WATER_WORLD))
        {
            tan_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
    else if (skinning <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < skinning; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if(!NN_G3D_TO_BOOL(WATER_WORLD))
            {
            	tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
            }
        }
    }
    else if (skinning <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if(!NN_G3D_TO_BOOL(WATER_WORLD))
            {
                tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
            }
        }
        for (int i = 4; i < skinning; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if(!NN_G3D_TO_BOOL(WATER_WORLD))
            {
                tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
            }
        }
    }
#elif COORDINATE == c_World
    vec4 pos_w = i_Position;
    vec3 nrm_w = i_Normal;
    vec3 tan_w = i_Tangent.xyz;
#endif

    float lodLevel = 0.0;

    if( NN_G3D_TO_BOOL(LOD_LEVEL) )
    {
        const int maxLodLevel = 8;
        int offsets[maxLodLevel] =
        {
            userArea[0].x,
            userArea[0].y,
            userArea[0].z,
            userArea[0].w,
            userArea[1].x,
            userArea[1].y,
            userArea[1].z,
            userArea[1].w,
        };

        for (int idxLodLevel = 1; idxLodLevel < maxLodLevel; ++idxLodLevel)
        {
            if (offsets[idxLodLevel] != -1)
            {
                if (gl_VertexID >= offsets[idxLodLevel])
                {
                    lodLevel += 1.0;
                }
            }
        }
    }

    if (NN_G3D_TO_BOOL(WATER_WORLD))
    {
        const vec3 waterPosition = vec3(0.0, -5.0, 0.0);
        const vec3 waterNormal = vec3(0.0, 1.0, 0.0);
        gl_ClipDistance[0] = dot(waterNormal, pos_w.xyz - waterPosition);
    }

    vec4 pos_v = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);
    vec4 out_pos = NN_G3D_PROJECT(projMtx, pos_v);

    vec2 uv1 = NN_G3D_TRANSFORM_UV(texsrt1, i_UV1);

    gl_Position = out_pos;
    OUT.pos_w = pos_w.xyz;
    OUT.pos_v = pos_v.xyz;
    OUT.nrm_w = normalize(nrm_w);
    if(!NN_G3D_TO_BOOL(WATER_WORLD))
    {
        OUT.tan_w = vec4(tan_w, i_Tangent.w);
    }
    OUT.color = i_Color;
    OUT.uv0_uv1.xy = NN_G3D_TRANSFORM_UV(texsrt0, i_UV0);
    OUT.uv0_uv1.zw = uv1;
    OUT.projection = out_pos;
    OUT.bake_uv.xy = uv1 * bake0_st.xy + bake0_st.zw;
    OUT.bake_uv.zw = uv1 * bake1_st.xy + bake1_st.zw;
    OUT.lodLevel = lodLevel;
}

#endif // NN_G3D_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

void main()
{
    vec3 nrm_w = IN.nrm_w;
    vec3 view_w = normalize(vec3(invCameraView[0].w, invCameraView[1].w, invCameraView[2].w) - IN.pos_w);

    vec2 albedo_uv = IN.uv0_uv1.xy;
    vec2 normal_uv = IN.uv0_uv1.xy;
    vec2 spm_uv = IN.uv0_uv1.xy;
    vec2 bake0_uv = IN.bake_uv.xy;
    vec2 bake1_uv = IN.bake_uv.zw;
    vec2 screen_uv = IN.projection.xy / IN.projection.w * 0.5 + 0.5;
    vec2 color_buffer_uv = screen_uv;
    float lodLevel = IN.lodLevel;

    float shadow_intensity = 0.0;
    float ao_intensity = 0.0;

#if ENABLE_MIX_TEX != 0
    vec4 mix_tex = texture(mixTex, albedo_uv);
#endif

    // indirect
    if (NN_G3D_TO_BOOL(ENABLE_INDIRECT_TEXTURE) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
        vec2 ind_offset;
        if (NN_G3D_TO_BOOL(ENABLE_COLOR_BUFFER_ALBEDO) || NN_G3D_TO_BOOL(ENABLE_WATER_BUFFER_ALBEDO))
        {
            ind_offset = texture(indirectTex, albedo_uv).rg * indirectScale;
        }
        else
        {
            vec2 ind_uv = NN_G3D_TRANSFORM_UV(texsrt2, color_buffer_uv);
            ind_offset = texture(indirectTex, ind_uv).rg * indirectScale;
        }
        color_buffer_uv += ind_offset;
        albedo_uv += ind_offset;
        normal_uv += ind_offset;
        spm_uv += ind_offset;
    }

    // normal
    vec3 normal = nrm_w;
    if (NN_G3D_TO_BOOL(ENABLE_NORMAL_MAP) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
        vec3 nrm_l;
        nrm_l.xy = texture(normalTex, normal_uv).xy;
        float lengthSq = 1 - nrm_l.x * nrm_l.x - nrm_l.y * nrm_l.y;
        lengthSq = clamp(lengthSq, 0, 1);
        nrm_l.z =  sqrt(lengthSq);
        vec3 tan_w = normalize(IN.tan_w.xyz);
        vec3 binrm_w = cross(nrm_w, tan_w) * IN.tan_w.w;
        normal = nrm_l.x * tan_w + nrm_l.y * binrm_w + nrm_l.z * nrm_w;
    }
    float view_dot_normal = dot(view_w, normal);

    // albedo
    vec4 albedo = vec4(0.0, 0.0, 0.0, 1.0);
    if (NN_G3D_TO_BOOL(ENABLE_ALBEDO_TEX))
    {
        albedo = texture(albedoTex0, albedo_uv);
    }
    if(!NN_G3D_TO_BOOL(WATER_WORLD))
    {
        if (NN_G3D_TO_BOOL(ENABLE_COLOR_BUFFER_ALBEDO))
        {
            albedo.rgb += texture(colorTex, color_buffer_uv).rgb * refractScale;
        }
        if (NN_G3D_TO_BOOL(ENABLE_WATER_BUFFER_ALBEDO))
        {
            vec2 water_buffer_uv = vec2(color_buffer_uv.x, 1.0 - color_buffer_uv.y);
            float fresnel = fresnelCoeff + ( 1.0 - fresnelCoeff ) * pow(1.0 - view_dot_normal, 5);
            albedo.rgb += texture(waterTex, water_buffer_uv).rgb * waterScale * fresnel;
        }
    }
    albedo.rgb *= albedoColor;

    // bake
    vec3 bake_color = vec3(0);
    if (NN_G3D_TO_BOOL(ENABLE_BAKE_TEX) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
        vec2 ao_shadow = texture(bakeTex0, bake0_uv).rg;
        ao_intensity += 1.0 - ao_shadow.r;
        shadow_intensity += 1.0 - ao_shadow.g;

        vec4 bake_color0 = texture(bakeTex1, vec3(bake1_uv, 0.0));
        vec4 bake_color1 = texture(bakeTex1, vec3(bake1_uv, 1.0));
        vec4 bake_color2 = texture(bakeTex1, vec3(bake1_uv, 2.0));

        const vec3 basis0 = vec3(0.8165, 0.0, 0.577);
        const vec3 basis1 = vec3(-0.408, 0.707, 0.577);
        const vec3 basis2 = vec3(-0.408, -0.707, 0.577);

        vec3 dp;
        dp.x = clamp(dot(normal, basis0), 0, 1);
        dp.y = clamp(dot(normal, basis1), 0, 1);
        dp.z = clamp(dot(normal, basis2), 0, 1);

        bake_color += bake_color0.rgb * ( bake_color0.a * dp.x );
        bake_color += bake_color1.rgb * ( bake_color1.a * dp.y );
        bake_color += bake_color2.rgb * ( bake_color2.a * dp.z );
    }

    // ao
    if (NN_G3D_TO_BOOL(ENABLE_SSAO_TEX) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
#if ENABLE_MIX_TEX != 0
        float ao_value = mix_tex.g;
#else
        float ao_value = texture(extensionTex, albedo_uv).r;
#endif
        ao_intensity += 1.0 - ao_value;
    }

    // trans
    vec3 trans = vec3(0);
    if (NN_G3D_TO_BOOL(ENABLE_TRANS_TEX) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
        vec4 tmp = texture(transTex, albedo_uv);
        trans = tmp.rgb * tmp.a * refractScale;
    }

    vec3 refl_w = 2 * view_dot_normal * normal - view_w;

    // light
    vec4 lpp = vec4(0);
    if(!NN_G3D_TO_BOOL(WATER_WORLD))
    {
        lpp = texture(lightTex, color_buffer_uv);
    }
    vec3 diffRefl = lpp.rgb;
    float specRefl = 0;

    // hemisphere
    if (NN_G3D_TO_BOOL(ENABLE_LIGHTING))
    {
        vec3 up = vec3(0, 1, 0);
        float rate = dot(normal, up) * 0.5 + 0.5;
        diffRefl += ( rate * skyColor + (1 - rate) * groundColor ) * intensity_amb_sha_ao_bake.r;
    }

    // directional
    float inv_dynamic_shadow = 1.0;
    if (NN_G3D_TO_BOOL(ENABLE_LIGHTING))
    {
        if (NN_G3D_TO_BOOL(RECEIVE_DYNAMIC_SHADOW))
        {
            float camera_depth = -IN.pos_v.z;
            int shadow_slice = int(camera_depth > cascadeDepth.x)
                + int(camera_depth > cascadeDepth.y)
                + int(camera_depth > cascadeDepth.z);
            vec4 shadow_coord = NN_G3D_PROJECT_OFFSET(lightViewProj, 4 * shadow_slice, vec4(IN.pos_w, 1.0));
            shadow_coord.xyz = shadow_coord.xyz / shadow_coord.w;
            shadow_coord.xy = shadow_coord.xy * 0.5 + 0.5;
            if(!NN_G3D_TO_BOOL(WATER_WORLD))
            {
                // z座標が1を超えるとアーティファクトが出るのでクランプする。
                inv_dynamic_shadow = texture(shadowTex, vec4(shadow_coord.xy, float(shadow_slice), clamp(shadow_coord.z, 0.0, 1.0)));
            }
        }
        vec3 lightColor = dirLightColor + bake_color * ( intensity_amb_sha_ao_bake.a * bakeLightScale );
        vec3 lightDir = normalize(-dirLightDir);
        float diff_intensity = diffIntensity;
        if (NN_G3D_TO_BOOL(ENABLE_WATER_BUFFER_ALBEDO))
        {
            diff_intensity = 1.0;
        }
        diffRefl += lightColor * clamp(dot(normal, lightDir), 0, 1) * inv_dynamic_shadow * diff_intensity;
        if(!NN_G3D_TO_BOOL(WATER_WORLD))
        {
            specRefl += pow(clamp(dot(lightDir, refl_w) + lpp.a, 0, 1), specPower);
        }
    }

    // specular
    float specular = specRefl * specInt;
    if (NN_G3D_TO_BOOL(ENABLE_SPEC_MASK) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
#if ENABLE_MIX_TEX != 0
        float specMask = mix_tex.r;
#else
        float specMask = texture(specularTex, albedo_uv).r;
#endif
        specular *= specMask;
    }

    // reflect
    vec3 reflectColor = vec3(0);
    if (NN_G3D_TO_BOOL(ENABLE_REFLECT) && !NN_G3D_TO_BOOL(WATER_WORLD))
    {
        vec4 tmp;
        tmp = textureLod(envTex, vec4(refl_w, 0.0), 0.0);
        reflectColor = tmp.rgb * reflectScale;
    }

    float ice = 0.0;
    if (NN_G3D_TO_BOOL(ICE_WORLD))
    {
        ice = pow(clamp(dot(normal, vec3(0.0, 1.0, 0.0)), 0.0, 1.0), 8.0);
        ice *= 1.0 - shadow_intensity * 0.5;
        ice *= inv_dynamic_shadow * 0.5 + 0.5;
        ice *= 1.0 + ao_intensity * 2.0;
    }

    // alpha test
    if (NN_G3D_TO_BOOL(ENABLE_ALPHA_TEST))
    {
        if (albedo.a < alphaTest)
        {
            discard;
        }
    }

    vec4 diffuse = albedo;
    diffuse.rgb *= 1.0 - ao_intensity * intensity_amb_sha_ao_bake.b;
    diffuse.rgb *= 1.0 - shadow_intensity * intensity_amb_sha_ao_bake.g;
    if (NN_G3D_TO_BOOL(ENABLE_LIGHTING))
    {
        diffuse.rgb *= diffRefl * (NN_G3D_TO_BOOL(ENABLE_VERTEX_COLOR) ? IN.color.rgb * IN.color.a : diffColor);
    }

    vec4 out_color = vec4(0);
    out_color += diffuse;
    out_color.rgb += vec3(specular);
    out_color.rgb += reflectColor.rgb;
    out_color.rgb += trans;
    out_color.rgb += vec3(ice);
    out_color.rgb = clamp((out_color.rgb - contrast) * ( 1.0 / ( 1.0 - contrast * 2.0 ) ), 0, 1);

    if( NN_G3D_TO_BOOL(LOD_LEVEL) )
    {
        if (lodLevel == 2.0)
        {
            out_color.rgb *= vec3(1.0, 0.0, 0.0);
        }
        else if (lodLevel == 1.0)
        {
            out_color.rgb *= vec3(1.0, 1.0, 0.0);
        }
        else if (lodLevel == 0.0)
        {
            out_color.rgb *= vec3(0.0, 1.0, 0.0);
        }
        else
        {
            out_color.rgb *= vec3(1.0, 1.0, 1.0);
        }
    }

    // Miplevel
    if ( NN_G3D_TO_BOOL(MIP_LEVEL) )
    {
        const vec3 colors[] = {
            vec3 (1.0,.0,.0),
            vec3 (.0,1.0,.0),
            vec3 (.0,.0,1.0)
        };

        int   mipCount = textureQueryLevels(albedoTex0);
        float mipLevel = textureQueryLod(albedoTex0,albedo_uv).x;
        mipLevel = clamp(mipLevel / float (mipCount),0,1);
        if (mipLevel < .5)
        {
            out_color.rgb = mix(colors[0],colors[1],mipLevel * 2.0);
        }
        else
        {
            out_color.rgb = mix(colors[1],colors[2],(mipLevel - .5) * 2.0);
        }
    }

    o_Color = out_color;
    //o_Color = diffuse;
    //o_Color.rgb = vec3(specular);
    //o_Color = lpp;
}

#endif // NN_G3D_FRAGMENT_SHADER
