﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <common.glsl>
#include <skeleton_block.glsl>
#include <shape_block.glsl>
#include <view_block.glsl>

#define c_Local             ( 0 )
#define c_World             ( c_Local + 1 )

#define c_None              ( 0 )
#define c_Blend             ( c_None + 1 )
#define c_Mul               ( c_Blend + 1 )
#define c_Add               ( c_Mul + 1 )

#define NUM_LIGHT           ( 2 )

#include <legacy.autogen.glsl>

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 4)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 5)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 6)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
layout(location = 7)  in vec4   i_Color;        // @@ id="_c0"   hint="color0"
layout(location = 8)  in vec2   i_UV0;          // @@ id="_u0"   hint="uv0"
layout(location = 9)  in vec2   i_UV1;          // @@ id="_u1"   hint="uv0"
layout(location = 10) in vec2   i_UV2;          // @@ id="_u2"   hint="uv0"

/* @@
    attrib_id="_p0"   label="位置座標"                    editable="false"
    attrib_id="_n0"   label="法線"                        editable="false"
    attrib_id="_t0"   label="接線"                        editable="false"
    attrib_id="_i0"   label="ブレンドインデックス1-4"     editable="false"
    attrib_id="_i1"   label="ブレンドインデックス5-8"     editable="false"
    attrib_id="_w0"   label="ブレンドウェイト1-4"         editable="false"
    attrib_id="_w1"   label="ブレンドウェイト5-8"         editable="false"
    attrib_id="_c0"   label="ディフューズカラー"
    attrib_id="_u0"   label="アルベドテクスチャ0用UV"
    attrib_id="_u1"   label="アルベドテクスチャ1用UV"
    attrib_id="_u2"   label="法線マップ用UV"
    interleave="_p0 _n0 _t0 _i0 _i1 _w0 _w1 _c0 _u0 _u1 _u2"
    interleave="" binarize="false"
*/

#endif // NN_G3D_VERTEX_SHADER

#ifndef NN_G3D_GEOMETRY_SHADER

NN_G3D_RASTER
{
    vec3 nrm_w;
    vec3 nrm_v;
    vec4 tan_w;
    vec4 tan_v;
    vec4 color;
    vec4 uv0_uv1;
    vec4 uv2_uv3;
    vec3 projXYW;
    vec4 debug;
} nn_g3d_raster;

#endif // !NN_G3D_GEOMETRY_SHADER

#ifdef NN_G3D_FRAGMENT_SHADER

layout(location = 0)    out vec4 o_Color;

#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void main()
{
#if COORDINATE == c_Local
    vec4 pos_w = vec4(0, 0, 0, 1);
    vec3 nrm_w = vec3(0);
    vec3 tan_w = vec3(0);

    int skinning = SKINNING != -1 ? SKINNING : vtxSkinCount;
    if (skinning == 0)
    {
        // rigid body
        pos_w.xyz = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Tangent);
    }
    else if (skinning == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
    }
    else if (skinning <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < skinning; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
    else if (skinning <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
        for (int i = 4; i < skinning; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
#elif COORDINATE == c_World
    vec4 pos_w = i_Position;
    vec3 nrm_w = i_Normal;
    vec3 tan_w = i_Tangent.xyz;
#endif

    vec4 pos_v = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);
    vec3 nrm_v = NN_G3D_TRANSFORM_VEC(cameraMtx, nrm_w);
    vec3 tan_v = NN_G3D_TRANSFORM_VEC(cameraMtx, tan_w);

    gl_Position = NN_G3D_PROJECT(projMtx, pos_v);
    OUT.nrm_w = nrm_w;
    OUT.nrm_v = nrm_v;
    OUT.tan_w = vec4(tan_w, i_Tangent.w);
    OUT.tan_v = vec4(tan_v, i_Tangent.w);
    OUT.color = i_Color;
    OUT.uv0_uv1.xy = NN_G3D_TRANSFORM_UV(texsrt0, i_UV0);
    OUT.uv0_uv1.zw = NN_G3D_TRANSFORM_UV(texsrt1, i_UV1);
    OUT.uv2_uv3.xy = NN_G3D_TRANSFORM_UV(texsrt2, i_UV2);
    OUT.projXYW = NN_G3D_TRANSFORM_POS(texsrtProj, pos_w);
}

#endif // NN_G3D_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

void main()
{
    vec3 nrm_w = normalize(IN.nrm_w);

    vec3 normal = nrm_w;
    if (NN_G3D_TO_BOOL(USE_NORMAL_MAP))
    {
        vec3 nrm_l;
        nrm_l.xy = texture(normalTex, IN.uv2_uv3.xy).xy;
        float lengthSq = 1 - nrm_l.x * nrm_l.x - nrm_l.y * nrm_l.y;
        lengthSq = clamp(lengthSq, 0, 1);
        nrm_l.z =  sqrt(lengthSq);
        vec3 tan_w = normalize(IN.tan_w.xyz);
        vec3 binrm_w = cross(nrm_w, tan_w) * IN.tan_w.w;
        normal = nrm_l.x * tan_w + nrm_l.y * binrm_w + nrm_l.z * nrm_w;
    }

    vec3 view_w = cameraMtx[2].xyz;
    vec3 refl_w = 2 * dot(view_w, normal) * normal - view_w;

    // 決め打ちライトベクトル
    vec3[2] light_w = vec3[2](
        vec3(1, 1, 1), vec3(-1, 1, 1)
    );

    // light
    vec3 diffRefl = vec3(0);
    float specRefl = 0;

    for (int i = 0; i < NUM_LIGHT; ++i)
    {
        light_w[i] = normalize(light_w[i]); // 本来は CPU で正規化
        vec3 light = light_w[i]; // TODO: 減衰やカラー

        // diffuse
        diffRefl += clamp(dot(light, nrm_w), 0, 1);

        // specular
        specRefl += pow(clamp(dot(light, refl_w), 0, 1), specPower);
    }

    // projection
    if (NN_G3D_TO_BOOL(USE_PROJECTION_MAP))
    {
        vec2 projUV = IN.projXYW.xy / IN.projXYW.z;
        vec4 proj = texture(projTex, projUV);
        diffRefl += proj.rgb;
    }

    vec4 albedo = texture(albedoTex0, IN.uv0_uv1.xy);
    // multi texture
    vec4 albedo1 = texture(albedoTex1, IN.uv0_uv1.zw);
    switch (MULTI_TEX)
    {
        case c_Blend:
            albedo = albedo * (1 - weightTex) + albedo1 * weightTex;
            break;
        case c_Mul:
            albedo *= albedo1;
            break;
        case c_Add:
            albedo += albedo1 * weightTex;
            break;
    }

    o_Color = albedo * (NN_G3D_TO_BOOL(USE_VERTEX_COLOR) ? IN.color : diffColor);
    o_Color.rgb *= diffRefl;

    vec3 specular = vec3(specRefl);

    // reflection
    if (NN_G3D_TO_BOOL(USE_REFLECTION_MAP))
    {
        refl_w.z *= -1;
        vec4 refl = texture(reflTex, refl_w);
        specular += refl.rgb;
    }

    // environment
    if (NN_G3D_TO_BOOL(USE_ENVIRONMENT_MAP))
    {
        vec3 envNormal = NN_G3D_TRANSFORM_POS(texsrtEnv, vec4(normal, 1));
        if (envNormal.z >= 0)
        {
            vec4 env = texture(envTex, envNormal.xy);
            specular += env.rgb;
        }
    }

    o_Color.rgb += specInt * specular;

    //o_Color = IN.debug;
}

#endif // NN_G3D_FRAGMENT_SHADER
