﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <common.glsl>
#include <skeleton_block.glsl>
#include <shape_block.glsl>
#include <view_block.glsl>

#define c_Local ( 0 )
#define c_World ( c_Local + 1 )

#define ENABLE_NORMAL_MAP        ( 0 )       // @@ id="enable_normal_map"
#define ENABLE_INDIRECT_TEXTURE  ( 0 )       // @@ id="enable_indirect"
#define ENABLE_ALPHA_TEST        ( 0 )       // @@ id="enable_alpha_test"
#define COORDINATE               ( c_Local ) // @@ id="coord" choice="c_Local, c_World" type="dynamic"
#define SKINNING                 ( -1 )      // @@ id="skinning" default="-1" choice="[-1, 8]" type="dynamic"

layout(std140) uniform Option    // @@ id="opt" type="option"
{
    int enableVtxColor;          // @@ id="enable_vertex_color"
    int enableAlbedoTex;         // @@ id="enable_albedo"
    int enableBakeTex;           // @@ id="enable_bake_texture"
    int enableNormMap;           // @@ id="enable_normal_map"
    int enableSpecMask;          // @@ id="enable_specular_mask"
    int enableAOTex;             // @@ id="enable_ssao"
    int enableIndirectTex;       // @@ id="enable_indirect"
    int enableColorBufferAlbedo; // @@ id="enable_color_buffer_albedo"
    int receiveDynamicShadow;    // @@ id="receive_dynamic_shadow"
    int enableTransTexture;      // @@ id="enable_trans_texture"
    int enableWaterBufferAlbedo; // @@ id="enable_water_buffer_albedo"
    int enableReflect;           // @@ id="enable_reflect"
    int enableAlphaTest;         // @@ id="enable_alpha_test"
};

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 4)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 5)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 6)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
layout(location = 8)  in vec2   i_UV0;          // @@ id="_u0"   hint="uv0"
layout(location = 12) in vec4   i_Binormal;     // @@ id="_b0"   hint="binormal0"

/* @@
    attrib_id="_p0"   label="位置座標"                    editable="false"
    attrib_id="_n0"   label="法線"                        editable="false"
    attrib_id="_t0"   label="接線"                        editable="false"
    attrib_id="_i0"   label="ブレンドインデックス1-4"     editable="false"
    attrib_id="_i1"   label="ブレンドインデックス5-8"     editable="false"
    attrib_id="_w0"   label="ブレンドウェイト1-4"         editable="false"
    attrib_id="_w1"   label="ブレンドウェイト5-8"         editable="false"
    attrib_id="_u0"   label="アルベドテクスチャ用UV"
    interleave="_p0 _n0 _t0 _i0 _i1 _w0 _w1 _u0 "
*/

#endif // NN_G3D_VERTEX_SHADER

#ifndef NN_G3D_GEOMETRY_SHADER

NN_G3D_RASTER
{
    vec3 pos_w;
    vec3 nrm_w;
    vec4 tan_w;
    vec2 uv;
    vec4 projection;
} nn_g3d_raster;

#endif // !NN_G3D_GEOMETRY_SHADER

#ifdef NN_G3D_FRAGMENT_SHADER

layout(location = 0)    out vec4 o_Normal;

#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

uniform sampler2D albedoTex;    // @@ id="_a0" hint="albedo0"
uniform sampler2D normalTex;    // @@ id="_n0" hint="normal0"
uniform sampler2D indirectTex;  // @@ id="_a3" hint="albedo3"

layout(std140) uniform Material // @@ id="mat" type="material"
{
    vec4 texsrt0[2];            // @@ id="texsrt0" type="texsrt" hint="albedo0"
    vec4 texsrt1[2];            // @@ id="texsrt1" type="texsrt" hint="bake"
    vec4 texsrt2[2];            // @@ id="texsrt2" type="texsrt" hint="indirect"
    vec3 diffColor;             // @@ id="diffuse" default="1 1 1" item="color" hint="diffuse"
    float specInt;              // @@ id="spec_int" min="0" max="1" default="0.2"
    float specPower;            // @@ id="spec_power" min="0" default="50"
    vec3 albedoColor;           // @@ id="albedo_color" default="1 1 1" item="color" hint="albedo"
    vec4 bake0_st;              // @@ id="bake0_st" default="1 1 0 0" hint="bake0_st"
    vec4 bake1_st;              // @@ id="bake1_st" default="1 1 0 0" hint="bake1_st"
    vec2 indirectScale;         // @@ id="ind_scale" default="1 1" hint="ind_scale"
    float reflectScale;         // @@ id="reflect_scale" default="0.02" hint="reflect_scale"
    float refractScale;         // @@ id="refract_scale" default="0.5" hint="refract_scale"
    float waterScale;           // @@ id="water_scale" default="0.5" hint="water_scale"
    float fresnelCoeff;         // @@ id="fresnel_coeff" default="0.1" hint="fresnel_coeff"
    float bakeLightScale;       // @@ id="bake_light_scale" default="1"
    float alphaTest;            // @@ id="alpha_test" default="0.0"
};

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void main()
{
#if COORDINATE == c_Local
    vec4 pos_w = vec4(0, 0, 0, 1);
    vec3 nrm_w = vec3(0);
    vec3 tan_w = vec3(0);

    int skinning = SKINNING != -1 ? SKINNING : vtxSkinCount;
    if (skinning == 0)
    {
        // rigid body
        pos_w.xyz = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC(shapeMtx, i_Tangent);
    }
    else if (skinning == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        nrm_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        tan_w.xyz = NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
    }
    else if (skinning <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < skinning; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
    else if (skinning <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
        for (int i = 4; i < skinning; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            tan_w.xyz += weight * NN_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Tangent);
        }
    }
#elif COORDINATE == c_World
    vec4 pos_w = i_Position;
    vec3 nrm_w = i_Normal;
    vec3 tan_w = i_Tangent.xyz;
#endif

    vec4 pos_v = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);
    vec4 out_pos = NN_G3D_PROJECT(projMtx, pos_v);

    gl_Position = out_pos;
    OUT.pos_w = pos_w.xyz;
    OUT.nrm_w = normalize(nrm_w);
    OUT.tan_w = vec4(tan_w, i_Tangent.w);
    OUT.uv = NN_G3D_TRANSFORM_UV(texsrt0, i_UV0);
    OUT.projection = out_pos;
}

#endif // NN_G3D_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

void main()
{
    vec3 nrm_w = normalize(IN.nrm_w);
    vec2 normal_uv = IN.uv;
    vec2 color_buffer_uv = IN.projection.xy / IN.projection.w * 0.5 + 0.5;

    // indirect
    if (NN_G3D_TO_BOOL(ENABLE_INDIRECT_TEXTURE))
    {
        normal_uv += texture(indirectTex, normal_uv).rg * indirectScale;
    }

    // normal
    vec3 normal = nrm_w;
    if (NN_G3D_TO_BOOL(ENABLE_NORMAL_MAP))
    {
        vec3 nrm_l;
        nrm_l.xy = texture(normalTex, normal_uv).xy;
        float lengthSq = 1 - nrm_l.x * nrm_l.x - nrm_l.y * nrm_l.y;
        lengthSq = clamp(lengthSq, 0, 1);
        nrm_l.z =  sqrt(lengthSq);
        vec3 tan_w = normalize(IN.tan_w.xyz);
        vec3 binrm_w = cross(nrm_w, tan_w) * IN.tan_w.w;
        normal = nrm_l.x * tan_w + nrm_l.y * binrm_w + nrm_l.z * nrm_w;
    }

    vec4 out_normal = vec4(normal * 0.5 + 0.5, 1.0);
    if (NN_G3D_TO_BOOL(ENABLE_ALPHA_TEST))
    {
        out_normal.a = texture(albedoTex, normal_uv).a;
        if (out_normal.a < alphaTest)
        {
            discard;
        }
    }
    o_Normal = out_normal;
}

#endif // NN_G3D_FRAGMENT_SHADER
