﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Edit.h"
#include <nn/g3d/g3d_Viewer.h>

#define LOG_CURRENT_CALLBACK()  NN_LOG("Callback %s called\n", __FUNCTION__)

// Edit 内でロードされたシェーダーを参照します。
extern nn::g3d::ResShaderArchive*   g_pDemoShaderArchive;
extern nn::g3d::ResShadingModel*    g_pDebugShader;
extern nn::g3d::ResShaderArchive*   g_pBranchShader;

namespace
{
    bool    g_IsBranchShaderAttached = false;

    // 未初期化の RenderMaterial を初期化し、RenderShape に設定します。
    void InitializeMaterialsAndShapes(nns::g3d::RenderModel* pRenderModel, int passIndex, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
    {
        nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
        nn::g3d::ResModel* pResModel = pRenderModel->GetResModel();

        // RenderMaterial の初期化
        int materialCount = pResModel->GetMaterialCount();
        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
        {
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);
            if (pRenderMaterial->IsInitialized())
            {
                continue;
            }

            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
            bool isInitialized = pRenderMaterial->Initialize(pDevice, pResMaterial, pResShadingModel);
            NN_ASSERT(isInitialized);
            NN_UNUSED(isInitialized);
        }

        // RenderShape の初期化
        int shapeCount = pResModel->GetShapeCount();
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            nns::g3d::RenderShape* pRenderShape = pRenderModel->GetRenderShape(passIndex, shapeIndex);
            if (pRenderShape->IsInitialized())
            {
                continue;
            }

            nn::g3d::ResShape* pResShape = pResModel->GetShape(shapeIndex);
            int materialIndex = pResShape->GetMaterialIndex();
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(passIndex, materialIndex);
            NN_ASSERT(pRenderMaterial->IsInitialized());

            bool isInitialized = pRenderShape->Initialize(pDevice, pResShape, pRenderMaterial);
            NN_ASSERT(isInitialized);
            NN_UNUSED(isInitialized);
        }
    }
}

EditViewer::EditViewer() NN_NOEXCEPT
    : m_ModelInstanceCount(1)
{
}

int EditViewer::GetModelInstanceCount() const NN_NOEXCEPT
{
    return m_ModelInstanceCount;
}

// 3DEditor でモデルがロードされた時に作成するモデルのインスタンス数を設定します。
void EditViewer::SetModelInstanceCount(int count) NN_NOEXCEPT
{
    NN_ASSERT_GREATER(count, 0);
    m_ModelInstanceCount = count;
}

void EditViewer::Update() NN_NOEXCEPT
{
    // ビューアーライブラリーに登録したコマンド、3DEditor から受信したコマンドを処理します。
    nn::g3d::viewer::ViewerServer::GetInstance().ExecuteCommands();
}

// 3DEditor でテクスチャーがロードされたときにコールされます。
void EditViewer::TextureFileLoaded(const nn::g3d::viewer::TextureFileLoadedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    g3ddemo::SetupTexture(g3ddemo::GetGfxFramework()->GetDevice(), arg.pResTextureFile);
}

// 3DEditor でテクスチャーがアンロードされたときにコールされます。
void EditViewer::TextureFileUnloaded(const nn::g3d::viewer::TextureFileUnloadedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    g3ddemo::CleanupTexture(g3ddemo::GetGfxFramework()->GetDevice(), arg.pResTextureFile);
}

// 3DEditor でモデルがロードされたときにコールされます。
void EditViewer::ModelFileLoaded(nn::g3d::viewer::ModelFileLoadedOutArg& outArg, const nn::g3d::viewer::ModelFileLoadedArg& inArg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    // 3DEditor が開いたモデルがリソースとして送られてきます。
    // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
    nn::g3d::ResFile* pResFile = g3ddemo::CreateCopiedResFile(inArg.pResFile, inArg.fileSize, inArg.alignment);
    if (pResFile == nullptr)
    {
        g3ddemo::SendErrorMessageTo3dEditor("メモリー確保に失敗しました");
        return;
    }

    // コピーした ResFile は nn::g3d::viewer に渡す必要があります。
    outArg.pResFile = pResFile;

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();

    pHolder->files.PushBack(pResFile);
    g3ddemo::RegisterSamplerToDescriptorPool(pResFile);

    // このコールバックの外で遅延初期化を行うことも可能ですが、
    // このサンプルではコールバック内で ResFile からモデルを作って nn::g3d::viewer に登録します。
    // モデルインスタンス構築処理は、アプリケーションで使用している構築処理を
    // そのまま使用することができます。
    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        CreateRenderModelObjs(pResModel, m_ModelInstanceCount, QueueAttachModel);
    }
}

// 3DEditor でモデルがアンロードされたときにコールされます。
void EditViewer::ModelFileUnloaded(const nn::g3d::viewer::ModelFileUnloadedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    // 3DEditor でファイルが閉じられたので、関連するデータを破棄します。
    // GPU が参照中のデータを破棄してしまわないよう、アプリケーションの
    // フレームワークに応じて描画完了待ちを行ったり、描画をスキップするなど、
    // 必要な対策を講じてください。
    // このデモでは GPU による描画が完了してから呼ばれているため、
    // 特に対策は行っていません。

    // このコールバックの外でモデルの遅延破棄処理を行うこともできますが、
    // このサンプルではコールバック内ですぐに破棄を行います。

    nn::g3d::ResFile*           pResFile = arg.pResFile;
    nn::gfx::Device*            pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder*    pHolder = GetResourceHolder();

    for (int resModelIndex = 0, resModelCount = pResFile->GetModelCount(); resModelIndex < resModelCount; ++resModelIndex)
    {
        nn::g3d::ResModel* pTargetResModel = pResFile->GetModel(resModelIndex);

        // RenderModelObj を破棄します。
        {
            int renderModelObjIndex = 0;
            while (renderModelObjIndex < pHolder->renderModelObjs.GetCount())
            {
                nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[renderModelObjIndex];
                const nn::g3d::ResModel*    pResModel = pRenderModelObj->GetModelObj()->GetResource();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyRenderModelObj(pRenderModelObj);
                    pHolder->renderModelObjs.Erase(renderModelObjIndex);
                    continue;
                }
                ++renderModelObjIndex;
            }
        }

        // modelAnimObj を破棄します。
        {
            int modelAnimObjIndex = 0;
            while (modelAnimObjIndex < pHolder->modelAnimObjs.GetCount())
            {
                nns::g3d::ModelAnimObj*     pModelAnimObj = pHolder->modelAnimObjs[modelAnimObjIndex];
                const nn::g3d::ResModel*    pResModel = pModelAnimObj->GetModelObj()->GetResource();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyModelAnimObj(pDevice, pModelAnimObj);
                    pHolder->modelAnimObjs.Erase(modelAnimObjIndex);
                    continue;
                }
                ++modelAnimObjIndex;
            }
        }

        // RenderModel を破棄します。
        {
            int renderModelIndex = 0;
            while (renderModelIndex < pHolder->renderModels.GetCount())
            {
                nns::g3d::RenderModel*    pRenderModel = pHolder->renderModels[renderModelIndex];
                const nn::g3d::ResModel*    pResModel = pRenderModel->GetResModel();
                if (pResModel == pTargetResModel)
                {
                    nns::g3d::DestroyRenderModel(pDevice, pRenderModel);
                    pHolder->renderModels.Erase(renderModelIndex);
                    continue;
                }
                ++renderModelIndex;
            }
        }

    }

    // リソースファイルを破棄します。
    for (int fileIndex = 0, fileCount = pHolder->files.GetCount(); fileIndex < fileCount; ++fileIndex)
    {
        if (pHolder->files[fileIndex] == pResFile)
        {
            g3ddemo::UnregisterSamplerFromDescriptorPool(pResFile);
            pResFile->Cleanup(pDevice);
            g3ddemo::FreeMemory(pResFile);
            pHolder->files.Erase(fileIndex);
            break;
        }
    }

    // モデルのインデックスがずれるので範囲外のインデックスを 0 に戻します。
    if (GetSelectedModelIndex() >= pHolder->modelAnimObjs.GetCount())
    {
        SetSelectedModelIndex(pHolder->modelAnimObjs.GetCount() - 1);
    }
}

void EditViewer::ModelAttached(const nn::g3d::viewer::ModelAttachedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    NN_UNUSED(arg);
}

void EditViewer::ModelDetached(const nn::g3d::viewer::ModelDetachedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    for (nns::g3d::ModelAnimObj* pModelAnimObj : GetResourceHolder()->modelAnimObjs)
    {
        if (pModelAnimObj->GetBindModelObj() != arg.pModelObj)
        {
            continue;
        }
        pModelAnimObj->BindTo(nullptr, -1);
    }
}

void EditViewer::ShaderAssignUpdated(const nn::g3d::viewer::ShaderAssignUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    switch (arg.state)
    {
    case nn::g3d::viewer::ShaderAssignUpdatedArg::State_ModelBegin:
        NN_LOG("ModelBegin\n");
        break;
    case nn::g3d::viewer::ShaderAssignUpdatedArg::State_ModelUpdate:
        NN_LOG("ModelUpdate\n");
        break;
    case nn::g3d::viewer::ShaderAssignUpdatedArg::State_ModelEnd:
        NN_LOG("ModelEnd\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // モデルリソース (ResModel) の構造変更を伴う編集が 3DEditor で行われたため、
    // ビューアーライブラリーは古いモデルリソースの破棄と新しいモデルリソースの構築を行います。
    // モデルインスタンス (ModelObj) が新しいリソースを参照するように、
    // ビューアーライブラリーは参照関係を変更します。
    nn::gfx::Device*            pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder*    pHolder = GetResourceHolder();

    int materialCount = arg.pNewResModel->GetMaterialCount();
    nn::g3d::ResShaderArchive** pResShaderArchivePtr = nullptr;
    size_t size = sizeof(nn::g3d::ResShaderArchive*) * materialCount;
    pResShaderArchivePtr = g3ddemo::AllocateMemory<nn::g3d::ResShaderArchive*>(size);

    for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
    {
        // サンプラーの再構築
        {
            nn::g3d::ResMaterial* pOldResMaterial = arg.pOldResModel->GetMaterial(materialIndex);
            nn::g3d::ResMaterial* pNewResMaterial = arg.pNewResModel->GetMaterial(materialIndex);
            for (int samplerIndex = 0, samplerCount = pOldResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pOldResMaterial->GetSampler(samplerIndex);
                if (!pSampler->GetUserPtr())
                {
                    continue;
                }

                g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
            }

            for (int samplerIndex = 0, samplerCount = pNewResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pNewResMaterial->GetSampler(samplerIndex);
                if (pSampler->GetUserPtr())
                {
                    continue;
                }

                nn::gfx::DescriptorSlot descriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);
                pNewResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
            }
        }

        // シェーダーアーカイブを設定
        {
            if (arg.pMaterialShaderInfos)
            {
                pResShaderArchivePtr[materialIndex] = arg.pMaterialShaderInfos[materialIndex].pResShaderArchive;
                if (pResShaderArchivePtr[materialIndex])
                {
                    continue;
                }
            }

            pResShaderArchivePtr[materialIndex] = g_IsBranchShaderAttached ? g_pBranchShader : g_pDemoShaderArchive;
        }
    }

    // モデルリソースの構造の変化に伴い RenderModel の再構築を行います。
    for (nns::g3d::RenderModel* pRenderModel : pHolder->renderModels)
    {
        if (arg.pOldResModel != pRenderModel->GetResModel())
        {
            continue;
        }

        // RenderModel の再構築
        bool isShaderAssigned = true;
        {
            pRenderModel->Finalize(pDevice);
            pRenderModel->Initialize(pDevice, arg.pNewResModel, DrawPassType_Count);
            isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Default, pResShaderArchivePtr, materialCount);
            isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Debug, g_pDebugShader);
        }

        // アプリケーションがサポートしないシェーダーを割り当てたモデルが 3DEditor に読み込まれていたり、
        // 読み込み済みのモデルにサポート外のシェーダーが割り当てられた場合に 3DEditor にエラーログを出力します。
        if (!isShaderAssigned)
        {
            nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
                "シェーダーのバインドに失敗しました。demo シェーダーを割り当てます。",
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Error,
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
            nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

            nn::g3d::ResShadingModel* pDemoResShadingModel = g_pDemoShaderArchive->FindShadingModel("basic");
            NN_ASSERT_NOT_NULL(pDemoResShadingModel);
            nn::g3d::ResShadingModel* pResShadingModels[DrawPassType_Count] =
            {
                pDemoResShadingModel,
                g_pDebugShader
            };

            for (int passIndex = 0; passIndex < DrawPassType_Count; ++passIndex)
            {
                InitializeMaterialsAndShapes(pRenderModel, passIndex, pResShadingModels[passIndex]);
            }
        }
    }

    // ShaderScratchMemory のサイズを再計算します。
    {
        g3ddemo::Vector<nn::g3d::ResShaderArchive*> shaderArchives;
        g3ddemo::CreateVector<nn::g3d::ResShaderArchive*>(&shaderArchives, materialCount + pHolder->shaderFiles.GetCount());

        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
        {
            shaderArchives.PushBack(pResShaderArchivePtr[materialIndex]);
        }

        for (nn::g3d::ResShaderFile* pResShaderFile : pHolder->shaderFiles)
        {
            shaderArchives.PushBack(pResShaderFile->GetResShaderArchive());
        }
        g3ddemo::SetShaderScratchMemory(shaderArchives);
        g3ddemo::DestroyVector<nn::g3d::ResShaderArchive*>(&shaderArchives);
    }

    g3ddemo::FreeMemory(pResShaderArchivePtr);
} // NOLINT

void EditViewer::MaterialUpdated(const nn::g3d::viewer::MaterialUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    // ShaderAssignUpdated() で行われた変更に伴って ModelObj に変更が必要な場合は
    // このコールバックで処理を行います。
    // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。
    // ShaderAssignUpdated() で初期化したリソースに従って MaterialObj や ShapeObj が
    // ビューアーライブラリーによって作り直されているので、必要な初期化を行います。

    nn::g3d::ModelObj* pModelObj = arg.pModelObj;

    // クリアカラーを更新します。
    SetClearColorFromEnvModel();

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();

    // RenderModel の変更に追従するため RenderModelObj を再構築します。
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        if (pRenderModelObj->GetModelObj() != pModelObj)
        {
            continue;
        }

        nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
        pRenderModelObj->Finalize();
        pRenderModelObj->Initialize(pModelObj, pRenderModel);

        // サンプラーやユニフォームブロックの割り当てなどを行います。
        SetupRenderModelObjForEdit(pRenderModelObj);

        // シェーダーが再構築されたので更新します。
        pRenderModelObj->UpdateShader(g3ddemo::GetGfxFramework()->GetDevice());
    }
}

void EditViewer::ShapeUpdated(const nn::g3d::viewer::ShapeUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    nns::g3d::ModelAnimObj* pModelAnimObj = arg.pModelObj->GetUserPtr<nns::g3d::ModelAnimObj>();

    // シェイプが更新されているので計算します。
    pModelAnimObj->Calculate();
}

void EditViewer::BoneBindUpdated(const nn::g3d::viewer::BoneBindUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
    // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
    nns::g3d::ModelAnimObj* pModelAnimObj = arg.pModelObj->GetUserPtr<nns::g3d::ModelAnimObj>();
    pModelAnimObj->BindTo(arg.pParentModelObj, arg.parentBoneIndex);

    // バインド後 位置と回転の情報を初期化します。
    pModelAnimObj->SetTranslate(nn::util::Vector3f(.0f, .0f, .0f));
    pModelAnimObj->SetRotate(nn::util::Vector3f(.0f, .0f, .0f));
    nn::g3d::viewer::ViewerServer::LayoutModelArg layoutModelArg(
        arg.pModelObj,
        pModelAnimObj->GetScale(),
        pModelAnimObj->GetRotate(),
        pModelAnimObj->GetTranslate()
    );
    nn::g3d::viewer::ViewerServer::GetInstance().QueueLayoutModelCommand(layoutModelArg);
}

void EditViewer::SamplerParamUpdated(const nn::g3d::viewer::SamplerParamUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    // サンプラーが更新されたのでディスクリプタプールへ再登録します。
    g3ddemo::ReregisterSamplerToDescriptorPool(arg.pModelObj);
}

// 3DEditor でモデルの配置情報が編集されたときにコールされます。
void EditViewer::ModelLayoutUpdated(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    SetModelLayout(arg);

    nns::g3d::ModelAnimObj* pModelAnimObj = arg.pModelObj->GetUserPtr<nns::g3d::ModelAnimObj>();
    if (g3ddemo::IsInitialTranslate(arg) && !pModelAnimObj->IsBound())
    {
        // 3DEditor で移動がリセットされた場合
        ResetModelPosition(arg.pModelObj);

        nn::g3d::viewer::ViewerServer::LayoutModelArg layoutModelArg(
            arg.pModelObj,
            pModelAnimObj->GetScale(),
            pModelAnimObj->GetRotate(),
            pModelAnimObj->GetTranslate()
        );
        nn::g3d::viewer::ViewerServer::GetInstance().QueueLayoutModelCommand(layoutModelArg);
    }
}

// 3DEditor からモデルの配置情報を要求されたときにコールされます。
void EditViewer::SendModelLayoutRequested(
    nn::g3d::viewer::SendModelLayoutRequestedOutArg& outputData,
    const nn::g3d::viewer::SendModelLayoutRequestedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();

    const nns::g3d::ModelAnimObj* pModelAnimObj = arg.pModelObj->GetUserPtr<nns::g3d::ModelAnimObj>();
    outputData.scale = pModelAnimObj->GetScale();
    outputData.rotate = pModelAnimObj->GetRotate();
    outputData.translate = pModelAnimObj->GetTranslate();
}

void EditViewer::ShaderAttached(const nn::g3d::viewer::ShaderAttachedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    NN_LOG("Shader \"%s\" attached\n", arg.pShaderArchive->GetName());

    // モデルシェーダーはリアルタイム編集しやすくするために、ユニフォーム変数分岐のシェーダーに切り替えます。
    if (arg.pShaderArchive == g_pBranchShader)
    {
        g_IsBranchShaderAttached = true;
        ChangeShader(arg.pShaderArchive);
    }

    // シェーダーが切り替わったので更新します。
    UpdateShader();
}

void EditViewer::ShaderDetached(const nn::g3d::viewer::ShaderDetachedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    NN_LOG("Shader \"%s\" detached\n", arg.pShaderArchive->GetName());

    // モデルシェーダーはプリプロセッサ分岐のシェーダーに切り替えます。
    if (arg.pShaderArchive == g_pBranchShader)
    {
        g_IsBranchShaderAttached = false;
        ChangeShader(g_pDemoShaderArchive);
    }

    // シェーダーが切り替わったので更新します。
    UpdateShader();
}

void EditViewer::SceneAnimBound(const nn::g3d::viewer::SceneAnimBoundArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    for (int i = 0; i < arg.cameraAnimObjCount; ++i)
    {
        nn::g3d::CameraAnimObj* pCameraAnimObj = arg.pCameraAnimObjs[i];
        NN_LOG("Camera Animation \"%s\" bound\n", pCameraAnimObj->GetResource()->GetName());
    }

    for (int i = 0; i < arg.fogAnimObjCount; ++i)
    {
        nn::g3d::FogAnimObj* pFogAnimObj = arg.pFogAnimObjs[i];
        NN_LOG("Fog Animation \"%s\" bound\n", pFogAnimObj->GetResource()->GetName());
    }

    for (int i = 0; i < arg.lightAnimObjCount; ++i)
    {
        nn::g3d::LightAnimObj* pLightAnimObj = arg.pLightAnimObjs[i];
        NN_LOG("Light Animation \"%s\" bound\n", pLightAnimObj->GetResource()->GetName());
    }
}

void EditViewer::SceneAnimUnbound(const nn::g3d::viewer::SceneAnimUnboundArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    for (int i = 0; i < arg.cameraAnimObjCount; ++i)
    {
        nn::g3d::CameraAnimObj* pCameraAnimObj = arg.pCameraAnimObjs[i];
        NN_LOG("Camera Animation \"%s\" unbound\n", pCameraAnimObj->GetResource()->GetName());
    }

    for (int i = 0; i < arg.fogAnimObjCount; ++i)
    {
        nn::g3d::FogAnimObj* pFogAnimObj = arg.pFogAnimObjs[i];
        NN_LOG("Fog Animation \"%s\" unbound\n", pFogAnimObj->GetResource()->GetName());
    }

    for (int i = 0; i < arg.lightAnimObjCount; ++i)
    {
        nn::g3d::LightAnimObj* pLightAnimObj = arg.pLightAnimObjs[i];
        NN_LOG("Light Animation \"%s\" unbound\n", pLightAnimObj->GetResource()->GetName());
    }
}

void EditViewer::ApplySceneAnimRequested(const nn::g3d::viewer::ApplySceneAnimRequestedArg& arg) NN_NOEXCEPT
{
    int cameraAnimCount = arg.cameraAnimObjCount;
    nn::g3d::CameraAnimObj** pCameraAnimObjs = arg.pCameraAnimObjs;
    for (int viewIndex = 0, viewCount = ViewCount; viewIndex < viewCount && viewIndex < cameraAnimCount; ++viewIndex)
    {
        const nn::g3d::CameraAnimObj* pCameraAnimObj = pCameraAnimObjs[viewIndex];
        const nn::g3d::CameraAnimResult* result = pCameraAnimObj->GetResult();
        nn::util::Matrix4x3fType viewMtx;
        nn::util::Vector3fType pos, up, aim;
        nn::util::VectorSet(&pos, result->view.pos[0], result->view.pos[1], result->view.pos[2]);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&aim, result->view.aim[0], result->view.aim[1], result->view.aim[2]);
        nn::util::MatrixLookAtRightHanded(
            &viewMtx, pos, aim, up);
        nn::util::Matrix4x4fType projMtx;
        MatrixPerspectiveFieldOfViewRightHanded(
            &projMtx, result->proj.fovy, result->proj.aspect, result->proj.nearZ, result->proj.farZ);
        SetViewProjectionMatrix(viewIndex, viewMtx, projMtx);
    }
}

void EditViewer::RenderInfoUpdated(const nn::g3d::viewer::RenderInfoUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    const nn::g3d::ModelObj* pModel = arg.pModelObj;
    const nn::g3d::MaterialObj* pMaterial = pModel->GetMaterial(arg.materialIndex);
    const nn::g3d::ResMaterial* pResMaterial = pMaterial->GetResource();
    const nn::g3d::ResRenderInfo* pResRenderInfo = pResMaterial->FindRenderInfo(arg.name);

    // アプリ内の環境モデルに対して変更された場合のみ更新します。
    nn::g3d::ModelObj* pEnvModel = GetResourceHolder()->modelAnimObjs[0]->GetModelObj();
    if (pEnvModel != pModel)
    {
        return;
    }

    if (pResRenderInfo)
    {
        if (strcmp(arg.name, RenderInfoClearColorName) == 0)
        {
            // クリアカラーの更新を行います。
            SetClearColor(pResRenderInfo);
        }
    }
}

void EditViewer::SendRenderInfoRequested(const nn::g3d::viewer::SendRenderInfoRequestedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    const nn::g3d::ModelObj* pModel = arg.pModelObj;
    const nn::g3d::MaterialObj* pMaterial = pModel->GetMaterial(arg.materialIndex);
    const nn::g3d::ResMaterial* pResMaterial = pMaterial->GetResource();
    const nn::g3d::ResRenderInfo* pResRenderInfo = pResMaterial->FindRenderInfo(RenderInfoClearColorName);
    if (pResRenderInfo)
    {
        // クリアカラーの値範囲を送信
        nn::g3d::viewer::ViewerServer::GetInstance().QueueSetRenderInfoFloatRangeCommand(RenderInfoClearColorName, 0.0f, 1.0f);

        // クリアカラー R のデフォルト値を送信
        nn::g3d::viewer::ViewerServer::GetInstance().QueuePushRenderInfoFloatDefaultCommand(RenderInfoClearColorName, 0.3f);

        // クリアカラー G のデフォルト値を送信
        nn::g3d::viewer::ViewerServer::GetInstance().QueuePushRenderInfoFloatDefaultCommand(RenderInfoClearColorName, 0.4f);

        // クリアカラー B のデフォルト値を送信
        nn::g3d::viewer::ViewerServer::GetInstance().QueuePushRenderInfoFloatDefaultCommand(RenderInfoClearColorName, 0.5f);
    }
}

// 3DEditor でモデルやマテリアルが選択されたときに実行される処理です。
void EditViewer::TargetSelected(const nn::g3d::viewer::TargetSelectedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    if (arg.targetKind == nn::g3d::viewer::TargetSelectedArg::TargetKind_Model)
    {
        // 選択されたモデルにフォーカスします。
        for (int modelIndex = 0, modelCount = GetResourceHolder()->modelAnimObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
        {
            nns::g3d::ModelAnimObj* pModelAnimObj = GetResourceHolder()->modelAnimObjs[modelIndex];
            if (pModelAnimObj->GetModelObj() == arg.pModelObj)
            {
                SetSelectedModelIndex(modelIndex);
                return;
            }
        }
    }
}

// アタッチシェーダー時、遅延コンパイル後にシェーダープログラムを変更した後に実行される処理です。
void EditViewer::ShaderProgramUpdated(const nn::g3d::viewer::ShaderProgramUpdatedArg& arg) NN_NOEXCEPT
{
    LOG_CURRENT_CALLBACK();
    NN_UNUSED(arg);
    g3ddemo::SetShaderScratchMemory(GetResourceHolder()->shaderFiles);
}

void EditViewer::ChangeShader(nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pResShaderArchive);
    NN_ASSERT_EQUAL(strcmp("demo", pResShaderArchive->GetName()), 0);

    nn::gfx::Device*            pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder*    pHolder = GetResourceHolder();

    // 利用するシェーダーリソースの構造の変化に伴い RenderModel の再構築を行います。
    for (nns::g3d::RenderModel* pRenderModel : pHolder->renderModels)
    {
        pRenderModel->DestroyDrawPass(pDevice, DrawPassType_Default);
        bool isInitialized = pRenderModel->CreateDrawPass(pDevice, DrawPassType_Default, pResShaderArchive);

        if (!isInitialized)
        {
            nn::g3d::ResShadingModel* pDemoResShadingModel = pResShaderArchive->FindShadingModel("basic");
            InitializeMaterialsAndShapes(pRenderModel, DrawPassType_Default, pDemoResShadingModel);
        }
    }

    // RenderModelObj の再構築を行います
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
        nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
        pRenderModelObj->Finalize();
        pRenderModelObj->Initialize(pModelObj, pRenderModel);

        // ユニフォームブロックの再割り当てをします。
        RegisterUniformBlock(pRenderModelObj);

         // ダミーサンプラーを割り当てます。
        nn::g3d::TextureRef dummyTextureRef = g3ddemo::GetDummyTextureRef();
        pRenderModelObj->BindDummySampler(dummyTextureRef.GetDescriptorSlot(), g3ddemo::GetDummySamplerDescriptorSlot());

        WriteDynamicKey(pRenderModelObj);

        // シェーダーが再構築されたので更新します。
        pRenderModelObj->UpdateShader(g3ddemo::GetGfxFramework()->GetDevice());
    }
}

void EditViewer::UpdateShader() NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : GetResourceHolder()->renderModelObjs)
    {
        pRenderModelObj->UpdateShader(g3ddemo::GetGfxFramework()->GetDevice());
    }
}

void EditViewer::QueueAttachModel(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT
{
    // ビューアーへの登録。
    // ビューアーライブラリーのために専用のクラスなどを作成する必要はありません。
    nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachModelCommand(pModelObj);
    NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success);
}
