﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
 * @examplesource{Edit.cpp,PageSampleG3dEdit}
 *
 * @brief
 * ビューアーライブラリーを使用して 3DEditor によるリアルタイム編集を行うサンプルプログラム
 */

/**
 * @page PageSampleG3dEdit G3dDemo Edit
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\edit.png
 *
 * @brief
 * サンプルプログラム Edit の解説です。ビューアーライブラリーを使用して 3DEditor によるリアルタイム編集を行います。
 *
 * @section PageSampleG3dEdit_SectionBrief 概要
 * ビューアーライブラリーを使用して 3DEditor によるリアルタイム編集を行うサンプルプログラムです。
 *
 * @section PageSampleG3dEdit_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dEdit_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dEdit_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボードによる入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左右ボタンまたは左右キー       </td><td> 選択モデルを切り替えます。 </td></tr>
 * <tr><td> A ボタンまたは A キー  </td><td> 選択モデルをアタッチします。           </td></tr>
 * <tr><td> B ボタンまたは B キー  </td><td> 選択モデルがアタッチされている場合にデタッチします。           </td></tr>
 * <tr><td> X ボタンまたは X キー  </td><td> デバッグシェーダーをアタッチします。           </td></tr>
 * <tr><td> Y ボタンまたは Y キー  </td><td> マテリアルシェーダーをアタッチします。           </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dEdit_SectionPrecaution 注意事項
 * リアルタイム編集を行うためには 3DEditor、Target Manager を起動してアプリケーションと接続する必要があります。
 *
 * @section PageSampleG3dEdit_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから Edit を選択し、実行してください。
 *
 * @section PageSampleG3dEdit_SectionDetail 解説
 * このサンプルプログラムは、3DEditor と接続することにより 3DEditor に読み込まれているモデルを表示することができます。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - コマンドバッファーを初期化
 * - フレームバッファーを初期化
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャー初期化
 * - モデルとアニメーションの関連付け
 * - シェーダー、テクスチャーとモデルの関連付け
 * - ビューアーライブラリーの初期化
 * - 3DEditor との通信スレッドを開始
 * - ループ開始
 * - ボタン入力処理(ビューアーライブラリーへのコマンド登録など)
 * - 各行列計算
 * - モデル更新
 * - コマンドリスト生成
 * - コマンドリスト実行
 * - スキャンバッファーをスワップ
 * - ビューアーライブラリーに登録したコマンドを実行
 * - ループ開始に戻る
 *
 */

#include "Edit.h"
#include <nn/g3d/g3d_Viewer.h>

using namespace nn::util;
using namespace nn::g3d;
using namespace nn::g3d::viewer;

nn::g3d::ResShaderArchive*  g_pDemoShaderArchive = nullptr;
nn::g3d::ResShadingModel*   g_pDebugShader = nullptr;
nn::g3d::ResShaderArchive*  g_pBranchShader = nullptr;

namespace {

#define RESOURCE_PATH "Resource:/"
#define SHADER_PATH RESOURCE_PATH "shader/"

const char* BfshaPath[] = {
    SHADER_PATH "demo.bfsha",
    SHADER_PATH "debug.bfsha",
    SHADER_PATH "demo_branch.bfsha", // アタッチ用の静的分岐シェーダー
};

const char* BfresPath[] = {
    RESOURCE_PATH "env.bfres",
    RESOURCE_PATH "human.bfres",
};

EditViewer  g_EditViewer;
EditMenu    g_EditMenu;

g3ddemo::ResourceHolder     g_Holder;
nn::g3d::ResShaderOption*   g_pOutputOption = nullptr;

nns::g3d::RenderView        g_RenderView;
nn::g3d::ModelObj*          g_pEnvModelObj;

g3ddemo::ScreenInfo    g_ScreenInfo;
nn::util::Vector3fType g_ClearColor;

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraTarget;
nn::util::Vector3fType g_CameraUp;

int g_ViewportWidth;
int g_ViewportHeight;

nn::gfx::ViewportScissorState g_ViewportScissorStateForNormal;
nn::gfx::ViewportScissorState g_ViewportScissorStateForDebug;
nn::gfx::ViewportScissorState g_ViewportScissorState;

int g_OutputOptionIndex = -1;
const int g_BufferingCount = 1;

// nns::g3d::RenderModelObj を生成し、ResourceHolder に登録します。
void CreateRenderModelObjs(nn::g3d::ResModel* pResModel, int instanceCount) NN_NOEXCEPT
{
    ::CreateRenderModelObjs(pResModel, instanceCount, nullptr);
}

// リソースの初期化
void InitializeResource(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_Holder.Initialize();

    // サンプラーの参照テクスチャーが空の場合に割り当てられるダミーテクスチャーの初期化
    g3ddemo::CreateDummyTextureAndSampler(pDevice);

    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    nn::g3d::ResShaderArchive* pDebugShaderArchive;
    nn::g3d::ResShaderArchive* pDemoBranchArchive;

    nn::g3d::ResShaderArchive** pShaderArchivePtr[] = {
        &pDemoShaderArchive,
        &pDebugShaderArchive,
        &pDemoBranchArchive
    };

    // シェーダーの初期化
    for (int pathIndex = 0; pathIndex < NN_ARRAY_SIZE(BfshaPath); ++pathIndex)
    {
        nn::g3d::ResShaderFile* pResShader = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(BfshaPath[pathIndex]);
        pResShader->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pResShader);

        nn::g3d::ResShaderArchive* pResShaderArchive = pResShader->GetResShaderArchive();
        if (strstr(BfshaPath[pathIndex], "demo_branch"))
        {
            g_pBranchShader = pResShaderArchive;
        }

        *pShaderArchivePtr[pathIndex] = pResShaderArchive;
    }

    g_pDemoShaderArchive = pDemoShaderArchive;

    for (nn::g3d::ResShaderFile* pShaderFile : g_Holder.shaderFiles)
    {
        nn::g3d::ResShaderArchive* pShaderArchive = pShaderFile->GetResShaderArchive();
        g_pDebugShader = pShaderArchive->FindShadingModel("debug");
        if (g_pDebugShader)
        {
            g_OutputOptionIndex = g_pDebugShader->FindDynamicOptionIndex("output");
            g_pOutputOption = g_pDebugShader->FindDynamicOption("output");
            NN_ASSERT_NOT_NULL(g_pOutputOption);
            g_EditMenu.SetSelectedItemIndex(EditMenu::ItemShader, g_pOutputOption->GetDefaultIndex());
            break;
        }
    }

    NN_ASSERT_NOT_NULL(g_pDebugShader);
    NN_ASSERT_NOT_NULL(g_pBranchShader);
    NN_ASSERT_NOT_NULL(g_pDemoShaderArchive);

    nn::g3d::ResFile* pEnvModelFile;
    nn::g3d::ResFile* pHumanModelFile;

    nn::g3d::ResFile** pResFilePtrArray[] =
    {
        &pEnvModelFile,
        &pHumanModelFile
    };

    // リソースファイルの初期化
    for (int pathIndex = 0; pathIndex < NN_ARRAY_SIZE(BfresPath); ++pathIndex)
    {
        nn::g3d::ResFile* pResFile = g3ddemo::LoadResource<nn::g3d::ResFile>(BfresPath[pathIndex]);
        pResFile->Setup(pDevice);
        g_Holder.files.PushBack(pResFile);

        // テクスチャーの初期化
        g3ddemo::SetupTexture(pDevice, pResFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pResFile);
        *pResFilePtrArray[pathIndex] = pResFile;
    }

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&g_Holder);

    // モデルリソースとシェーダーの関連付け
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
        {
            nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);

            // 同じリソースから複数のインスタンスを作る例として human_mini は 2 体作成します。
            int instanceCount = (0 == strcmp("human_mini", pResModel->GetName())) ? 2 : 1;
            CreateRenderModelObjs(pResModel, instanceCount);
        }
    }

    // ユニフォームブロックの管理を簡単にするため、ライティングなどの設定がマテリアルに設定されているモデルを生成し、
    // そのモデルのユニフォームブロックを他のモデルに割り当てます。
    g_pEnvModelObj = g_Holder.renderModelObjs[0]->GetModelObj();

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        // ユニフォームブロックを設定
        RegisterUniformBlock(pRenderModelObj);

        // ダミーテクスチャー・サンプラーを設定
        nn::gfx::DescriptorSlot dummyTextureDescriptorSlot;
        g3ddemo::GetDummyTexture()->GetUserDescriptorSlot(&dummyTextureDescriptorSlot);
        pRenderModelObj->BindDummySampler(dummyTextureDescriptorSlot, g3ddemo::GetDummySamplerDescriptorSlot());
    }

    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);
} // NOLINT

// モデルの位置をバウンディング球を元に計算し配置
void ResetModelPositions() NN_NOEXCEPT
{
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        ModelObj* pModelObj = pModelAnimObj->GetModelObj();
        ResetModelPosition(pModelObj);
    }
}

// スクリーン情報表示を初期化
void InitializeScreenInfo(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);
}

// ビューポートシザーステートの初期化
void InitializeViewportScissorState(nn::gfx::Device* pDevice, nn::gfx::ViewportScissorState* pViewportScissorState, int x, int y, int width, int height) NN_NOEXCEPT
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetOriginX(static_cast<float>(x));
        viewportInfo.SetOriginY(static_cast<float>(y));
        viewportInfo.SetWidth(static_cast<float>(width));
        viewportInfo.SetHeight(static_cast<float>(height));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetOriginX(x);
        scissorInfo.SetOriginY(y);
        scissorInfo.SetWidth(width);
        scissorInfo.SetHeight(height);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    pViewportScissorState->Initialize(pDevice, info);
}

// ビューポート初期化
void InitializeViewports(nns::gfx::GraphicsFramework* pGfxFramework, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    const int viewWidth = pGfxFramework->GetDisplayWidth();
    const int viewHeight = pGfxFramework->GetDisplayHeight();
    const int viewportWidth = viewWidth;
    const int viewportHeight = viewHeight / 2;
    {
        nn::util::VectorSetX(&g_ClearColor, 0.0f);
        nn::util::VectorSetY(&g_ClearColor, 0.0f);
        nn::util::VectorSetZ(&g_ClearColor, 0.0f);
    }

    g_ViewportWidth = viewportWidth;
    g_ViewportHeight = viewportHeight;

    // ビューポート初期化
    InitializeViewportScissorState(pDevice, &g_ViewportScissorStateForNormal, 0, viewportHeight, viewportWidth, viewportHeight);
    InitializeViewportScissorState(pDevice, &g_ViewportScissorStateForDebug, 0, 0, viewportWidth, viewportHeight);
    InitializeViewportScissorState(pDevice, &g_ViewportScissorState, 0, 0, viewWidth, viewHeight);
}

// ビューポートシザーステートの破棄
void FinalizeViewports(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_ViewportScissorStateForNormal.Finalize(pDevice);
    g_ViewportScissorStateForDebug.Finalize(pDevice);
    g_ViewportScissorState.Finalize(pDevice);
}

// カメラ初期化
void InitializeView(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::util::VectorSet(&g_CameraPosition, 0.0f, 82.5f, 330.0f);
    nn::util::VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
    nn::util::VectorSet(&g_CameraTarget, 0.0f, 33.0f, 0.0f);

    g_RenderView.Initialize(pDevice, ViewCount, g_BufferingCount);
}

// ビューアーライブラリーの初期化
void InitializeViewer() NN_NOEXCEPT
{
    g_EditViewer.Initialize(g_Holder);

    // ViewerServer の通信処理は別スレッドにすることができます
    g_EditViewer.StartPollThread();

    g_EditViewer.Open();
}

// ビューアーライブラリーの終了処理
void FinalizeViewer() NN_NOEXCEPT
{
    g_EditViewer.Close();
    g_EditViewer.StopPollThread();
    g_EditViewer.Finalize();
}

// 更新関数
void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData) NN_NOEXCEPT;

// 描画関数
void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT;

// Edit デモの初期化
void InitializeEdit(nns::gfx::GraphicsFramework* pGfxFramework) NN_NOEXCEPT
{
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_Immediate);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    InitializeScreenInfo(pDevice);

    // ビューポート初期化
    InitializeViewports(pGfxFramework, pDevice);

    // カメラ初期化
    InitializeView(pDevice);

    // リソース初期化
    InitializeResource(pDevice);

    // モデルの位置を初期化
    ResetModelPositions();

    // 環境モデルの描画情報のクリアカラーを適用
    SetClearColorFromEnvModel();

    // メニューの初期化
    g_EditMenu.Initialize(g_Holder, g_pBranchShader, g_pOutputOption);

    // ビューアーライブラリーの初期化
    InitializeViewer();

    // フレームワークに更新・描画関数を設定
    pGfxFramework->SetCalculateCallback(Calculate, nullptr);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, nullptr);
    pGfxFramework->ResetFrame();
}

// Edit デモの終了処理
void FinalizeEdit(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // ビューアーライブラリーの終了処理
    FinalizeViewer();

    // ビューアーライブラリーが所有しているモデルを破棄しないように、
    // ビューアの終了処理後に残りのデータの破棄を行います
    g3ddemo::DestroyAll(pDevice, &g_Holder);

    // ビュー用ユニフォームブロックの破棄
    g_RenderView.Finalize(pDevice);

    // ダミーテクスチャーの破棄
    g3ddemo::DeleteDummyTextureAndSampler(pDevice);

    // ビューポートの破棄
    FinalizeViewports(pDevice);

    // スクリーン情報表示の破棄
    g_ScreenInfo.Cleanup(pDevice);
}

// Env ユニフォームブロックの更新
void RegisterEnvUniformBlock(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    const nn::gfx::Buffer* buffer = g_pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);

    // シェーダーアサインに定義されてるシェーディングモデル名が env だった場合には、
    // マテリアルユニフォームブロックを利用するため設定を行いません。
    nn::g3d::ModelObj* pModelObj = pRenderModelObj->GetModelObj();
    for (int shapeIndex = 0, shapeCount = pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        int materialIndex = pModelObj->GetShape(shapeIndex)->GetMaterialIndex();
        const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
        const nn::g3d::ResShaderAssign* pShaderAssign = pMaterialObj->GetResource()->GetShaderAssign();
        const char* shadingModelName = pShaderAssign->GetShadingModelName();

        if (strcmp(shadingModelName, "env") == 0)
        {
            continue;
        }

        pRenderModelObj->GetRenderUnitObj(shapeIndex)->BindUniformBlock(DrawPassType_Default, "env", &buffer, g_BufferingCount);
    }
}

// Edit デモの更新処理
void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    g_EditMenu.CalculateCPU(&g_EditViewer, pad);

    // ビューアーの更新
    g_EditViewer.Update();

    // モデルの更新
    {
        for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
        {
            pModelAnimObj->CalculateAnimations();

            // 3DEditor でスケルタルア二メーションが再生されていた場合には、
            // CalculateAnimations() で更新されたアニメーションを上書きします。
            // また、キャッシュを活用するためにスケルタルアニメーションの計算から
            // ワールド行列計算までを連続して行います。
            nn::g3d::viewer::ViewerServer::GetInstance().CalculateSkeletalAnimations(pModelAnimObj->GetModelObj());

            pModelAnimObj->CalculateWorld();
            pModelAnimObj->CalculateBounding();
            pModelAnimObj->CalculateView(0, 0, g_RenderView.GetViewMtx(0));
        }

        nn::g3d::viewer::ViewerServer::GetInstance().CalculateAnimations();

        for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
        {
            pModelAnimObj->CalculateUniformBlock(0);
        }

        for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
        {
            RegisterEnvUniformBlock(pRenderModelObj);
            pRenderModelObj->UpdateShader(pDevice);
        }
    }

    // カメラの更新
    {
        int selectedModelIndex = g_EditMenu.GetSelectedItemIndex(EditMenu::ItemModel);
        if (selectedModelIndex < g_Holder.modelAnimObjs.GetCount())
        {
            // アニメーションしたときにも選択モデルの動きを追従できるよう毎フレーム
            // カメラの位置を更新する
            nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[selectedModelIndex];
            g3ddemo::LookAtModel(&g_CameraPosition, &g_CameraTarget, pModelAnimObj->GetModelObj());
        }

        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(
                &matrix, g_CameraPosition, g_CameraTarget, g_CameraUp);
            g_RenderView.SetViewMtx(0, matrix);
        }

        const float aspect = static_cast<float>(g_ViewportWidth) / static_cast<float>(g_ViewportHeight);
        {
            // 投影行列
            nn::util::Matrix4x4fType matrix;
            MatrixPerspectiveFieldOfViewRightHanded(
                &matrix, nn::util::DegreeToRadian(45.0f), aspect, 10.0f, 40000.0f);
            g_RenderView.SetProjectionMtx(0, matrix);
        }

        g_RenderView.Calculate(0);
    }
}

// Edit デモの描画処理
void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);

    pGfxFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pGfxCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        // カラーターゲットをクリア
        nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
        pGfxCommandBuffer->ClearColor(pColor, nn::util::VectorGetX(g_ClearColor), nn::util::VectorGetY(g_ClearColor), nn::util::VectorGetZ(g_ClearColor), 1.0f, nullptr);

        // 深度ステンシルをクリア
        nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
        pGfxCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        // レンダーターゲットをセット
        pGfxCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

        // 通常パスの描画
        {
            // ビューポートをセットする
            pGfxCommandBuffer->SetViewportScissorState(&g_ViewportScissorStateForNormal);

            for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
            {
                pRenderModelObj->Draw(pGfxCommandBuffer, DrawPassType_Default, 0, 0);
            }
        }

        // デバッグパスの描画
        {
            // ビューポートをセットする
            pGfxCommandBuffer->SetViewportScissorState(&g_ViewportScissorStateForDebug);

            for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
            {
                pRenderModelObj->Draw(pGfxCommandBuffer, DrawPassType_Debug, 0, 0);
            }
        }

        // テキスト描画 コマンド生成
        {
            // ビューポートを全画面に戻します。
            pGfxCommandBuffer->SetViewportScissorState(&g_ViewportScissorState);

            g_EditMenu.CalculateGPU(g_ScreenInfo);
            g_ScreenInfo.Draw(pGfxCommandBuffer);
        }
    }
    pGfxFramework->EndFrame(bufferIndex);
}

} // anonymous namespace

// nns::g3d::RenderModelObj を生成し、ResourceHolder に登録します。
void CreateRenderModelObjs(nn::g3d::ResModel* pResModel, int instanceCount, CreateModelInstanceCallback pFunction) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();

    // モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステートを管理する RenderModel を作成
    nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel, DrawPassType_Count);
    pRenderModel->CreateDrawPass(pDevice, DrawPassType_Default, g_pDemoShaderArchive);
    pRenderModel->CreateDrawPass(pDevice, DrawPassType_Debug, g_pDebugShader);
    g_Holder.renderModels.PushBack(pRenderModel);

    // 3dEditor からシェイプアニメーションが送られてくる可能性があるため、動的頂点バッファーの生成を有効にします。
    pResModel->ActivateDynamicVertexAttrForShapeAnim();

    // モデルインスタンスの情報を設定
    nn::g3d::ModelObj::Builder builder(pResModel);
    {
        builder.SetBoundingEnabled();
        builder.ShapeBufferingCount(g_BufferingCount);
        builder.SkeletonBufferingCount(g_BufferingCount);
        builder.MaterialBufferingCount(g_BufferingCount);
        builder.ViewCount(ViewCount);
    }

    for (int instanceIndex = 0; instanceIndex < instanceCount; ++instanceIndex)
    {
        // モデルの計算処理を簡略化する ModelAnimObj を作成
        nns::g3d::ModelAnimObj* pModelAnimObj;
        pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);

        nn::g3d::ModelObj* pModelObj = pModelAnimObj->GetModelObj();
        pModelObj->SetShapeAnimCalculationEnabled();
        pModelObj->SetUserPtr(pModelAnimObj);
        g_Holder.modelAnimObjs.PushBack(pModelAnimObj);

        // RenderModel と ModelObj を組み合わせ、描画を行う RenderModelObj を作成
        nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pModelObj, pRenderModel);
        g_Holder.renderModelObjs.PushBack(pRenderModelObj);

        // モデルインスタンス単位で必要な処理が設定されていた場合、実行します。
        if (pFunction)
        {
            pFunction(pModelObj);
        }
    }
}

// RenderModelObj を描画するために必要なユニフォームブロックを割り当てます。
void RegisterUniformBlock(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    RegisterEnvUniformBlock(pRenderModelObj);

    const nn::gfx::Buffer* buffer;
    buffer = g_RenderView.GetViewUniformBlock(0, 0);
    pRenderModelObj->BindUniformBlock("view", &buffer, g_BufferingCount);
}

void SetViewProjectionMatrix(int index, const nn::util::Matrix4x3fType& viewMtx, const nn::util::Matrix4x4fType& projMtx) NN_NOEXCEPT
{
    g_RenderView.SetViewMtx(index, viewMtx);
    g_RenderView.SetProjectionMtx(index, projMtx);
}

void SetClearColorFromEnvModel() NN_NOEXCEPT
{
    NN_ASSERT(g_pEnvModelObj->GetMaterialCount() > 0);

    const ResMaterial* pResMaterial = g_pEnvModelObj->GetMaterial(0)->GetResource();
    for (int renderInfoIndex = 0, renderInfoCount = pResMaterial->GetRenderInfoCount();
        renderInfoIndex < renderInfoCount; ++renderInfoIndex)
    {
        const ResRenderInfo* pResRenderInfo = pResMaterial->GetRenderInfo(renderInfoIndex);
        if (strcmp(RenderInfoClearColorName, pResRenderInfo->GetName()) == 0)
        {
            SetClearColor(pResRenderInfo);
            break;
        }
    }
}

void SetClearColor(const nn::g3d::ResRenderInfo* pResRenderInfo) NN_NOEXCEPT
{
    int length = pResRenderInfo->GetArrayLength();
    NN_ASSERT(pResRenderInfo->GetType() == nn::g3d::ResRenderInfo::Type_Float);
    NN_ASSERT(0 <= length && length <= 3);

    if (length > 0)
    {
        nn::util::VectorSetX(&g_ClearColor, pResRenderInfo->GetFloat(0));
    }
    if (length > 1)
    {
        nn::util::VectorSetY(&g_ClearColor, pResRenderInfo->GetFloat(1));
    }
    if (length > 2)
    {
        nn::util::VectorSetZ(&g_ClearColor, pResRenderInfo->GetFloat(2));
    }
}

void WriteDynamicKey() NN_NOEXCEPT
{
    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        WriteDynamicKey(pRenderModelObj);
    }
}

void WriteDynamicKey(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    g3ddemo::WriteSkinningOption(pRenderModelObj);

    for (int shapeIndex = 0, shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        pRenderModelObj->GetRenderUnitObj(shapeIndex)->GetShaderSelector(DrawPassType_Debug)->WriteDynamicKey(g_OutputOptionIndex, g_EditMenu.GetSelectedItemIndex(EditMenu::ItemShader));
    }
}

void ResetModelPosition(const ModelObj* pTargetModelObj) NN_NOEXCEPT
{
    nn::util::Vector3f position(0.0f, 0.0f, 0.0f);
    for(nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        ModelObj* pModelObj = pModelAnimObj->GetModelObj();
        pModelAnimObj->Calculate();
        nn::g3d::Sphere* pBoundingSphere = pModelObj->GetBounding();
        nn::util::VectorSetX(&position, nn::util::VectorGetX(position) + pBoundingSphere->radius);
        if (pModelObj == pTargetModelObj)
        {
            pModelAnimObj->SetTranslate(position);
            break;
        }

        nn::util::VectorSetX(&position, nn::util::VectorGetX(position) + pBoundingSphere->radius);
    }
}

void SetSelectedModelIndex(int index) NN_NOEXCEPT
{
    g_EditMenu.SetSelectedItemIndex(EditMenu::ItemModel, index);
}

int GetSelectedModelIndex() NN_NOEXCEPT
{
    return g_EditMenu.GetSelectedItemIndex(EditMenu::ItemModel);
}

void SetupRenderModelObjForEdit(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT
{
    // ダミーサンプラーを割り当てます。
    nn::g3d::TextureRef dummyTextureRef = g3ddemo::GetDummyTextureRef();
    pRenderModelObj->BindDummySampler(dummyTextureRef.GetDescriptorSlot(), g3ddemo::GetDummySamplerDescriptorSlot());

    // ユニフォームブロックを割り当てます。
    RegisterUniformBlock(pRenderModelObj);

    // シェーダーキーの更新
    WriteDynamicKey(pRenderModelObj);
}

int EditMain() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_Immediate);
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // 初期化
    InitializeEdit(pGfxFramework);

    bool restart = false;
    g3ddemo::Pad& pad = g3ddemo::GetPad();

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        g3ddemo::GetControllerManager()->Update();
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_R))
            {
                restart = true;
            }
            break;
        }

        pGfxFramework->ProcessFrame();
    }
    pGfxFramework->QueueFinish();

    FinalizeEdit(pDevice);

    return restart ? -1 : EXIT_SUCCESS;
} // NOLINT
