﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"

namespace g3ddemo = nn::g3d::demo;

namespace {

// タッチボタン
struct TouchButton
{
    const char* title;
    float positionX;
    float positionY;
    float textPositionX;
    float textPositionY;

    enum {
        width = 256,
        height = 56,
        space = 18,
    };
};

// 選択項目を記録する
int g_TouchButtonFocus = 1;

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int DemoLauncherMain(nn::g3d::demo::CmdMenu::Item* pItems)
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    g3ddemo::ScreenInfo screenInfo;
    screenInfo.Initialize(pDevice);
    screenInfo.SetWindowPosition(16, 8);
    screenInfo.SetWindowSize(0, 0);
    screenInfo.SetFontSize(20);

    // ボタンを作成する
    const int MaxButton = 9;
    TouchButton touchButtons[MaxButton];
    int buttonCount = 0;
    int buttonCountPerColumn = 6;
    int columnCount = 0;

    float basePositionX = 16.0f;
    float basePositionY = 144.0f;
    float offsetPositionX = TouchButton::width + TouchButton::space;
    float offsetPositionY = TouchButton::height + TouchButton::space;
    {
        float stringHeight = screenInfo.GetFontHeight();
        float positionX = basePositionX;
        float positionY = basePositionY;

        // 終了ボタン
        {
            // 文字列の幅高さ
            const char* exitTitle = "Exit";
            float stringWidth = screenInfo.GetTextWidth(exitTitle);

            TouchButton* pButton = &touchButtons[buttonCount++];
            pButton->title = exitTitle;
            pButton->positionX = positionX;
            pButton->positionY = positionY;
            pButton->textPositionX = positionX + (TouchButton::width - stringWidth) * 0.5f;
            pButton->textPositionY = positionY + (TouchButton::height - stringHeight) * 0.5f;
            positionY += offsetPositionY;
        }

        // デモを登録する
        for (nn::g3d::demo::CmdMenu::Item* pItem = pItems; pItem->pFunc; ++pItem)
        {
            if (buttonCount % buttonCountPerColumn == 0)
            {
                positionX += offsetPositionX;
                positionY = basePositionY;
                columnCount++;
            }
            // 文字列の幅高さ
            float stringWidth = screenInfo.GetTextWidth(pItem->title);

            TouchButton* pButton = &touchButtons[buttonCount++];
            pButton->title = pItem->title;
            pButton->positionX = positionX;
            pButton->positionY = positionY;
            pButton->textPositionX = positionX + (TouchButton::width - stringWidth) * 0.5f;
            pButton->textPositionY = positionY + (TouchButton::height - stringHeight) * 0.5f;
            positionY += offsetPositionY;
        }
    }

    pGfxFramework->ResetFrame();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();

    // メインループ
    uint32_t bufferIndex = 0;
    int result = -1;
    while (result == -1)
    {
        // タイトルテキスト
        screenInfo.StartPrint();
        screenInfo.Print("[g3d demo launcher]\n\n");
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        screenInfo.Print("You can operate this demo by keyboard.\n");
        screenInfo.Print("Up down key       : Select demo\n");
        screenInfo.Print("Return key        : Start demo\n");
        screenInfo.Print("Space key         : Return menu\n");
#else
        screenInfo.Print("You can operate this demo by Classic controler Pro.\n");
        screenInfo.Print("Up down button : Select demo\n");
        screenInfo.Print("A button       : Start demo\n");
        screenInfo.Print("Start button   : Return menu\n");
#endif

        // テクスチャー取得と vsync 待ち
        pGfxFramework->AcquireTexture(bufferIndex);
        pGfxFramework->WaitDisplaySync(bufferIndex, nn::TimeSpan::FromSeconds(2));

        // ボタンを表示する
        for (int idxButton = 0; idxButton < buttonCount; ++idxButton)
        {
            nn::util::Color4u8 color;
            if (idxButton == g_TouchButtonFocus)
            {
               color = nn::util::Color4u8(0, 150, 50, 255);
            }
            else
            {
               color = nn::util::Color4u8(64, 64, 64, 255);
            }

            // 矩形とデモ名称
            TouchButton* pButton = &touchButtons[idxButton];
            screenInfo.AddQuad(pButton->positionX, pButton->positionY, TouchButton::width, TouchButton::height, color);
            screenInfo.Print(pButton->textPositionX, pButton->textPositionY, pButton->title);
        }

         pGfxFramework->BeginFrame(bufferIndex);
         {
            nn::gfx::CommandBuffer* pGfxCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

            // カラーターゲットをクリアする
            nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
            pGfxCommandBuffer->ClearColor(pColor, 0.0f, 0.0f, 0.0f, 1.0f, nullptr);

            // 深度ステンシルをクリアする
            nn::gfx::DepthStencilView* pDepth = pGfxFramework->GetDepthStencilView();
            pGfxCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

            // レンダーターゲットをセットする
            pGfxCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

            // ビューポートシザーステートをセットする
            pGfxCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

            // 情報表示スクリーン
            screenInfo.Draw(pGfxCommandBuffer);
        }
        pGfxFramework->EndFrame(bufferIndex);

        // コマンドの実行と描画
        pGfxFramework->ExecuteCommand(bufferIndex);
        pGfxFramework->QueuePresentTexture(1);

        // GPU の完了待ち
        pGfxFramework->WaitGpuSync(bufferIndex, nn::TimeSpan::FromSeconds(2));

        bufferIndex = 1 - bufferIndex;

        // デバッグパッド
        if (pad.Read())
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
            {
                if (--g_TouchButtonFocus < 0)
                {
                    g_TouchButtonFocus = buttonCount - 1;
                }
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
            {
                if (++g_TouchButtonFocus >= buttonCount)
                {
                    g_TouchButtonFocus = 0;
                }
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                g_TouchButtonFocus -= buttonCountPerColumn;
                if (g_TouchButtonFocus < 0)
                {
                    int originalMod = (g_TouchButtonFocus + buttonCountPerColumn) % buttonCountPerColumn;
                    g_TouchButtonFocus = std::min(originalMod + (columnCount * buttonCountPerColumn), buttonCount - 1);
                }
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                g_TouchButtonFocus += buttonCountPerColumn;
                if ((g_TouchButtonFocus / buttonCountPerColumn) > columnCount)
                {
                    g_TouchButtonFocus %= buttonCountPerColumn;
                }
                else if (g_TouchButtonFocus >= buttonCount)
                {
                    g_TouchButtonFocus = buttonCount - 1;
                }
            }
            else
            {
                if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
                {
                    // 選択された
                    result = g_TouchButtonFocus;
                    break;
                }
            }
        }

        // タッチスクリーン
        nns::hid::ControllerManager* pControllerManager = g3ddemo::GetControllerManager();
        const nns::hid::TouchScreen* const pTouchScreen =
            reinterpret_cast<nns::hid::TouchScreen*>(pControllerManager->GetController(nns::hid::ControllerId_TouchScreen, 0));
        const std::vector<nns::hid::TouchScreen::TouchState>& states = pTouchScreen->GetTouchStates();
        if (states.size() > 0)
        {
            // ボタンが表示されている座標
            float marginPosX = screenInfo.GetWindowPositionX();
            float marginPosY = screenInfo.GetWindowPositionY();
            marginPosX += screenInfo.GetMarginWidth();
            marginPosY += screenInfo.GetMarginHeight();

            // タッチを検出
            const nns::hid::TouchScreen::TouchState& state = states[0];
            for (int idxButton = 0; idxButton < buttonCount; ++idxButton)
            {
                TouchButton* pButton = &touchButtons[idxButton];
                const float buttonPositionX = pButton->positionX + marginPosX;
                const float buttonPositionY = pButton->positionY + marginPosY;
                if (state.position.x > buttonPositionX && state.position.x < buttonPositionX + TouchButton::width)
                {
                    if (state.position.y > buttonPositionY && state.position.y < buttonPositionY + TouchButton::height)
                    {
                        if (state.IsEnded())
                        {
                            if (idxButton == g_TouchButtonFocus)
                            {
                                // 選択された
                                result = idxButton;
                                break;
                            }
                            else
                            {
                                // 選択を変更する
                                g_TouchButtonFocus = idxButton;
                            }
                        }
                    }
                }
            }
        }
    }

    // スクリーン情報表示の破棄
    screenInfo.Cleanup(pDevice);

    return result;
} // NOLINT
