﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsSetAllocator.cpp,PageSampleFsSetAllocator}
 *
 * @brief
 *  nn::fs::SetAllocator() のサンプルプログラム
 */

/**
 * @page PageSampleFsSetAllocator FS ライブラリ用アロケータの設定
 * @tableofcontents
 *
 * @brief
 *  nn::fs::SetAllocator() のサンプルプログラムの解説です。
 *
 * @section PageSampleFsSetAllocator_SectionBrief 概要
 *  ここでは、nn::fs::SetAllocator() のサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsSetAllocator_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsSetAllocator Samples/Sources/Applications/FsSetAllocator @endlink 以下にあります。
 *
 * @section PageSampleFsSetAllocator_SectionNecessaryEnvironment 必要な環境
 *  特になし。
 *
 * @section PageSampleFsSetAllocator_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsSetAllocator_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 *  また、このデモでは C:/Windows/Temp/FsSmplFile というエントリを作成し、操作後削除します。
 *
 *  デモ実行時に当該エントリが存在する場合や、デモプログラムが
 *  当該パスにアクセスできない場合正しく動作しませんのでご注意ください。
 *
 * @section PageSampleFsSetAllocator_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsSetAllocator_SectionDetail 解説
 *
 * @subsection PageSampleFsSetAllocator_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsSetAllocator.cpp
 *  @includelineno FsSetAllocator.cpp
 *
 * @subsection PageSampleFsSetAllocator_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - SetAllocator 関数によるアロケータの設定
 *  - メモリ確保・解放を伴う関数の実行
 *
 *  SetAllocator 関数の呼び出しにより、FS ライブラリでのメモリ確保に使用されるアロケータを指定することができます。
 *  アロケータには、8 バイトアライメントのアドレスを返すアロケータを設定してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *  FS ライブラリによってアロケータから確保されるメモリのサイズ、アドレス、確保回数は実行ターゲットによって変わる点にご注意ください。
 *
 *  @verbinclude  FsSetAllocator_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

namespace {
/**
   @brief FS用アロケータ
*/
    void* Allocate(size_t size)
    {
        auto p = std::malloc(size);
        NN_LOG("Allocated: 0x%p (%4d bytes)\n", p, size);
        return p;
    }

/**
   @brief FS用デアロケータ
*/
    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
        NN_LOG("Deallocated: 0x%p (%4d bytes)\n", p, size);
    }
} // namespace

extern "C" void nnMain()
{
    nn::Result result;
    const int FileSize = 1024;

    // FS ライブラリでのメモリ確保に使用されるアロケータを設定します。
    // アロケータには、8 バイトアライメントのアドレスを返すアロケータを設定する必要があります。
    // 設定したアロケータは、マウントやファイルオープンの際に使用されます。
    NN_LOG("Set allocator.\n");
    nn::fs::SetAllocator(Allocate, Deallocate);

    NN_LOG("Mount Host Root.\n");
    result = nn::fs::MountHostRoot();
    if( nn::fs::ResultTargetNotFound::Includes(result) )
    {
        // ホスト PC が認識されませんでした。
        // ホスト PC に正しく接続してください。
        NN_ASSERT(false, "Target not found.\n");
        return;
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Create C:/Windows/Temp/FsSmplFile.\n");
    result = nn::fs::CreateFile("C:/Windows/Temp/FsSmplFile", FileSize);
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // パスに含まれるディレクトリが存在しません。
        // ディレクトリを作成してください。
        NN_ASSERT(false, "Parent directory not found.");
        return;
    }
    else if( nn::fs::ResultPathAlreadyExists::Includes(result)
          || nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既に存在しています。
        // 対象ファイルが存在していても構わない場合はエラーハンドリングは不要です。
        // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
        // 必要であれば対象ファイルを削除してから再度作成してください。
    }
    else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
    {
        // 空き領域が不足しています。
        // ホスト側の空き領域を増やしてください。
        NN_ABORT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    NN_LOG("Open file.\n");
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "C:/Windows/Temp/FsSmplFile", nn::fs::OpenMode_Read);
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // 対象ファイルが存在しません。
        // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既に OpenMode_Write を指定してオープンされています。
        // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    NN_LOG("Close file.\n");
    nn::fs::CloseFile(fileHandle);

    NN_LOG("Delete file.\n");
    result = nn::fs::DeleteFile("C:/Windows/Temp/FsSmplFile");
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // 対象ファイルが存在しません。
        // 対象ファイルが必ず存在する場合は、このエラーハンドリングは不要です、
    }
    else if( nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既にオープンされています。
        // ファイルがオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    NN_LOG("Unmount Host Root.\n");
    nn::fs::UnmountHostRoot();

    return;
}
