﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsSdCardForDebug.cpp,PageSampleFsSdCardForDebug}
 *
 * @brief
 *  デバッグ用途の SD カードのファイルシステム操作のサンプルプログラム
 */

/**
 * @page PageSampleFsSdCardForDebug デバッグ用途の SD カードのファイルシステム操作
 * @tableofcontents
 *
 * @brief
 *  デバッグ用途の SD カードのファイルシステム操作のサンプルプログラムの解説です。
 *
 * @section PageSampleFsSdCardForDebug_SectionBrief 概要
 *  ここでは、デバッグ用途の SD カードファイルシステムを使ったサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsSdCardForDebug_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsSdCardForDebug Samples/Sources/Applications/FsSdCardForDebug @endlink 以下にあります。
 *
 * @section PageSampleFsSdCardForDebug_SectionNecessaryEnvironment 必要な環境
 *  NX32, NX64 プラットフォームでは FAT32 にフォーマットされた SD カードが開発機の SD カードスロットに挿入されていること
 *
 * @section PageSampleFsSdCardForDebug_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsSdCardForDebug_SectionPrecaution 注意事項
 *  本サンプルプログラムは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleFsSdCardForDebug_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsSdCardForDebug_SectionDetail 解説
 *
 * @subsection PageSampleFsSdCardForDebug_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsSdCardForDebug.cpp
 *  @includelineno FsSdCardForDebug.cpp
 *
 * @subsection PageSampleFsSdCardForDebug_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - nn::fs::MountSdCardForDebug 関数を利用した SD カードのマウント
 *  - SD カード上の FAT32 ファイルシステムの操作例
 *
 *  nn::fs::MountSdCardForDebug 関数で SD カードをマウントすることが出来ます。
 *  この関数を利用するためには、nn/fs/fs_Debug.h をインクルードしてください。
 *  この関数はデバッグ用途のみに利用でき、製品では利用できない点にご注意ください。
 *
 *  本サンプルプログラムは NX32, NX64 プラットフォームでは SD カードのルートディレクトリに Sample_FsSdCardForDebug.txt というファイルを作成します。
 *  開発機に FAT32 でフォーマットされた SD カードが挿入されている必要があります。
 *
 *  Win32, x64 プラットフォームでは、実行ディレクトリに sdcard/Sample_FsSdCardForDebug.txt というパスで作成します。
 *
 *  本サンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FsSdCardForDebug_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

extern "C" void nnMain()
{
    nn::Result result;

    NN_LOG("Mount SD card as \"sd:/\"\n");

    // SD カードをマウント名 "sd" でマウントします。
    result = nn::fs::MountSdCardForDebug("sd");
    if( nn::fs::ResultSdCardAccessFailed::Includes(result) )
    {
        // SD カードが挿入されていません。
        // SD カードの挿入を促すメッセージを表示してください。
        NN_LOG("SD card not found.");
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // ファイルが存在しない場合は作成します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, "sd:/Sample_FsSdCardForDebug.txt");
        if( nn::fs::ResultPathNotFound().Includes(result) )
        {
            // 対象ファイルが存在しません。
            // 対象ファイルを作成します。
            NN_LOG("Create \"sd:/Sample_FsSdCardForDebug.txt\"\n");
            result = nn::fs::CreateFile("sd:/Sample_FsSdCardForDebug.txt", 0);
            if( nn::fs::ResultPathNotFound::Includes(result) )
            {
                // 対象ファイルが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if( nn::fs::ResultPathAlreadyExists::Includes(result)
                  || nn::fs::ResultTargetLocked::Includes(result) )
            {
                // 対象ファイルが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // 空き領域が不足しています。
                // SD カードの空き領域を確認してください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ファイル末尾に追記します。
    {
        nn::fs::FileHandle fileHandle;
        NN_LOG("Open \"sd:/Sample_FsSdCardForDebug.txt\"\n");
        result = nn::fs::OpenFile(&fileHandle, "sd:/Sample_FsSdCardForDebug.txt", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        int64_t fileSize = 0;

        // ファイルサイズの取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::GetFileSize(&fileSize, fileHandle);

        char writeBuffer[64];
        int length = nn::util::SNPrintf(writeBuffer, sizeof(writeBuffer), "File access via nn::fs::MountSdCardForDebug().\n");

        NN_LOG("Write to \"sd:/Sample_FsSdCardForDebug.txt\"\n");
        result = nn::fs::WriteFile(fileHandle, fileSize, writeBuffer, length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // 空き領域が不足しています。
            // 書き込みサイズが想定範囲内の値となっているか、SD カードの空き領域は十分かを確認してください。
            NN_ABORT("Usable space not enough.\n");
            return;
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        NN_LOG("Close \"sd:/Sample_FsSdCardForDebug.txt\"\n");
        nn::fs::CloseFile(fileHandle);
    }

    // アンマウントします。
    NN_LOG("Unmount \"sd:/\"\n");
    nn::fs::Unmount("sd");

    return;
}
