﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FriendsSimple.cpp,PageSampleFriendsSimple}
 *
 * @brief
 *  フレンドのサンプルプログラム
 */

/**
 * @page PageSampleFriendsSimple フレンド機能の利用
 * @tableofcontents
 *
 * @brief
 *  フレンドライブラリの基本機能を利用するサンプルプログラムの解説です。
 *
 * @section PageSampleFriendsSimple_SectionBrief 概要
 *  このプログラムは、フレンドライブラリの基本的な機能を紹介するサンプルです。@n
 *  関数リファレンス (nn::friends) も併せて参照してください。
 *
 * @section PageSampleFriendsSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FriendsSimple Samples/Sources/Applications/FriendsSimple @endlink 以下にあります。
 *
 * @section PageSampleFriendsSimple_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleFriendsSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFriendsSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFriendsSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFriendsSimple_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - ユーザーアカウントを Open する。
 *  - フレンドリストを取得する。
 *  - ブロックリストを取得する。
 *  - プロフィールリストを取得する。
 *  - ユーザーアカウントを Close する。
 */

#include <nn/friends.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/account.h>
#include <nn/nifm.h>

// プレゼンスの状態を文字列で取得する。
const char* GetPresenceStatusString(nn::friends::PresenceStatus status) NN_NOEXCEPT
{
    switch (status)
    {
    case nn::friends::PresenceStatus_Offline:
        return "Offline";
    case nn::friends::PresenceStatus_Online:
        return "Online";
    case nn::friends::PresenceStatus_OnlinePlay:
        return "OnlinePlay";
    default:
        return "";
    }
}

// フレンドリストを取得する。
void GetFriendList(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    nn::account::Uid user;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user, handle));

    static nn::friends::Friend friends[nn::friends::FriendCountMax] = {};
    int count;
    nn::friends::FriendFilter filter = {};

    // 同じプレゼンスグループ ID を指定したアプリケーションを遊んでいるフレンドで絞り込みたい場合、以下のフィルターを使用します。
    // filter.presenceStatus = nn::friends::PresenceStatusFilter_OnlineOrOnlinePlay;
    // filter.isSameAppPresenceOnly = true;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, 0, nn::friends::FriendCountMax, filter));

    NN_LOG("FriendCount = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("Friends[%3d]\n", i);
        NN_LOG("    - id = %016llx\n", friends[i].GetAccountId().id);
        NN_LOG("    - name = %s\n", friends[i].GetNickname().name);
        NN_LOG("    - status = %s\n", GetPresenceStatusString(friends[i].GetPresence().GetStatus()));
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

// ブロックリストを取得する。
void GetBlockedUserList(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    nn::account::Uid user;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user, handle));

    static nn::account::NetworkServiceAccountId accountIds[nn::friends::BlockedUserCountMax] = {};
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, accountIds, 0, nn::friends::BlockedUserCountMax));

    NN_LOG("BlockedUserCount = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------------------------------------\n");
        NN_LOG("BlockedUser[%3d]\n", i);
        NN_LOG("    - id = %016llx\n", accountIds[i].id);
    }

    NN_LOG("--------------------------------------------------------------------------------\n");
}

// プロフィールリストを取得する。
void GetProfileList(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    nn::account::Uid user;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user, handle));

    nn::friends::Profile profile = {};
    nn::account::NetworkServiceAccountId accountId = {};
    nn::friends::AsyncContext async;

    // サンプルでは、自身のプロフィールを取得します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountId(&accountId, handle));

    // 通信が発生する API を呼び出す場合、事前にネットワークに接続しておく必要があります。
    NN_LOG("SubmitNetworkRequest ...\n");

    nn::nifm::SubmitNetworkRequest();

    while (nn::nifm::IsNetworkRequestOnHold())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    NN_LOG("SubmitNetworkRequest done!\n");

    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetProfileList(&async, &profile, &accountId, 1));

    NN_LOG("GetProfileList ...\n");

    // 非同期処理を待つ方法は 2 通りあります。
    // 非同期処理の完了フラグを見る方法と、イベントを待機する方法です。

#if 1

    bool done = false;

    while (async.HasDone(&done).IsSuccess() && !done)
    {
        // 非同期処理を中断する場合、 AsyncContext::Cancel を呼び出してください。
        // AsyncContext::Cancel 呼び出し後も、非同期処理が完了するまで待機する必要があります。
        // async.Cancel();

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

#else

    nn::os::SystemEvent completionEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

#endif

    NN_LOG("GetProfileList done!\n");

    // 非同期処理の結果は、 AsyncContext::GetResult で取得します。
    // GetProfileList の戻り値でハンドリングするのではないことに注意してください。
    if (async.GetResult().IsSuccess())
    {
        NN_LOG("GetProfileList succeeded!\n");

        // GetProfileList は指定したネットワークサービスアカウント ID が存在するユーザーか否かに関わらず成功します。
        // 正しくプロフィールが取得できたかどうかは、 Profile::IsValid で確認することができます。
        if (profile.IsValid())
        {
            NN_LOG("Profile[self]\n");
            NN_LOG("    - id = %016llx\n", profile.GetAccountId().id);
            NN_LOG("    - name = %s\n", profile.GetNickname().name);
        }
    }
    else
    {
        NN_LOG("GetProfileList failed. e = %08x\n", async.GetResult().GetInnerValueForDebug());
    }
}

extern "C" void nnMain()
{
    nn::account::Initialize();
    nn::nifm::Initialize();

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 1)
    {
        // サンプルでは、実装簡略化のため先頭のユーザーを選択したと仮定します。
        // 実際は、プレイヤー選択で選ばれたユーザーを指定してください。
        nn::account::UserHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, users[0]));

        GetFriendList(handle);
        GetBlockedUserList(handle);
        GetProfileList(handle);

        nn::account::CloseUser(handle);
    }
    else
    {
        NN_LOG("[sample] Warning: Please create user account.\n");
    }

    NN_LOG("[sample] End.\n");
}
