﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FriendsProfileImage.cpp,PageSampleFriendsProfileImage}
 *
 * @brief
 *  プロフィール画像ダウンロードのサンプルプログラム
 */

/**
 * @page PageSampleFriendsProfileImage プロフィール画像のダウンロード
 * @tableofcontents
 *
 * @brief
 *  プロフィール画像をダウンロードするサンプルプログラムの解説です。
 *
 * @section PageSampleFriendsProfileImage_SectionBrief 概要
 *  このプログラムは、curl ライブラリを使用してプロフィール画像のダウンロード処理を行うサンプルです。@n
 *  関数リファレンス (nn::friends) も併せて参照してください。
 *
 * @section PageSampleFriendsProfileImage_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FriendsProfileImage Samples/Sources/Applications/FriendsProfileImage @endlink 以下にあります。
 *
 * @section PageSampleFriendsProfileImage_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleFriendsProfileImage_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFriendsProfileImage_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFriendsProfileImage_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFriendsProfileImage_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - ユーザーアカウントを Open する。
 *  - 自分のプロフィール画像を取得する。
 *  - ユーザーアカウントを Close する。
 */

#include <nn/friends.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/account.h>
#include <nn/nifm.h>
#include <nn/socket.h>
#include <nn/image/image_JpegDecoder.h>

#include "FileDownloader.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

// ダウンロードの完了コールバックです。
void DownloadCompletionCallback(uint64_t requestId, CURLcode curlCode, long statusCode,
    void* pBuffer, size_t downloadedSize, void* pParam)
{
    NN_UNUSED(pParam);

    NN_LOG("Download request was completed. id = %llu\n", requestId);

    if (curlCode != CURLE_OK || statusCode / 100 != 2)
    {
        NN_LOG("Download failed. curlCode = %d, statusCode = %d\n", curlCode, statusCode);
        return;
    }

    NN_LOG("Size = %zu\n", downloadedSize);

    // コールバック内で重い処理を行うとパフォーマンスに影響が出るため、別スレッドでデコードすることを推奨します。
    // 本サンプルは、記述の簡潔さを優先してここでデコードを行います。

    nn::image::JpegDecoder decoder;
    decoder.SetImageData(pBuffer, downloadedSize);

    nn::image::JpegStatus status = decoder.Analyze();

    if (status != nn::image::JpegStatus_Ok)
    {
        NN_LOG("Jpeg analyzing failed. status = %d\n", status);
        return;
    }

    nn::image::Dimension dimension = decoder.GetAnalyzedDimension();

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("JpegInfo:\n");
    NN_LOG("    dimension.w = %d\n", dimension.width);
    NN_LOG("    dimension.h = %d\n", dimension.height);
    NN_LOG("--------------------------------------------------------------------------------\n");
}

// プロフィール画像（複数のサイズ）をダウンロードします。
void DownloadProfileImages(nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_LOG("SubmitNetworkRequestAndWait ...\n");

    nn::nifm::SubmitNetworkRequestAndWait();

    NN_LOG("SubmitNetworkRequestAndWait done!\n");

    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    nn::friends::Profile profile = {};

    // nn::friends::GetProfileList には呼び出し頻度に関するガイドラインが設けられています。
    // 過度な呼び出しが行われないよう注意してください。
    nn::friends::AsyncContext async;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetProfileList(&async, &profile, &accountId, 1));

    nn::os::SystemEvent completionEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    if (async.GetResult().IsFailure())
    {
        NN_LOG("GetProfileList failed. e = %08x\n", async.GetResult().GetInnerValueForDebug());
        return;
    }

    nn::friends::Url imageUrls[3] = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(profile.GetProfileImageUrl(&imageUrls[0], nn::friends::ImageSize_Standard));
    NN_ABORT_UNLESS_RESULT_SUCCESS(profile.GetProfileImageUrl(&imageUrls[1], nn::friends::ImageSize_64x64));
    NN_ABORT_UNLESS_RESULT_SUCCESS(profile.GetProfileImageUrl(&imageUrls[2], nn::friends::ImageSize_128x128));

    if (imageUrls[0].value[0] == '\0')
    {
        // プロフィール画像がアップロードされていない場合があります。
        NN_LOG("The image url is empty.\n");
        return;
    }

    // プロフィール画像の最大サイズは nn::friends::ProfileImageSizeMax です。
    static nn::Bit8 buffer[NN_ARRAY_SIZE(imageUrls)][nn::friends::ProfileImageSizeMax];

    nn::os::Tick startTick = nn::os::GetSystemTick();

    // プロフィール画像を並列でダウンロードします。
    // リクエスト順にダウンロードが完了することは保証されません。
    // リクエスト順にダウンロードしたいのであれば、最大同時実行数に 1 を指定します。
    FileDownloader downloader(4);

    downloader.SetCompletionCallback(DownloadCompletionCallback, nullptr);

    for (int i = 0; i < NN_ARRAY_SIZE(imageUrls); i++)
    {
        NN_LOG("URL = %s\n", imageUrls[i].value);

        uint64_t requestId;
        downloader.AddRequest(&requestId, imageUrls[i].value, buffer[i], sizeof (buffer[i]));

        NN_LOG("Request id = %llu\n", requestId);
    }

    downloader.Execute();

    nn::os::Tick endTick = nn::os::GetSystemTick();

    NN_LOG("Time = %lld ms\n", (endTick - startTick).ToTimeSpan().GetMilliSeconds());
}

extern "C" void nnMain()
{
    nn::account::Initialize();
    nn::nifm::Initialize();
    nn::socket::Initialize(g_SocketConfigWithMemory);

    curl_global_init(CURL_GLOBAL_DEFAULT);

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 1)
    {
        // サンプルでは、実装簡略化のため先頭のユーザーを選択したと仮定します。
        // 実際は、プレイヤー選択で選ばれたユーザーを指定してください。
        nn::account::UserHandle handle = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, users[0]));

        nn::account::NetworkServiceAccountId accountId = {};

        if (nn::account::GetNetworkServiceAccountId(&accountId, handle).IsSuccess())
        {
            DownloadProfileImages(accountId);
        }
        else
        {
            NN_LOG("[sample] Warning: Please link the available network service account to user account.\n");
        }

        nn::account::CloseUser(handle);
    }
    else
    {
        NN_LOG("[sample] Warning: Please create user account.\n");
    }

    NN_LOG("[sample] End.\n");
}
