﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FriendsPresence.cpp,PageSampleFriendsPresence}
 *
 * @brief
 *  ユーザープレゼンス更新とフレンドプレゼンス取得のサンプルプログラム
 */

/**
 * @page PageSampleFriendsPresence プレゼンス
 * @tableofcontents
 *
 * @brief
 *  ユーザープレゼンスの更新とフレンドプレゼンスの取得を行うサンプルプログラムの解説です。
 *
 * @section PageSampleFriendsPresence_SectionBrief 概要
 *  このプログラムは、ユーザープレゼンスの更新とフレンドプレゼンスの取得を行い、プレゼンスの状態変化を見るためのサンプルです。@n
 *  関数リファレンス (nn::friends) も併せて参照してください。
 *
 * @section PageSampleFriendsPresence_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FriendsPresence Samples/Sources/Applications/FriendsPresence @endlink 以下にあります。
 *
 * @section PageSampleFriendsPresence_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleFriendsPresence_SectionHowToOperate 操作方法
 *  Npad または DebugPad を使用します。
 *  - A: ユーザーアカウントを Open する。
 *  - B: ユーザーアカウントを Close する。
 *  - X: ユーザープレゼンスを更新する。
 *  - Y: フレンドプレゼンスを取得する。
 *  - L: オンラインプレイの開始を宣言する。
 *  - R: オンラインプレイの終了を宣言する。
 *  - ↑: ログを上にスクロールする。
 *  - ↓: ログを下にスクロールする。
 *  - ←: ログの先頭にスクロールする。
 *  - →: ログの末尾にスクロールする。
 *  - -: ログをクリアする。
 *  - +: 終了する。
 *
 * @section PageSampleFriendsPresence_SectionPrecaution 注意事項
 *  フレンドプレゼンスを確認したい場合、事前にフレンドを作成してください。
 *
 * @section PageSampleFriendsPresence_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFriendsPresence_SectionDetail 解説
 *  Npad または DebugPad を利用してユーザーの Open/Close、ユーザープレゼンスの更新、フレンドプレゼンスの取得を行います。@n
 *  Npad はフルキー操作と携帯機操作に対応しています。
 */

#include <nn/friends.h>

#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/account.h>
#include <nn/util/util_FormatString.h>

#include "Console.h"
#include "Input.h"

#define SAMPLE_CONSOLE_INDEX 1 // NOLINT(preprocessor/const)

#define SAMPLE_LOG(...) console::Printf(SAMPLE_CONSOLE_INDEX, __VA_ARGS__)

// 使用方法を表示します。
void ShowUsage(bool isExitOnly = false) NN_NOEXCEPT
{
    console::Printf(0, "------------------------------------\n");
    console::Printf(0, "USAGE\n");
    console::Printf(0, "------------------------------------\n");

    if (!isExitOnly)
    {
        console::Printf(0, "A: Open user account\n");
        console::Printf(0, "B: Close user account\n");
        console::Printf(0, "X: Update user presence\n");
        console::Printf(0, "Y: Get friend presence\n");
        console::Printf(0, "L: Declare open OnlinePlay session\n");
        console::Printf(0, "R: Declare close OnlinePlay session\n");
        console::Printf(0, "↑: Scroll (Up)\n");
        console::Printf(0, "↓: Scroll (Down)\n");
        console::Printf(0, "←: Scroll (To Top)\n");
        console::Printf(0, "→: Scroll (To Bottom)\n");
        console::Printf(0, "-: Clear\n");
    }

    console::Printf(0, "+: Exit\n");
    console::Printf(0, "------------------------------------\n");
}

// プレゼンスの状態を文字列で取得する。
const char* GetPresenceStatusString(nn::friends::PresenceStatus status) NN_NOEXCEPT
{
    switch (status)
    {
    case nn::friends::PresenceStatus_Offline:
        return "Offline";
    case nn::friends::PresenceStatus_Online:
        return "\033[32m" "Online" "\033[m";
    case nn::friends::PresenceStatus_OnlinePlay:
        return "\033[36m" "OnlinePlay" "\033[m";
    default:
        return "";
    }
}

// フレンドリストを表示する。
void ShowFriendList(const nn::account::Uid& user) NN_NOEXCEPT
{
    static nn::friends::Friend friends[nn::friends::FriendCountMax] = {};
    int count;
    nn::friends::FriendFilter filter = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, user, 0, nn::friends::FriendCountMax, filter));

    for (int i = 0; i < count; i++)
    {
        SAMPLE_LOG("--------------------------------------------------------------------------------\n");
        SAMPLE_LOG("Friends[%3d]\n", i);
        SAMPLE_LOG("    - id = %016llx\n", friends[i].GetAccountId().id);
        SAMPLE_LOG("    - name = %s\n", friends[i].GetNickname().name);
        SAMPLE_LOG("    - status = %s\n", GetPresenceStatusString(friends[i].GetPresence().GetStatus()));
        SAMPLE_LOG("    - description = %s\n", friends[i].GetPresence().GetDescription());
        SAMPLE_LOG("    - isSamePresenceGroupApplication = %s\n", friends[i].GetPresence().IsSamePresenceGroupApplication() ? "true" : "false");
    }
    if (count == 0)
    {
        SAMPLE_LOG("You have no friends.\n");
    }
    else
    {
        SAMPLE_LOG("--------------------------------------------------------------------------------\n");
    }
}

// フレンドを表示する。
void ShowFriend(const nn::account::Uid& user, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    nn::friends::Friend f = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::UpdateFriendInfo(&f, user, &accountId, 1));

    if (f.IsValid())
    {
        SAMPLE_LOG("--------------------------------------------------------------------------------\n");
        SAMPLE_LOG("Friend\n");
        SAMPLE_LOG("    - id = %016llx\n", f.GetAccountId().id);
        SAMPLE_LOG("    - name = %s\n", f.GetNickname().name);
        SAMPLE_LOG("    - status = %s\n", GetPresenceStatusString(f.GetPresence().GetStatus()));
        SAMPLE_LOG("    - description = %s\n", f.GetPresence().GetDescription());
        SAMPLE_LOG("    - isSamePresenceGroupApplication = %s\n", f.GetPresence().IsSamePresenceGroupApplication() ? "true" : "false");
        SAMPLE_LOG("--------------------------------------------------------------------------------\n");
    }
}

// メインループ処理を行う。
void MainLoop(const nn::account::Uid& user) NN_NOEXCEPT
{
    ShowUsage();

    nn::account::UserHandle handle = {};
    bool isOpened = false;

    nn::friends::NotificationQueue notificationQueue;

    nn::hid::NpadButtonSet prevButtons = {};

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::NpadButtonSet buttons = {};

        input::GetButtons(&buttons, 0);

        nn::hid::NpadButtonSet trigger = (~prevButtons) & buttons;
        prevButtons = buttons;

        if (trigger.Test<nn::hid::NpadButton::A>())
        {
            if (!isOpened)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, user));
                isOpened = true;

                NN_ABORT_UNLESS_RESULT_SUCCESS(notificationQueue.Initialize(user));

                SAMPLE_LOG("The user was opened.\n");
            }
            else
            {
                SAMPLE_LOG("The user is already opened.\n");
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::B>())
        {
            if (isOpened)
            {
                notificationQueue.Finalize();

                nn::account::CloseUser(handle);
                isOpened = false;

                SAMPLE_LOG("The user was closed.\n");
            }
            else
            {
                SAMPLE_LOG("The user is already closed.\n");
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::X>())
        {
            if (isOpened)
            {
                static int counter = 0;

                // サンプルでは、ユーザープレゼンスの更新毎にゲームモード説明文のカウンターを更新しています。
                char description[64] = {};
                nn::util::SNPrintf(description, sizeof (description), "Description(%d)", counter++);

                nn::friends::UserPresence presence;
                NN_ABORT_UNLESS_RESULT_SUCCESS(presence.Initialize(user));

                presence.SetDescription(description);
                NN_ABORT_UNLESS_RESULT_SUCCESS(presence.Commit());

                SAMPLE_LOG("The user presence was updated. (description = %s)\n", description);
            }
            else
            {
                SAMPLE_LOG("The user has not been opened.\n");
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::Y>())
        {
            if (isOpened)
            {
                ShowFriendList(user);
            }
            else
            {
                SAMPLE_LOG("The user has not been opened.\n");
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::L>())
        {
            if (isOpened)
            {
                // プレゼンスの更新とオンラインプレイの開始宣言を確実に同時に行いたい場合、 UserPresence::DeclareOpenOnlinePlaySession を使用します。
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::DeclareOpenOnlinePlaySession(user));
                SAMPLE_LOG("Declared.\n");
            }
            else
            {
                SAMPLE_LOG("The user has not been opened.\n");
            }
        }
        else if (trigger.Test<nn::hid::NpadButton::R>())
        {
            if (isOpened)
            {
                // プレゼンスの更新とオンラインプレイの終了宣言を確実に同時に行いたい場合、 UserPresence::DeclareCloseOnlinePlaySession を使用します。
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::DeclareCloseOnlinePlaySession(user));
                SAMPLE_LOG("Declared.\n");
            }
            else
            {
                SAMPLE_LOG("The user has not been opened.\n");
            }
        }
        else if (buttons.Test<nn::hid::NpadButton::Up>())
        {
            console::Scroll(SAMPLE_CONSOLE_INDEX, 1);
        }
        else if (buttons.Test<nn::hid::NpadButton::Down>())
        {
            console::Scroll(SAMPLE_CONSOLE_INDEX, -1);
        }
        else if (trigger.Test<nn::hid::NpadButton::Left>())
        {
            console::SetScrollPosition(SAMPLE_CONSOLE_INDEX, INT_MAX);
        }
        else if (trigger.Test<nn::hid::NpadButton::Right>())
        {
            console::SetScrollPosition(SAMPLE_CONSOLE_INDEX, 0);
        }
        else if (trigger.Test<nn::hid::NpadButton::Minus>())
        {
            console::Clear(SAMPLE_CONSOLE_INDEX);
        }
        else if (trigger.Test<nn::hid::NpadButton::Plus>())
        {
            break;
        }

        if (isOpened && notificationQueue.GetSystemEvent()->TryWait())
        {
            notificationQueue.GetSystemEvent()->Clear();

            nn::friends::NotificationInfo info = {};

            // 通知キューが空になるまで通知を取得します。
            while (notificationQueue.Pop(&info).IsSuccess())
            {
                switch (info.type)
                {
                case nn::friends::NotificationType_FriendListUpdated:
                    {
                        SAMPLE_LOG("'FriendListUpdated' event was notified.\n");

                        ShowFriendList(notificationQueue.GetUid());
                    }
                    break;
                case nn::friends::NotificationType_FriendPresenceUpdated:
                    {
                        SAMPLE_LOG("'FriendPresenceUpdated' event was notified.\n");

                        ShowFriend(notificationQueue.GetUid(), info.accountId);
                    }
                    break;
                default:
                    // ハンドリングしていない通知は無視してください。
                    break;
                }
            }
        }

        console::ProcessFrame();
    }

    if (isOpened)
    {
        nn::account::CloseUser(handle);
    }
} // NOLINT(impl/function_size)

// 待機用ループ処理を行う。
void WaitingLoop() NN_NOEXCEPT
{
    ShowUsage(true);

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::NpadButtonSet buttons = {};

        input::GetButtons(&buttons, 0);

        if (buttons.Test<nn::hid::NpadButton::Plus>())
        {
            break;
        }

        console::ProcessFrame();
    }
}

extern "C" void nnMain()
{
    console::Initialize();

    // USAGE 用
    console::ConsoleCreateParam cp0 = console::GetDefaultConsoleCreateParam();
    cp0.viewWidth = 300;
    cp0.viewHeight = 320;
    cp0.isTeeEnabled = false;

    // ログ用
    console::ConsoleCreateParam cp1 = console::GetDefaultConsoleCreateParam();
    cp1.bufferHeight = 10000;
    cp1.viewX = cp0.viewX + cp0.viewWidth + 10;
    cp1.viewWidth = console::DefaultScreenWidth - cp1.viewX;
    cp1.isTimestampEnabled = true;

    console::CreateConsole(0, cp0);
    console::CreateConsole(1, cp1);

    SAMPLE_LOG("Start.\n");

    input::Initialize();

    nn::account::Initialize();

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 1)
    {
        MainLoop(users[0]);
    }
    else
    {
        SAMPLE_LOG("\033[31m" "Warning: Please create user account." "\033[m" "\n");
        WaitingLoop();
    }

    SAMPLE_LOG("End.\n");

    console::Finalize();
}
