﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FriendsFriendRequest.cpp,PageSampleFriendsFriendRequest}
 *
 * @brief
 *  フレンド申請のサンプルプログラム
 */

/**
 * @page PageSampleFriendsFriendRequest フレンド申請
 * @tableofcontents
 *
 * @brief
 *  フレンド申請を行うサンプルプログラムの解説です。
 *
 * @section PageSampleFriendsFriendRequest_SectionBrief 概要
 *  このプログラムは、フレンド申請の手順を紹介するサンプルです。@n
 *  関数リファレンス (nn::friends) も併せて参照してください。
 *
 * @section PageSampleFriendsFriendRequest_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FriendsFriendRequest Samples/Sources/Applications/FriendsFriendRequest @endlink 以下にあります。
 *
 * @section PageSampleFriendsFriendRequest_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを 2 つ以上、作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleFriendsFriendRequest_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFriendsFriendRequest_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFriendsFriendRequest_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFriendsFriendRequest_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - インターネット通信経由でフレンド申請を行う。
 *  - ローカル通信経由でフレンド申請を行う。（対面フレンド申請）
 */

#include <nn/friends/friends_ApiFriendRequest.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/account.h>
#include <nn/nifm.h>

// インターネット通信経由でフレンド申請を行う。
void SendFriendRequestViaInternetCommunication(const nn::account::UserHandle& handle0, const nn::account::UserHandle& handle1) NN_NOEXCEPT
{
    NN_LOG("----------------------------------------------------------------------\n");
    NN_LOG("SendFriendRequestViaInternetCommunication\n");
    NN_LOG("----------------------------------------------------------------------\n");

    // 通信が発生する API を呼び出す場合、事前にネットワークに接続しておく必要があります。
    NN_LOG("SubmitNetworkRequest ...\n");

    nn::nifm::SubmitNetworkRequest();

    while (nn::nifm::IsNetworkRequestOnHold())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    NN_LOG("SubmitNetworkRequest done!\n");

    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Network is not available.\n");
        return;
    }

    nn::account::Uid user0 = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user0, handle0));

    nn::account::NetworkServiceAccountId accountId1 = {};
    nn::Result result = nn::account::GetNetworkServiceAccountId(&accountId1, handle1);

    if (result.IsFailure())
    {
        NN_LOG("Please link the available network service account to user account (1). e = %08x\n", result.GetInnerValueForDebug());
        return;
    }

    // いっしょに遊んだ時のアプリ内表示名です。
    nn::friends::InAppScreenName inAppScreenName0 = {"InAppScreenName[0]", nn::oe::GetDesiredLanguage()};
    nn::friends::InAppScreenName inAppScreenName1 = {"InAppScreenName[1]", nn::oe::GetDesiredLanguage()};

    nn::friends::AsyncContext async;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SendFriendRequest(&async, user0, accountId1, inAppScreenName1, inAppScreenName0));

    NN_LOG("SendFriendRequest ...\n");

    bool done = false;

    while (async.HasDone(&done).IsSuccess() && !done)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    NN_LOG("SendFriendRequest done!\n");

    if (async.GetResult().IsSuccess())
    {
        NN_LOG("SendFriendRequest succeeded!\n");
    }
    else
    {
        NN_LOG("SendFriendRequest failed. e = %08x\n", async.GetResult().GetInnerValueForDebug());
    }
}

// ローカル通信経由でフレンド申請を行う。（対面フレンド申請）
void SendFriendRequestViaLocalCommunication(const nn::account::UserHandle& handle0, const nn::account::UserHandle& handle1) NN_NOEXCEPT
{
    NN_LOG("----------------------------------------------------------------------\n");
    NN_LOG("SendFriendRequestViaLocalCommunication\n");
    NN_LOG("----------------------------------------------------------------------\n");

    nn::account::Uid user0 = {};
    nn::account::Uid user1 = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user0, handle0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&user1, handle1));

    // 対面フレンド申請の登録鍵を取得します。
    // ネットワークサービスアカウントを紐付けていない場合、失敗します。
    nn::friends::FacedFriendRequestRegistrationKey key0 = {};
    nn::friends::FacedFriendRequestRegistrationKey key1 = {};

    nn::Result result = nn::friends::GetFacedFriendRequestRegistrationKey(&key0, user0);

    if (result.IsFailure())
    {
        NN_LOG("Please link the available network service account to user account (0). e = %08x\n", result.GetInnerValueForDebug());
        return;
    }

    result = nn::friends::GetFacedFriendRequestRegistrationKey(&key1, user1);

    if (result.IsFailure())
    {
        NN_LOG("Please link the available network service account to user account (1). e = %08x\n", result.GetInnerValueForDebug());
        return;
    }

    // 相手のフレンド申請リストに表示されるプロフィール情報を取得します。
    nn::account::Nickname nickname0 = {};
    static nn::Bit8 image0[nn::account::ProfileImageBytesMax] = {};
    size_t imageSize0 = 0;

    nn::account::Nickname nickname1 = {};
    static nn::Bit8 image1[nn::account::ProfileImageBytesMax] = {};
    size_t imageSize1 = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname0, user0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::LoadProfileImage(&imageSize0, image0, sizeof (image0), user0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname1, user1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::LoadProfileImage(&imageSize1, image1, sizeof (image1), user1));

    // いっしょに遊んだ時のアプリ内表示名です。
    nn::friends::InAppScreenName inAppScreenName0 = {"InAppScreenName[0]", nn::oe::GetDesiredLanguage()};
    nn::friends::InAppScreenName inAppScreenName1 = {"InAppScreenName[1]", nn::oe::GetDesiredLanguage()};

    // ローカル通信経由で対面フレンド申請に必要なプロフィールを交換します。
    // ニックネーム・プロフィール画像は、アプリケーション独自のものではなく、必ず nn::account から取得したものを指定する必要があります。
    // サンプルでは 2 ユーザーとも同一デバイス上に存在するため、交換処理は特に行いません。
    // ExchangeProfile(key0, nickname0, image0, imageSize0, inAppScreenName0, key1, nickname1, image1, imageSize1, inAppScreenName1);

    // 対面フレンド申請を追加します。
    // 対面フレンド申請はシステムのセーブデータに保存され、インターネット接続後にサーバーに送信されます。
    result = nn::friends::AddFacedFriendRequest(user0, key1, nickname1, image1, imageSize1, inAppScreenName1, inAppScreenName0);

    if (result.IsSuccess())
    {
        NN_LOG("AddFacedFriendRequest(0) succeeded!\n");
    }
    else
    {
        NN_LOG("AddFacedFriendRequest(0) failed. e = %08x\n", result.GetInnerValueForDebug());
    }

    // 「マイページ」では、お互いが相手を指定して同時に対面フレンド申請の追加を行っています。
    // アプリケーションは、インターネット通信経由と同じく一方のユーザーのみが対面フレンド申請を追加しても問題ありません。
    result = nn::friends::AddFacedFriendRequest(user1, key0, nickname0, image0, imageSize0, inAppScreenName0, inAppScreenName1);

    if (result.IsSuccess())
    {
        NN_LOG("AddFacedFriendRequest(1) succeeded!\n");
    }
    else
    {
        NN_LOG("AddFacedFriendRequest(1) failed. e = %08x\n", result.GetInnerValueForDebug());
    }
}

extern "C" void nnMain()
{
    nn::account::Initialize();
    nn::nifm::Initialize();

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 2)
    {
        // サンプルでは、0 番目のユーザーが 1 番目のユーザーにフレンド申請を行います。
        // アプリケーションに組み込む際、フレンド申請を送信する前にユーザーの同意を得る必要があります。

        nn::account::UserHandle handle0 = {};
        nn::account::UserHandle handle1 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle0, users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle1, users[1]));

        SendFriendRequestViaInternetCommunication(handle0, handle1);
        SendFriendRequestViaLocalCommunication(handle0, handle1);

        nn::account::CloseUser(handle0);
        nn::account::CloseUser(handle1);
    }
    else
    {
        NN_LOG("Warning: Please create user accounts.\n");
    }

    NN_LOG("End.\n");
}
