﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fgm.h>
#include <nn/nn_Log.h>

#include "fgm_Worker.h"

namespace nn     {
namespace fgm    {
namespace sample {

Worker g_Worker1 NN_ALIGNAS(4096);
Worker g_Worker2 NN_ALIGNAS(4096);
Worker g_Worker3 NN_ALIGNAS(4096);
Worker g_Worker4 NN_ALIGNAS(4096);

// Adds basic min/max constraint on the resource.
// Based on the priorities of other requests
// this may or may not change the
// actual state of the resource.
void RequestSimple()
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting min = rand();
    nn::fgm::Setting max = nn::fgm::Setting_Max; // Highest possible for this resource
    nn::fgm::Setting actual;

    if ((result = request.Initialize(nn::fgm::Module_Test, Priority_Default)).IsFailure() ||
        (result = request.SetAndWait(min, max)).IsFailure() ||
        (result = request.Get(&actual)).IsFailure())
    {
        NN_LOG("failed to request new range, error %d:%d\n", result.GetModule(), result.GetDescription());
    }
    else
    {
        NN_LOG("set state to %u - %u\n", min, max);
    }

    // keep this request active for a couple of seconds
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    // removes this min/max constraint
    request.Finalize();
}

// Adds basic min/max constraint on the resource,
// waits for update notifications.
void RequestWithNotifications()
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting currentSetting;
    nn::fgm::Setting newSetting;
    nn::fgm::Setting min = rand();
    nn::fgm::Setting max = nn::fgm::Setting_Max; // Highest possible for this resource

    if ((result = request.Initialize(nn::fgm::Module_Test, Priority_Default)).IsFailure() ||
        (result = request.Get(&currentSetting)).IsFailure() ||
        (result = request.Set(min, max)).IsFailure())
    {
        NN_LOG("failed to request new range, error %d:%d\n", result.GetModule(), result.GetDescription());
        goto out;
    }

    while ((result = request.WaitWithTimeout(nn::TimeSpan::FromSeconds(5))).IsSuccess() &&
           (result = request.Get(&newSetting)).IsSuccess())
    {
        if (newSetting != currentSetting)
        {
            NN_LOG("state change detected, from %u to %u\n", currentSetting, newSetting);
            currentSetting = newSetting;
        }
    }

out:
    request.Finalize();
}

}}}

extern "C" void nnMain()
{
    nn::fgm::sample::g_Worker1.Initialize(nn::fgm::sample::RequestWithNotifications, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker2.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker3.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker4.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);

    nn::fgm::sample::g_Worker1.Finalize();
    nn::fgm::sample::g_Worker2.Finalize();
    nn::fgm::sample::g_Worker3.Finalize();
    nn::fgm::sample::g_Worker4.Finalize();
}
