﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FgmMultiModules.cpp,PageSampleFgmMultiModules}
 *
 * @brief
 *  クロックレート変更のサンプルプログラム
 */

/**
 * @page PageSampleFgmMultiModules クロックレート変更サンプルプログラム
 * @tableofcontents
 *
 * @brief
 *  クロックレート調停ライブラリを用いたクロックレート変更操作の解説です。
 *
 * @section PageSampleFgmMultiModules_SectionBrief 概要
 *  ここでは、クロックレート調停ライブラリを用いて複数のモジュールの性能を切り替える方法を解説します。
 *
 *  クロックレート調停ライブラリの使い方については、
 *  @ref nn::fgm "クロックレート調停ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFgmMultiModules_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FgmMultiModules
 *  Samples/Sources/Applications/FgmMultiModules @endlink 以下にあります。
 *
 * @section PageSampleFgmMultiModules_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleFgmMultiModules_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleFgmMultiModules_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleFgmMultiModules_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFgmMultiModules_SectionDetail 解説
 *
 * @subsection PageSampleFgmMultiModules_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FgmMultiModules.cpp
 *  @includelineno FgmMultiModules.cpp
 *
 * @subsection PageSampleFgmMultiModules_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムはプログラム内で CPU, GPU, メモリのクロックレートを一時的に変更し、元に戻します
 *  - CPU のクロックレートは 1224.00 MHz に再設定します。
 *  - GPU のクロックレートは 460.80 MHz に再設定します。
 *  - メモリのクロックレートは 800.00 MHz に再設定します。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - クロックレート設定要求の為の nn::fgm::Request オブジェクトを変更する対象の数と同じ数用意します。
 *  - CPU のクロックレートを変更するためクロックレート変更関数 CreateRequestSession を呼びます。
 *      - CPU モジュールを対象にして基本優先度で nn::fgm::Request オブジェクトを初期化します。
 *      - CPU のクロックレートを設定し要求が受理されるのを待ちます。（API の設計の都合上、同じクロックレートを 2 引数に指定する必要があります）
 *      - クロックレートの設定後の値を取得し望む値と比較します。
 *  - GPU のクロックレートを変更するためクロックレート変更関数 CreateRequestSession を呼びます。
 *  - メモリのクロックレートを変更するためクロックレート変更関数 CreateRequestSession を呼びます。
 *  - CPU のクロックレートを再変更するためクロックレート変更関数 CreateRequestSession を呼びます。
 *  - CPU のクロックレートの再変更を破棄します。
 *  - CPU のクロックレートを変更を破棄します。
 *  - GPU のクロックレートを変更を破棄します。
 *  - メモリ のクロックレートを変更を破棄します。
 *
 *  本プログラムでは明示的に変更を取り下げていますが、CPU のクロックレートの再変更で示したように nn::fgm::Request オブジェクトが破棄されるタイミングでも変更は破棄されます。
 *  変更を維持するためには nn::fgm::Request オブジェクトを破棄しないようにしてください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FgmMultiModules_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/fgm/fgm.h>
#include <nn/gfx.h>
#include <nv/nv_MemoryManagement.h>

//-----------------------------------------------------------------------------

namespace {

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

nn::gfx::Device g_Device;

//
// グラフィックデバイス初期化関数。
//
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    g_Device.Initialize(info);
}

void* Allocate(size_t size, size_t alignment, void*)
{
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void Free(void* addr, void*)
{
    free(addr);
}

void* Reallocate(void* addr, size_t newSize, void*)
{
    return realloc(addr, newSize);
}

//
// クロックレート変更関数
//
void CreateRequestSession(nn::fgm::Request& request, nn::fgm::Module module, nn::fgm::Setting setting)
{
    nn::fgm::Setting actual = 0;

    // クロックレート設定要求を初期化します。nn::fgm::Priority_Default のみ使用可です。
    NN_ABORT_UNLESS_RESULT_SUCCESS(request.Initialize(module, nn::fgm::Priority_Default));

    // クロックレートの設定要求をして、要求が受理されるまで待ちます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(request.SetAndWait(setting, setting));

    // 実際のクロックレートを取得します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(request.Get(&actual));

    if ( actual != setting )
    {
        NN_LOG("Failed to set clock rate. requested one is %u. actual one is %u.\n", setting, actual);
    }
    else
    {
        NN_LOG("Set clock rate to %u Hz\n", actual);
    }
}

//
// クロックレート表示関数
//
void PrintSetting(nn::fgm::Request& request)
{
    nn::fgm::Setting actual = 0;

    // 実際のクロックレートを取得します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(request.Get(&actual));

    NN_LOG("Now clock rate is %u Hz\n", actual);
}

//
// クロックレート変更破棄関数
//
void DeleteRequestSession(nn::fgm::Request& request)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(request.Finalize());
}

} // namespace

//
//  メイン関数
//
extern "C" void nnMain()
{
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    // GPU の電源を入れることを目的としてグラフィックスシステムの初期化を行います。
    nn::gfx::Initialize();
    InitializeDevice();

    NN_LOG("FgmMultiModules starts.\n");

    nn::fgm::Request requestCpu;
    nn::fgm::Request requestGpu;
    nn::fgm::Request requestEmc;

    // CPU のクロックレートを 1224.00 Mhz に変更します。
    CreateRequestSession(requestCpu, nn::fgm::Module_Cpu, 1224000000);

    // GPU のクロックレートを 460.80 Mhz に変更します。
    CreateRequestSession(requestGpu, nn::fgm::Module_Gpu, 460800000);

    // メモリのクロックレートを 800.00 Mhz に変更します。
    CreateRequestSession(requestEmc, nn::fgm::Module_Emc, 800000000);

    NN_LOG("Created requests.\n");

    // CreateRequestSession から DeleteRequestSession までの間、クロックレートの変更が維持されます。

    // CPU のクロックレートを 1683.00 MHz に再変更します。
    {
        nn::fgm::Request requestCpu2;
        CreateRequestSession(requestCpu2, nn::fgm::Module_Cpu, 1683000000);

        NN_LOG("Created a temporary request.\n");

        // CPU のクロックレートが再変更されたことは別の nn::fgm::Request オブジェクトからも確認できます。
        PrintSetting(requestCpu);
    }

    NN_LOG("Discarded a temporary request.\n");

    // requestCpu2 が破棄されると requestCpu の設定値に戻ります。
    PrintSetting(requestCpu);

    // CPU のクロックレートの変更を元に戻します。
    DeleteRequestSession(requestCpu);

    // GPU のクロックレートの変更を元に戻します。
    DeleteRequestSession(requestGpu);

    // メモリのクロックレートの変更を元に戻します。
    DeleteRequestSession(requestEmc);

    g_Device.Finalize();
    nn::gfx::Finalize();

    NN_LOG("Deleted requests.\n");
}
