﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include <nn/fgm.h>
#include <nn/fgm/fgm_Debugger.h>
#include <nn/fgm/fgm_Result.public.h>

#include "fgm_Worker.h"

namespace nn     {
namespace fgm    {
namespace sample {

const size_t        LogMemorySize  = (128 * 1024);
const size_t        EntriesPerRead = 2;
uint8_t             g_LogMemory[LogMemorySize] NN_ALIGNAS(4096);
nn::fgm::DebugEntry g_Entries[EntriesPerRead];
int                 g_TimeoutSeconds = 3600;

Worker g_Worker1 NN_ALIGNAS(4096);
Worker g_Worker2 NN_ALIGNAS(4096);
Worker g_Worker3 NN_ALIGNAS(4096);
Worker g_Worker4 NN_ALIGNAS(4096);

void RequestSimple()
{
    nn::Result result;
    nn::fgm::Request request;
    nn::fgm::Setting min = rand();
    nn::fgm::Setting max = nn::fgm::Setting_Max;
    nn::fgm::Setting actual;

    if ((result = request.Initialize(nn::fgm::Module_Test, Priority_Default)).IsFailure() ||
        (result = request.SetAndWait(min, max)).IsFailure() ||
        (result = request.Get(&actual)).IsFailure())
    {
        NN_LOG("failed to request new range, error %d:%d\n", result.GetModule(), result.GetDescription());
    }

    request.Finalize();
}

char* FormatTime(nn::os::Tick tick)
{
    static char buf[256];
    uint64_t usecs   = nn::os::ConvertToTimeSpan(tick).GetMicroSeconds();
    uint32_t msecs   = usecs   / 1000;
    uint32_t secs    = msecs   / 1000;
    uint32_t minutes = secs    / 60;
    uint32_t hours   = minutes / 60;

    nn::util::SNPrintf(
                buf, sizeof(buf),
                "%02u:%02u:%02u:%03u:%03u",
                hours % 24,
                minutes % 60,
                secs % 60,
                msecs % 1000,
                (uint32_t)(usecs % 1000));

    return buf;
}

void ParseLog(DebugEntry* entries, uint32_t nEntries)
{
    for (int i = 0; i < nEntries; i++)
    {
        switch (entries[i].event)
        {
        case nn::fgm::EventId_Set:
            NN_LOG("%s - module %u set req: pid %llu, priority %u, min %u, max %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.request.pid,
                    entries[i].u.request.priority,
                    entries[i].u.request.min,
                    entries[i].u.request.max);
            break;

        case nn::fgm::EventId_Unset:
            NN_LOG("%s - module %u unset req: pid %llu, priority %u, min %u, max %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.request.pid,
                    entries[i].u.request.priority,
                    entries[i].u.request.min,
                    entries[i].u.request.max);
            break;

        case nn::fgm::EventId_Completion:
            NN_LOG("%s - module %u completed, requested min %u, requested max %u, actual %u\n",
                    FormatTime(entries[i].timestamp),
                    entries[i].id,
                    entries[i].u.completion.min,
                    entries[i].u.completion.max,
                    entries[i].u.completion.actual);
            break;

        default:
            break;
        }
    }
}

void Log()
{
    nn::Result result;
    uint32_t   read;
    uint32_t   lost;
    uint32_t   remaining;

    nn::fgm::Debugger debugger;

    result = debugger.Initialize(g_LogMemory, sizeof(g_LogMemory));
    NN_ABORT_UNLESS(result.IsSuccess());

    while (debugger.WaitWithTimeout(nn::TimeSpan::FromSeconds(g_TimeoutSeconds)).IsSuccess())
    {
        do
        {
            result = debugger.Read(
                        g_Entries,
                        sizeof(g_Entries) / sizeof(g_Entries[0]),
                        &read,
                        &lost,
                        &remaining);

            if (result.IsSuccess() && read > 0)
            {
                ParseLog(g_Entries, read);
            }

        } while (remaining > 0);

        if (lost != 0)
        {
            NN_LOG("overrun(s) detected, discarded %u old events\n", lost);
        }
    }

    debugger.Finalize();
}

}}}

extern "C" void nninitStartup()
{

}

extern "C" void nnMain()
{
    nn::fgm::sample::g_Worker1.Initialize(nn::fgm::sample::Log, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker2.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker3.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);
    nn::fgm::sample::g_Worker4.Initialize(nn::fgm::sample::RequestSimple, nn::os::DefaultThreadPriority);

    nn::fgm::sample::g_Worker1.Finalize();
    nn::fgm::sample::g_Worker2.Finalize();
    nn::fgm::sample::g_Worker3.Finalize();
    nn::fgm::sample::g_Worker4.Finalize();
}
