﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{EnsMessage.cpp,PageSampleEnsMessage}
 *
 * @brief
 *  メッセージの送受信を扱うサンプルプログラム
 */

/**
 * @page PageSampleEnsMessage メッセージの送受信
 * @tableofcontents
 *
 * @brief
 *  メッセージの送受信を扱うサンプルプログラムの解説です。
 *
 * @section EnsMessage_SectionBrief 概要
 *  メッセージの送信には nn::ens::SendMessage() を利用します。
 *
 *  メッセージの受信には nn::ens::ReceiveMessageHeaderList(), nn::ens::ReceiveMessageBody() を利用します。
 *
 *  nn::ens のネットワークサービスの利用にはネットワークサービスアカウントのIDトークンが必要です。
 *  @ref nn::account "nn::account のリファレンス" やドキュメントも併せて参照してください。
 *
 * @section EnsMessage_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/EnsMessage Samples/Sources/Applications/EnsMessage @endlink 以下にあります。
 *
 * @section EnsMessage_SectionNecessaryEnvironment 必要な環境
 *  - ネットワーク接続できる環境
 *  - ネットワークサービスアカウントの準備
 *
 * @section EnsMessage_SectionHowToOperate 操作方法
 *  プレイヤー選択などはUIに従って操作してください。
 *  プレイヤー選択後、自動的に処理が実行され、サンプルが終了します。
 *
 * @section EnsMessage_SectionPrecaution 注意事項
 *  実行結果はログに出力されます。
 *
 * @section EnsMessage_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section EnsMessage_SectionDetail 解説
 *
 * @subsection EnsMessage_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  EnsMessage.cpp
 *  @includelineno EnsMessage.cpp
 *
 * @subsection EnsMessage_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - 各種ライブラリの初期化(account, nifm, socket, time, curl)
 * - セーブデータのマウント
 * - ユーザーの選択とオープン( SelectAndOpenUser() )
 * - ネットワーク接続( EnsureNetworkConnection() )
 * - ネットワークサービスアカウントのIDトークンの取得( GetNsaIdToken() )
 * - nn::ens の利用開始
 * - メッセージ送信者の認証情報の生成( GenerateEnsCredential() )
 * - メッセージ受信者の認証情報の生成( GenerateEnsCredential() )
 * - 通知の有効化
 * - メッセージの送信
 * - 通知の受信
 * - メッセージのヘッダーリストの受信
 * - メッセージの本文の受信
 * - nn::ens の利用終了
 *
 * サンプルコードの簡潔化のため、送信者と受信者の認証情報( nn::ens::Credential )は、
 * ともに1つのネットワークサービスアカウントの ID トークンを使って生成しています。
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 * ただし、 nn::ens::UserId, nn::ens::MessageId の値や時刻は実行するたびに異なるものが表示されます。
 *
 * @verbinclude EnsMessage_ExampleOutput.txt
 *
 */

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_Result.h>
#include <nn/ens.h>
#include <nn/ens/ens_ApiForAcbaa.h>
#include <nn/err.h>
#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <curl/curl.h>

#include "EnsFsUtility.h"

namespace
{
    // socket 用メモリインスタンス
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // nn::ens::StartServiceLoop() を実行するスレッドには、nn::ens::RequiredStackSize 以上のスタックサイズが必要。
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[nn::ens::RequiredStackSize];

    // nn::ens を動かすワークメモリーは最低でも nn::ens::RequiredMemorySizeMin が必要で、
    // さらに一度のリクエストで扱う通信データサイズを加算する必要がある。
    //
    // ここでは 4MB を加算する。この場合一度に 4MB までのデータの送信、もしくは受信が行えるようになる。
    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];

    // nn::ens::ActivateNotificationService() の実行をサーバー負荷低減のため間引きます。
    // そのためのセーブデータ関連のパラメータ定義です。
    const char* SaveDataMountName = "save";
    const char* SaveDataCredentialPath = "save:/Credential.dat";
    const char* SaveDataLastNotificationActivateTimePointPath = "save:/LastNotificationActivateTimePoint.dat";
}

// nn::ens::StartServiceLoop() を別スレッドで実行するための関数
void WorkerThread(void*) NN_NOEXCEPT
{
    NN_LOG("Start Extended Network Services Loop.\n");

    // 第一引数は"利用するサーバーを示すキー文字列"です。
    // このサンプルでは空文字列を指定していますが、実際にはアプリケーションに割り振られた値を指定してください。
    // nn::ens::StartServiceLoop() はブロック関数です。 nn::ens::StopServiceLoop() の呼び出しによって抜けます。
    nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof(g_ServiceWorkMemory));

    NN_LOG("End Extended Network Services Loop.\n");
}

// ユーザー選択、オープンに成功すると true が返る
bool SelectAndOpenUser(nn::account::UserHandle* pOut) NN_NOEXCEPT
{
    nn::account::Uid uid;
    auto result = nn::account::ShowUserSelector(&uid);

    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        NN_LOG("nn::account::ShowUserSelector() is canceled by user.\n");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(pOut, uid));

    return true;
}

// ネットワーク利用要求が通ると true が返る
bool EnsureNetworkConnection() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if (nn::nifm::IsNetworkAvailable())
        {
            return true;
        }

        auto result = nn::nifm::HandleNetworkRequestResult();
        if(result.IsSuccess())
        {
            return true;
        }
        else if (!nn::nifm::ResultErrorHandlingCompleted::Includes(result))
        {
            // エラーが解消できない
            NN_LOG("Network is not available.\n");
            return false;
        }
        // リトライでインターネット接続できる可能性がある
    }
}

// ネットワークサービスアカウントのIDトークンの取得に成功すると true が返る
bool GetNsaIdToken(char* pToken, size_t tokenSize, const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        // ネットワークサービスアカウントの確定
        auto result = nn::account::EnsureNetworkServiceAccountAvailable(handle);
        if (nn::account::ResultCancelledByUser::Includes(result))
        {
            NN_LOG("nn::account::EnsureNetworkServiceAccountAvailable is canceled by user.");
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ネットワークサービスアカウントのIDトークンのキャッシュを確保
        nn::account::AsyncContext context;
        result = nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&context, handle);
        if(nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
        {
            // ネットワークサービスアカウントが利用可能でない(nn::account::EnsureNetworkServiceAccountAvailable()で解消可能)
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&e));
        e.Wait();
        result = context.GetResult();
        if(result.IsFailure())
        {
            if(nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
            {
                // ネットワークサービスアカウントが利用可能でない(nn::account::EnsureNetworkServiceAccountAvailable()で解消可能)
                continue;
            }
            else
            {
                // エラー表示が必要(ネットワーク通信に関するエラーなど)
                nn::err::ShowError(result);
                return false;
            }
        }

        // ネットワークサービスアカウントのIDトークンのキャッシュを取得
        size_t nsaIdTokenActualSize;
        result = nn::account::LoadNetworkServiceAccountIdTokenCache(
            &nsaIdTokenActualSize,
            pToken,
            tokenSize,
            handle);

        if (nn::account::ResultNetworkServiceAccountUnavailable::Includes(result) || nn::account::ResultTokenCacheUnavailable::Includes(result))
        {
            // nn::account::EnsureNetworkServiceAccountAvailable()で解消可能もしくは、
            // nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync()の実行が必要。
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return true;
    }
}

// nn::ens のエラーリザルトのハンドリング
void HandleEnsResult(nn::Result result, const char* caption) NN_NOEXCEPT
{
    NN_ASSERT(result.IsFailure());

    NN_LOG("Failed to %s. (%03d-%04d)\n",
        caption,
        result.GetModule(), result.GetDescription());

    // nn::ens の API 失敗は、基本的にはエラービューアでのエラー表示が必要ですが、
    // ネットワークサービスアカウントのIDトークンの失効時のエラーは、トークン取り直しによって解消することができます。
    // 現時点ではこのトークン失効のエラーをハンドリングすることはできません。将来のリリースにて改定予定です。
    nn::err::ShowError(result);
}

// nn::ens のネットワークサービスを利用するための認証情報の生成に成功すると true が返る
bool GenerateEnsCredential(nn::ens::Credential* pOut, const char* nsaIdToken) NN_NOEXCEPT
{
    nn::ens::AsyncContext context;
    nn::ens::Credential credential;

    nn::ens::GenerateCredential(&context, &credential, nsaIdToken);

    context.GetEvent().Wait();
    auto result = context.GetResult();

    if(result.IsSuccess())
    {
        // nn::ens::Credential には、nn::ens のネットワークサービスを利用するユーザーを一意に決める nn::ens::UserId が含まれています。
        // 同じユーザーIDに対する認証情報は変化することはありません。
        // nn::ens::Credential 構造体をセーブデータなどに永続化し、次回以降の起動へ引き継ぐことで、同じユーザーとしてサービスを利用することができます。

        *pOut = credential;

        return true;
    }
    else
    {
        HandleEnsResult(result, "nn::ens::GenerateCredential()");
        return false;
    }
}

extern "C" void nnMain()
{
    // 各種ライブラリ初期化
    nn::account::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));
    curl_global_init(CURL_GLOBAL_DEFAULT);
    NN_UTIL_SCOPE_EXIT
    {
        curl_global_cleanup();
        nn::socket::Finalize();
    };

    // ユーザー選択、オープン
    nn::account::UserHandle userHandle;
    if(!SelectAndOpenUser(&userHandle))
    {
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(userHandle);
    };

    // セーブデータのマウント
    {
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, userHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::EnsureSaveData(uid));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveData(SaveDataMountName, uid));
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(SaveDataMountName);
    };

    // ネットワーク利用要求
    if(!EnsureNetworkConnection())
    {
        return;
    }

    // ネットワークサービスアカウントのIDトークン取得
    char nsaIdToken[nn::account::NetworkServiceAccountIdTokenLengthMax];
    if(!GetNsaIdToken(nsaIdToken, sizeof(nsaIdToken), userHandle))
    {
        return;
    }

    // Extended Network Services の利用開始
    nn::os::ThreadType thread;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    // メッセージの送信者となる nn::ens::Credential の発行
    nn::ens::Credential senderCredential;
    if(!GenerateEnsCredential(&senderCredential, nsaIdToken))
    {
        return;
    }
    NN_LOG("Sender nn::ens::UserId = 0x%016llx\n", senderCredential.userId.value);

    // メッセージの受信者となる nn::ens::Credential の準備
    // セーブデータがあればそれを採用します
    nn::ens::Credential receiverCredential = nn::ens::InvalidCredential;
    if(!nns::ens::ReadFile(&receiverCredential, SaveDataCredentialPath))
    {
        // メッセージの受信者となる nn::ens::Credential の発行
        if(!GenerateEnsCredential(&receiverCredential, nsaIdToken))
        {
            return;
        }

        // 次回も同じ nn::ens::Credential を利用するため永続化
        nns::ens::WriteFile(receiverCredential, SaveDataMountName, SaveDataCredentialPath);

        // 最後に nn::ens::ActivateNotificationService() した瞬間の SteadyClockTimePoint は、
        // nn::ens::Credential ごとに管理する必要があるため削除
        nns::ens::DeleteFile(SaveDataMountName, SaveDataLastNotificationActivateTimePointPath);
    }
    NN_LOG("Receiver nn::ens::UserId = 0x%016llx\n", receiverCredential.userId.value);

    // 通知の有効化
    {
        // メッセージ受信時に通知が届きます。
        // このサンプルではメッセージの受信者のみ通知を有効化しています。

        // サーバー負荷低減のため、nn::ens::ActivateNotificationService() の実行を間引くことを推奨します。
        // 下記は前回の実行からの経過時間が7日未満であれば、 nn::ens::ActivateNotificationService() をスキップするコードです。
        // 閾値はサーバー側と調整の上決定してください。
        const nn::TimeSpan ActivationThresholdSpan = nn::TimeSpan::FromDays(7);

        bool activateNotificationServiceRequired = true;

        nn::time::SteadyClockTimePoint currentTimePoint;
        nn::time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint);

        nn::time::SteadyClockTimePoint lastTimePoint;
        if(nns::ens::ReadFile(&lastTimePoint, SaveDataLastNotificationActivateTimePointPath)) // 前回の SteadyClockTimePoint を読み込む
        {
            int64_t elapsed;
            auto result = nn::time::GetSpanBetween(&elapsed, lastTimePoint, currentTimePoint);

            // 前回からの経過時間が7日未満であれば、 nn::ens::ActivateNotificationService() をスキップします
            if(result.IsSuccess() && nn::TimeSpan::FromSeconds(elapsed) < ActivationThresholdSpan)
            {
                NN_LOG("Skip nn::ens::ActivateNotificationService(). elapsed:%lld[sec]\n", elapsed);
                activateNotificationServiceRequired = false;
            }
        }

        // セーブデータが別デバイスへ移行した場合にも通知を正しく受信できるようにするため、
        // nn::time::GetSpanBetween() 失敗時には、必ず nn::ens::ActivateNotificationService() を実行してください。

        if(activateNotificationServiceRequired)
        {
            nn::account::Uid uid;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, userHandle));

            nn::ens::AsyncContext context;
            nn::ens::ActivateNotificationService(&context, uid, receiverCredential, nsaIdToken);

            context.GetEvent().Wait();

            auto result = context.GetResult();
            if(result.IsFailure())
            {
                HandleEnsResult(result, "nn::ens::ActivateNotificationService()");
                return;
            }

            // nn::ens::ActivateNotificationService() 成功後、 currentTimePoint を次回の
            // lastTimePoint となるようセーブデータに永続化します。
            nns::ens::WriteFile(currentTimePoint, SaveDataMountName, SaveDataLastNotificationActivateTimePointPath);
        }
    }

    // メッセージの送信
    {
        // メタは、メッセージそのものや送信者の属性などを表す比較的小さいデータを想定しており、
        // nn::ens::ReceiveMessageHeaderList() で複数を一括して受信できることが特徴です。
        // メタのサイズを小さくすることで、 nn::ens::ReceiveMessageHeaderList() で複数のヘッダーを
        // 一括してダウンロードする際にかかる時間やメモリを抑えることができます。

        struct MetaBody
        {
            nn::ens::SendBuffer meta;
            nn::ens::SendBuffer body;
        };

        MetaBody metaBodyList[2] =
        {
            // 1度目の送信データ
            {
                // meta
                {"meta 1", sizeof("meta 1")},
                // body
                {"This is body data 1.", sizeof("This is body data 1.")}
            },

            // 2度目の送信データ
            {
                // meta
                {"meta 2", sizeof("meta 2")},
                // body
                {"This is body data 2.", sizeof("This is body data 2.")}
            }
        };

        for(int i = 0; i < NN_ARRAY_SIZE(metaBodyList); i++)
        {
            const nn::ens::UserId targetUserId = receiverCredential.userId;
            nn::ens::AsyncContext context;

            nn::ens::SendMessage(&context,
                targetUserId, metaBodyList[i].meta, metaBodyList[i].body, senderCredential, nsaIdToken);

            context.GetEvent().Wait();

            auto result = context.GetResult();
            if(result.IsFailure())
            {
                HandleEnsResult(result, "nn::ens::SendMessage()");
                return;
            }

            // メッセージの送信時刻をずらすために待ちます。
            // これによって、2度目の送信データが先に受信されることを保証しています。
            // サンプルの挙動に一貫性を持たせるためであり、実際のアプリケーションで待つ必要はありません。
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    // 通知の受信
    {
        nn::ens::NotificationData notificationData;

        // 通知が来るまで適当に待ちます
        NN_LOG("Wait for notification...\n");
        nn::ens::GetNotificationEvent().TimedWait(nn::TimeSpan::FromSeconds(5));

        while(nn::ens::PopNotificationData(&notificationData))
        {
            NN_LOG("Notification received. Type:%s, nn::ens::UserId = 0x%016llx\n", notificationData.type, notificationData.receiver.value);
        }
    }

    // メッセージの受信
    {
        nn::ens::MessageHeader headerList[10];
        int outCount;

        // ヘッダーリストのメタを格納するバッファの初期化
        // nn::ens::ReceiveMessageHeaderList() を呼び出す前に、メタを格納するのに十分なメモリをセットしておく必要があります。
        char metaBuffers[NN_ARRAY_SIZE(headerList)][256];
        {
            for(int i = 0; i < NN_ARRAY_SIZE(headerList); i++)
            {
                headerList[i].metadata.pBuffer = metaBuffers[i];
                headerList[i].metadata.bufferSize = sizeof(metaBuffers[i]);
                headerList[i].metadata.receivedSize = 0u;
            }
        }

        // メッセージのヘッダーリストの受信
        {
            int outTotalCount;
            const int offset = 0;
            nn::ens::AsyncContext context;

            nn::ens::ReceiveMessageHeaderList(&context,
                &outCount, &outTotalCount, headerList, NN_ARRAY_SIZE(headerList), offset, receiverCredential, nsaIdToken);

            context.GetEvent().Wait();

            auto result = context.GetResult();
            if(result.IsFailure())
            {
                HandleEnsResult(result, "nn::ens::ReceiveMessageHeaderList()");

                return;
            }

            // headerList には送信時刻が新しい順にヘッダーが格納されます。

            NN_LOG("Received Header Info (count:%d, total count:%d) ...\n", outCount, outTotalCount);
            for(int i = 0; i < outCount; i++)
            {
                const nn::ens::MessageHeader& header = headerList[i];

                NN_LOG("  - headerList[%d]\n", i);

                NN_LOG("    - MessageId     : 0x%016llx\n", header.id.value);

                NN_LOG("    - Sender UserId : 0x%016llx\n", header.sender.value);

                auto c = nn::time::ToCalendarTimeInUtc(header.sentAt);
                NN_LOG("    - SentAt        : (UTC) %04d/%02d/%02d %02d:%02d:%02d\n", c.year, c.month, c.day, c.hour, c.minute, c.second);

                NN_LOG("    - Metadata\n");
                NN_LOG("      - receivedSize : %zu\n", header.metadata.receivedSize);
                NN_LOG("      - pBuffer      : \"%.256s\"\n", static_cast<const char*>(header.metadata.pBuffer));
            }
        }

        // メッセージ本文の受信
        for(int i = 0; i < outCount; i++)
        {
            // 本文を格納するバッファの初期化
            // nn::ens::ReceiveMessageBody() を呼び出す前に、本文を格納するのに十分なメモリをセットしておく必要があります。
            nn::ens::ReceiveBuffer bodyBuffer;
            char buffer[256];
            bodyBuffer.pBuffer = buffer;
            bodyBuffer.bufferSize = sizeof(buffer);
            bodyBuffer.receivedSize = 0u;

            const nn::ens::MessageId targetMessageId = headerList[i].id;
            nn::ens::AsyncContext context;

            nn::ens::ReceiveMessageBody(&context,
                &bodyBuffer, targetMessageId, receiverCredential, nsaIdToken);

            context.GetEvent().Wait();

            auto result = context.GetResult();
            if(result.IsFailure())
            {
                HandleEnsResult(result, "nn::ens::ReceiveMessageBody()");
                return;
            }

            NN_LOG("Received Body Info (MessageId:0x%016llx) ...\n", targetMessageId.value);
            NN_LOG("  - receivedSize : %zu\n", bodyBuffer.receivedSize);
            NN_LOG("  - pBuffer      : \"%.256s\"\n", bodyBuffer.pBuffer);
        }
    }

    // Extended Network Services の利用終了
    nn::ens::StopServiceLoop();
    nn::os::DestroyThread(&thread);
} // NOLINT(impl/function_size)

