﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_SocketApi.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/dhcps.h>

namespace
{
#define STATIC_IP_ADDR      "192.168.0.2"
#define STATIC_GW_ADDR      "192.168.0.1"
#define STATIC_SUBNET_MASK  "255.255.255.0"
#define DHCP_RANGE_BEGIN    "192.168.0.3"
#define DHCP_RANGE_END      "192.168.0.13"
#define STATIC_DNS_1        "8.8.8.8"
#define STATIC_DNS_2        "8.8.4.4"
#define WLAN_INTERFACE_NAME "wl0"
#define WLAN_SSID           "nxdhcptest"
#define WLAN_PRIORITY_MODE  true

const unsigned int g_PidLocalExperimental2 = 0x88B6;

const nn::wlan::ReceivedDataMatchInfo DummyMatchInfo = {
    { 0x00, 0x00, 0x00, 0x00, 0x00 }, 5
};

uint32_t g_LocalRxId = 0;

nn::socket::BpfOnlyConfig g_SocketConfig;

bool g_HadStopEvent = false;

nn::Result ConfigureInterface(const char* interfaceName, bool enable)
{
    nn::Result result;

    if (enable)
    {
        nn::bsdsocket::cfg::IfSettings ifcfg;
        memset(&ifcfg, 0, sizeof(ifcfg));

        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
        ifcfg.mtu  = 1500;
        nn::socket::InetAton(STATIC_IP_ADDR, &ifcfg.u.modeStatic.addr);
        nn::socket::InetAton(STATIC_GW_ADDR, &ifcfg.u.modeStatic.gatewayAddr);
        nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
        ifcfg.u.modeStatic.broadcastAddr.S_addr =
            (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
            ~ifcfg.u.modeStatic.subnetMask.S_addr;
        nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
        nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);

        result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
        if (result.IsFailure())
        {
            NN_LOG("failed to configure interface %s - %d:%d\n",
                   interfaceName,
                   result.GetModule(),
                   result.GetDescription());
        };
    }
    else
    {
        result = nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
    };

    return result;
}

nn::Result InitializeAccessPoint()
{
    nn::Result result;
    uint16_t pidOuiExp2 = g_PidLocalExperimental2;

    nn::wlan::MasterBssParameters param;

    result = nn::wlan::InitializeLocalManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::wlan::InitializeSocketManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::socket::Initialize(g_SocketConfig);
    NN_ASSERT(result.IsSuccess());

    result = ConfigureInterface(WLAN_INTERFACE_NAME, true);
    NN_ASSERT(result.IsSuccess());

    result = nn::wlan::Local::OpenMasterMode();
    NN_ASSERT(result.IsSuccess());

    param.channel = -1;
    param.autoKeepAlive = false;
    param.basicRates = nn::wlan::RateSetLegacy_11gMask;
    param.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    param.beaconInterval = 100;
    param.hiddenSsid = false;
    param.inactivePeriod = 10;
    param.security.privacyMode = nn::wlan::SecurityMode_Open;
    param.security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    param.ssid.Set(WLAN_SSID);

    nn::wlan::Local::CreateBss(param);

    result = nn::wlan::Local::CreateRxEntry(&g_LocalRxId,
                                            &pidOuiExp2,
                                            1,
                                            30);
    NN_ASSERT(result.IsSuccess());
    result = nn::wlan::Local::AddMatchingDataToRxEntry(g_LocalRxId, DummyMatchInfo);
    NN_ASSERT(result.IsSuccess());

    return result;
};

void FinalizeAccessPoint()
{
    NN_ASSERT(nn::wlan::Local::RemoveMatchingDataFromRxEntry(g_LocalRxId, DummyMatchInfo).IsSuccess());
    NN_ASSERT(nn::wlan::Local::DestroyBss().IsSuccess());
    NN_ASSERT(nn::wlan::Local::DeleteRxEntry(g_LocalRxId).IsSuccess());
    NN_ASSERT(nn::wlan::Local::CloseMasterMode().IsSuccess());
    NN_ASSERT(ConfigureInterface(WLAN_INTERFACE_NAME, false).IsSuccess());
    nn::socket::Finalize();
    nn::wlan::FinalizeSocketManager();
    nn::wlan::FinalizeLocalManager();
};

void DhcpServerCallback(void* context, nn::dhcps::Event event, const void* pData)
{
    const int IpStringSize = 17;
    char ipString[IpStringSize];

    switch (event)
    {
    case nn::dhcps::Event::OnStopped:
        g_HadStopEvent = true;
    case nn::dhcps::Event::OnStart:
    case nn::dhcps::Event::OnRunning:
    case nn::dhcps::Event::OnStopping:
    case nn::dhcps::Event::OnDetectedNetworkChange:
    case nn::dhcps::Event::OnRecoverableError:
    case nn::dhcps::Event::OnUnrecoverableError:
    {
        NN_LOG("DHCP Server callback event: %s (%u)\n",
               nn::dhcps::EventToString(event),
               event);
    };
    break;

    case nn::dhcps::Event::OnLeaseOfferAdvancedState:
    case nn::dhcps::Event::OnLeaseRequestInvalid:
    case nn::dhcps::Event::OnLeaseDeclineInvalid:
    case nn::dhcps::Event::OnLeaseInformInvalid:
    case nn::dhcps::Event::OnLeaseReleaseInvalid:
    {
        const nn::dhcps::DhcpClientIdentity* pIdentity =
            reinterpret_cast<const nn::dhcps::DhcpClientIdentity*>(pData);

        // NN_LOG broken out like this to reduce library thread stack utilization in the sample
        NN_LOG("DHCP event: %s (%u); ", nn::dhcps::EventToString(event), event);
        NN_LOG("hash: %llx, ", pIdentity->hash);
        NN_LOG("ip: %s, ", nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pIdentity->ipAddress, ipString, sizeof(ipString)));
        NN_LOG("mac: %02x:%02x:%02x:%02x:%02x:%02x\n",
               pIdentity->macAddress[0], pIdentity->macAddress[1], pIdentity->macAddress[2],
               pIdentity->macAddress[3], pIdentity->macAddress[4], pIdentity->macAddress[5]);
    };
    break;

    case nn::dhcps::Event::OnLeaseOffered:
    case nn::dhcps::Event::OnLeaseOfferResent:
    case nn::dhcps::Event::OnLeaseOfferExpired:
    case nn::dhcps::Event::OnLeaseDecline:
    case nn::dhcps::Event::OnLeaseInform:
    case nn::dhcps::Event::OnLeaseRenewing:
    case nn::dhcps::Event::OnLeaseRequest:
    case nn::dhcps::Event::OnLeaseRebinding:
    case nn::dhcps::Event::OnLeaseExpired:
    case nn::dhcps::Event::OnLeaseRelease:
    case nn::dhcps::Event::OnTableFull:
    case nn::dhcps::Event::OnTableAvailable:
    case nn::dhcps::Event::OnDiscoverWhileTableFull:
    case nn::dhcps::Event::OnDhcpUnhandledMessage:
    {
        const nn::dhcps::DhcpLease* pLease =
            reinterpret_cast<const nn::dhcps::DhcpLease*>(pData);

        // NN_LOG broken out like this to reduce library thread stack utilization in the sample
        NN_LOG("DHCP event: %s (%u); ", nn::dhcps::EventToString(event), event);
        NN_LOG("hash: %llx, ", pLease->client.hash);
        NN_LOG("ip: %s, ", nn::socket::InetNtop(nn::socket::Family::Af_Inet, &pLease->client.ipAddress, ipString, sizeof(ipString)));
        NN_LOG("mac: %02x:%02x:%02x:%02x:%02x:%02x ",
               pLease->client.macAddress[0], pLease->client.macAddress[1], pLease->client.macAddress[2],
               pLease->client.macAddress[3], pLease->client.macAddress[4], pLease->client.macAddress[5]);
        NN_LOG("state: %s (%u), ", nn::dhcps::StateToString(pLease->state), pLease->state);
        NN_LOG("lease: %u, ", pLease->lease);
        NN_LOG("t1: %u, ", pLease->t1);
        NN_LOG("t2: %u\n", pLease->t2);
    };
    break;

    default:
        NN_LOG("DHCP event: %s (%u), pData: %p\n",
               nn::dhcps::EventToString(event), event, pData);

    };
};

extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), "Main Thread");

    size_t memorySize = 0;
    nn::dhcps::UserConfiguration userConfiguration;
    void* pLeaseTableMemory = nullptr;
    unsigned int seconds = 5 * 24 * 60 * 60; // 5 days

    InitializeAccessPoint();

    userConfiguration.pInterfaceName = WLAN_INTERFACE_NAME;
    userConfiguration.pCallback = DhcpServerCallback;
    userConfiguration.pContext = nullptr;
    userConfiguration.offerSeconds = 10;
    userConfiguration.leaseSeconds = 600;
    userConfiguration.t1Ratio = nn::dhcps::DefaultT1Ratio;
    userConfiguration.t2Ratio = nn::dhcps::DefaultT2Ratio;
    userConfiguration.hasGateway = false;
    userConfiguration.dnsServerCount = 0;
    userConfiguration.pDnsServers = nullptr;
    nn::socket::InetPton(nn::socket::Family::Af_Inet, DHCP_RANGE_BEGIN, &userConfiguration.begin);
    nn::socket::InetPton(nn::socket::Family::Af_Inet, DHCP_RANGE_END, &userConfiguration.end);

    if (!nn::dhcps::Initialize(userConfiguration).IsSuccess())
    {
        NN_LOG("Initialization failure\n");
        goto bail;
    };

    if (!nn::dhcps::GetMemorySize(&memorySize).IsSuccess())
    {
        NN_LOG("GetMemorySize failure\n");
        goto bail;
    };

    if (nullptr == (pLeaseTableMemory = malloc(memorySize)))
    {
        NN_LOG("Lease table memory allocation failure\n");
        goto bail;
    };

    if (!nn::dhcps::SetMemory(pLeaseTableMemory, memorySize).IsSuccess())
    {
        NN_LOG("SetMemory failure\n");
        goto bail;
    };

    if (!nn::dhcps::Start().IsSuccess())
    {
        NN_LOG("Start failure\n");
        goto bail;
    };

    for (; seconds > 0; --seconds)
    {
        if (true == g_HadStopEvent)
        {
            break;
        };

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    };

    if (!nn::dhcps::Stop().IsSuccess())
    {
        NN_LOG("Stop failure\n");
        goto bail;
    };

bail:
    if (!nn::dhcps::Finalize().IsSuccess())
    {
        NN_LOG("Finalize failure\n");
    };

    if (nullptr != pLeaseTableMemory)
    {
        free(pLeaseTableMemory);
        pLeaseTableMemory = nullptr;
    };

    FinalizeAccessPoint();

    return;
};

};//namespace
