﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/gfx.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/util/util_Color.h>

#include <nns/hid.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "Demo1.h"
#include "Demo1PluginManager.h"
#include "Demo1TouchScreen.h"

namespace
{

SET_PLUGIN( "TouchScreen", TouchScreenDemo, PluginProperty_Drawable );

void WriteTouchScreenState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                           const nns::hid::TouchScreen* const pTouchScreen) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (pTouchScreen == NULL)
    {
        return;
    }

    size_t stepCounter = 1;
    float offsetX = 650.0f;
    float offsetY = 30.0f;

    pTextWriter->SetTextColor(Color::Yellow);
    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);
    pTextWriter->SetCursor(offsetX, offsetY * static_cast<float>(stepCounter));
    pTextWriter->Print("TouchScreen");

    if (pTouchScreen->GetTouchStates().size() == 0)
    {
        pTextWriter->SetTextColor(Color::Gray);
        pTextWriter->SetCursor(offsetX, offsetY * static_cast<float>(++stepCounter));
        pTextWriter->Print("Touch not found");
        return;
    }

    const std::vector<nns::hid::TouchScreen::TouchState
                      >& States = pTouchScreen->GetTouchStates();

    for (size_t i = 0; i < States.size(); ++i)
    {
        const nns::hid::TouchScreen::TouchState& State = States[i];
        pTextWriter->SetCursor(offsetX, offsetY * static_cast<float>(stepCounter + i + 1));
        pTextWriter->Print("id %08X (%4.0f, %4.0f)",
                           State.fingerId,
                           State.position.x,
                           State.position.y);
    }

    pTextWriter->SetScale(NormalFontScaleX, NormalFontScaleY);

    const float BarOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
    const float BarOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
    const float MinusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
    const float MinusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
    const float PlusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
    const float PlusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

    for (size_t i = 0; i < States.size(); ++i)
    {
        const nns::hid::TouchScreen::TouchState& State = States[i];
        pTextWriter->SetCursor(State.position.x - BarOffsetX - 45,
                               State.position.y - BarOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(State.position.x - BarOffsetX + 45,
                               State.position.y - BarOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(State.position.x - MinusOffsetX,
                               State.position.y - MinusOffsetY - 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(State.position.x - MinusOffsetX,
                               State.position.y - MinusOffsetY + 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(State.position.x - PlusOffsetX - 45,
                               State.position.y - PlusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(State.position.x - PlusOffsetX - 45,
                               State.position.y - PlusOffsetY + 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(State.position.x - PlusOffsetX + 45,
                               State.position.y - PlusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(State.position.x - PlusOffsetX + 45,
                               State.position.y - PlusOffsetY + 45);
        pTextWriter->Print("+");
    }
}
} // namespace

void TouchScreenDemo::MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT
{
    NN_UNUSED(frame);

    nn::gfx::CommandBuffer& commandBuffer = m_pGraphicsSystem->GetCommandBuffer();
    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();

    // テスト用にテキスト出力
    GetPluginManager().GetControllerManager().Update();

    WriteTouchScreenState(&debugFontTextWriter,
        reinterpret_cast<nns::hid::TouchScreen*>(
                GetPluginManager().GetControllerManager().GetController(
                        nns::hid::ControllerId_TouchScreen, 0)));

    //!< 共通操作説明を描画します。
    WriteCommonGuide( &debugFontTextWriter, pName);

    // 負荷メータを表示する
    DrawLoadMeter();

    // テキストを描画
    debugFontTextWriter.Draw(&commandBuffer);
}

void TouchScreenDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(
        s_Frame,
        m_Name.c_str());
    s_Frame++;
}
