﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ConsoleSimple.h"

#include <nn/nn_Log.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/util/util_FormatString.h>

namespace {
// キー入力から文字列を抽出する関数
void ExtractCharactors(char* buffer, size_t bufferSize, const nn::hid::KeyboardKeySet& key, bool capital) NN_NOEXCEPT;
} // namespace ~anonymous

TypeWriter::TypeWriter() NN_NOEXCEPT
    : m_IsShifted(false)
    , m_IsCapsLocked(false)
{
}

void TypeWriter::PrintStatus() NN_NOEXCEPT
{
    m_StatusConsole.Clear();
    char status[128];
    nn::util::SNPrintf(
        status, sizeof(status),
        "Shift:    %s\n"
        "CapsLock: %s\n",
        m_IsShifted? "ON" : "OFF",
        m_IsCapsLocked? "ON" : "OFF");
    m_StatusConsole.Puts(status);
}
void TypeWriter::Echo(const nn::hid::KeyboardKeySet& key) NN_NOEXCEPT
{
    char buffer[128];
    ExtractCharactors(buffer, sizeof(buffer), key, (m_IsShifted ^ m_IsCapsLocked));
    NN_LOG("%s", buffer); // 制御文字が入っている可能性があるので format に指定しない。
    m_LogConsole.Puts(buffer);
}
void TypeWriter::Reset() NN_NOEXCEPT
{
    m_IsShifted = m_IsCapsLocked = false;
    PrintStatus();
    m_LogConsole.Clear();
    NN_LOG("<cleared>\n");
}

void TypeWriter::EnableShift() NN_NOEXCEPT
{
    if (!m_IsShifted)
    {
        m_IsShifted = true;
        PrintStatus();
    }
}
void TypeWriter::DisableShift() NN_NOEXCEPT
{
    if (m_IsShifted)
    {
        m_IsShifted = false;
        PrintStatus();
    }
}

void TypeWriter::EnableCapsLock() NN_NOEXCEPT
{
    if (!m_IsCapsLocked)
    {
        m_IsCapsLocked = true;
        PrintStatus();
    }
}
void TypeWriter::DisableCapsLock() NN_NOEXCEPT
{
    if (m_IsCapsLocked)
    {
        m_IsCapsLocked = false;
        PrintStatus();
    }
}

void TypeWriter::OnKeyPressedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT
{
    if (modifier.Test<nn::hid::KeyboardModifier::Shift>())
    {
        EnableShift();
    }
    if (modifier.Test<nn::hid::KeyboardModifier::CapsLock>())
    {
        EnableCapsLock();
    }

    if (key.IsAnyOn())
    {
        Echo(key);
    }
}
void TypeWriter::OnKeyReleasedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT
{
    NN_UNUSED(key);
    if (modifier.Test<nn::hid::KeyboardModifier::Shift>())
    {
        DisableShift();
    }
    if (modifier.Test<nn::hid::KeyboardModifier::CapsLock>())
    {
        DisableCapsLock();
    }
}
void TypeWriter::OnKeyRepeatedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT
{
    NN_UNUSED(modifier);
    if (key.IsAnyOn())
    {
        Echo(key);
    }
}

void TypeWriter::OnButtonReleasedImpl(const nns::console::PadButtonSet& buttons) NN_NOEXCEPT
{
    if (buttons.Test<nns::console::PadButton::X>())
    {
        Reset();
    }
}

namespace
{
// キー入力から文字列を抽出する関数
void ExtractCharactors(char* buffer, size_t bufferSize, const nn::hid::KeyboardKeySet& key, bool capital) NN_NOEXCEPT
{
    NN_UNUSED(bufferSize);

    NN_SDK_REQUIRES(bufferSize >= 128);
    int index = 0;

#define LOAD_ALPHABET(_c) \
        do \
        { \
            NN_SDK_ASSERT(#_c[0] >= 'A' && #_c[0] <= 'Z'); \
            NN_SDK_ASSERT(#_c[1] == '\0'); \
            if (key.Test<nn::hid::KeyboardKey:: _c>()) \
            { \
                NN_SDK_ASSERT(static_cast<size_t>(index) < bufferSize); \
                buffer[index++] = (capital ? #_c[0] : #_c[0] - 'A' + 'a'); \
            } \
        } while (NN_STATIC_CONDITION(false))

    LOAD_ALPHABET(A);
    LOAD_ALPHABET(B);
    LOAD_ALPHABET(C);
    LOAD_ALPHABET(D);
    LOAD_ALPHABET(E);
    LOAD_ALPHABET(F);
    LOAD_ALPHABET(G);
    LOAD_ALPHABET(H);
    LOAD_ALPHABET(I);
    LOAD_ALPHABET(J);
    LOAD_ALPHABET(K);
    LOAD_ALPHABET(L);
    LOAD_ALPHABET(M);
    LOAD_ALPHABET(N);
    LOAD_ALPHABET(O);
    LOAD_ALPHABET(P);
    LOAD_ALPHABET(Q);
    LOAD_ALPHABET(R);
    LOAD_ALPHABET(S);
    LOAD_ALPHABET(T);
    LOAD_ALPHABET(U);
    LOAD_ALPHABET(V);
    LOAD_ALPHABET(W);
    LOAD_ALPHABET(X);
    LOAD_ALPHABET(Y);
    LOAD_ALPHABET(Z);

#undef LOAD_ALPHABET

#define LOAD_DIGIT(_d) \
        do \
        { \
            NN_SDK_ASSERT(#_d[0] >= '0' && #_d[0] <= '9'); \
            NN_SDK_ASSERT(#_d[1] == '\0'); \
            if (key.Test<nn::hid::KeyboardKey::D ## _d>()) \
            { \
                NN_SDK_ASSERT(static_cast<size_t>(index) < bufferSize); \
                buffer[index++] = #_d[0]; \
            } \
        } while (NN_STATIC_CONDITION(false))

    LOAD_DIGIT(0);
    LOAD_DIGIT(1);
    LOAD_DIGIT(2);
    LOAD_DIGIT(3);
    LOAD_DIGIT(4);
    LOAD_DIGIT(5);
    LOAD_DIGIT(6);
    LOAD_DIGIT(7);
    LOAD_DIGIT(8);
    LOAD_DIGIT(9);

#undef LOAD_DIGIT

    if (key.Test<nn::hid::KeyboardKey::Space>())
    {
        NN_SDK_ASSERT(static_cast<size_t>(index) < bufferSize);
        buffer[index++] = ' ';
    }

    if (key.Test<nn::hid::KeyboardKey::Return>())
    {
        NN_SDK_ASSERT(static_cast<size_t>(index) < bufferSize);
        buffer[index++] = '\n';
    }

    NN_SDK_ASSERT(static_cast<size_t>(index) < bufferSize);
    buffer[index] = '\0';
}
} // ~namespace <anonymous>
