﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{BreakpadSimple.cpp,PageSampleBreakpadSimple}
 *
 * @brief
 * breakpad による ELF ファイルからのデバッグ情報の取得
 */

/**
 * @page PageSampleBreakpadSimple breakpad による ELF ファイルからのデバッグ情報の取得
 * @tableofcontents
 *
 * @brief
 * breakpad による nss のデバッグ情報を取得するサンプルプログラムの解説です。
 *
 * @section PageSampleBreakpadSimple_SectionBrief 概要
 * 指定したアドレスが属する関数名を取得する breakpad ライブラリのサンプルになります。
 *
 * @section PageSampleBreakpadSimple_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/BreakpadSimple
 * Samples/Sources/Applications/BreakpadSimple @endlink 以下にあります。
 *
 * @section PageSampleBreakpadSimple_SectionNecessaryEnvironment 必要な環境
 * 本機能を利用するには、libbreakpad.a をリンクする必要があります。
 * libbreakpad.a は Breakpad パッケージで配布されていますので、別途NDIで取得する必要があります。
 *
 * @section PageSampleBreakpadSimple_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleBreakpadSimple_SectionPrecaution 注意事項
 * 本ライブラリを製品に含める場合、権利表記の追加が必要となります。
 * 詳細は SLIM をご参照ください。
 *
 * @section PageSampleBreakpadSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドして生成された BreakpadSimple.nss を dataSrc 以下にコピーして、リビルドしてから実行してください。
 *
 * @section PageSampleBreakpadSimple_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - dataSrc ディレクトリ以下に存在する自身のプログラムの nss ファイルを読み込みます。
 * - nss に含まれる debug_info の情報を表示します。
 * - nn::diag::GetBacktrace を利用して取得したバックトレースのアドレスが属する関数名を debug_info から見つけます。
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <breakpad/dwarf2reader.h>
#include <breakpad/functioninfo.h>

#include <new>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/diag.h>

using namespace dwarf2reader;

static uint64_t ReadDebugSections(ElfReader &rd, SectionMap *sections)
{
    uint64_t debug_info_len = 0;
    for (int shndx = 0; shndx < rd.GetNumSections(); ++shndx)
    {
        const char *sec_name = rd.GetSectionName(shndx);
        if (sec_name != nullptr && strncmp(sec_name, ".debug", 6) == 0)
        {
            size_t sec_size;
            auto *sec_data = rd.GetSectionByIndex(shndx, &sec_size);
            sections->insert(std::make_pair(sec_name, std::make_pair(reinterpret_cast<const uint8_t *>(sec_data), sec_size)));
            if (!strcmp(sec_name, ".debug_info"))
            {
                debug_info_len = sec_size;
            }
        }
    }
    return debug_info_len;
}

static void printFunction(const FunctionInfo &fi, LineMap &lm)
{
    // If DW_AT_decl_file(line) is missing we try to find file:line
    // using LineMap.
    std::string file = fi.file;
    unsigned line = fi.line;
    if (file.empty())
    {
        auto line_it = lm.lower_bound(fi.lowpc);
        if (line_it != lm.end()) std::tie(file, line) = (*line_it).second;
    }

    NN_LOG("{ '%s', '%s:%u', '[%p - %p]' }\n", fi.name.c_str(), file.c_str(),
            line, reinterpret_cast<void *>(fi.lowpc),
            reinterpret_cast<void *>(fi.highpc));
}

void SearchFunctionName(uintptr_t targetAddr, SectionMap sections, ByteReader br, const char* FilePath, uint64_t debug_info_len)
{
    uint64_t cu_len, cu_offset = 0;
    do
    {
        FunctionMap offset2fi, addr2fi;
        std::vector<std::string> dirs;
        std::vector<SourceFileInfo> files;
        LineMap lm;
        CULineInfoHandler lih(&files, &dirs, &lm);
        CUFunctionInfoHandler fih(&files, &dirs, nullptr, &offset2fi, &addr2fi, &lih, sections, &br);

        CompilationUnit cu(FilePath, sections, cu_offset, &br, &fih);
        cu_len = cu.Start();

        for(auto &p : addr2fi)
        {
            FunctionInfo* pFunctionInfo = p.second;

            // バックトレースで取得したアドレスが、指定されたアドレス内にあるかを確認する
            //NN_LOG("0x%X - 0x%X\n", pFunctionInfo->lowpc, pFunctionInfo->highpc);
            if( pFunctionInfo->lowpc <= targetAddr && targetAddr <= pFunctionInfo->highpc )
            {
                NN_LOG("%s\n", pFunctionInfo->name.c_str());
            }
        }

        cu_offset += cu_len;
    } while (cu_len && cu_offset < debug_info_len);
}


extern "C" void nnMain()
{
    nn::Result result;

    size_t cacheSize = 0;

    char* cacheBuffer = nullptr;

    // ファイルシステムをマウントします。
    // マウントにはキャッシュバッファが必要です。
    {
        NN_LOG("Mount Rom\n");

        // ファイルシステムのメタデータキャッシュに必要なバッファサイズを取得します。
        // 取得失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::QueryMountRomCacheSize(&cacheSize);

        // キャッシュバッファを確保します。
        cacheBuffer = new(std::nothrow) char[cacheSize];
        if (cacheBuffer == nullptr)
        {
            NN_ASSERT(false, "Cache buffer is null.\n");
            return;
        }

        // ファイルシステムをマウントします。
        // キャッシュバッファはアンマウントするまで解放しないでください。
        result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
        // 失敗した際は必ずアボートしてください。
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // リソースデータを読み込みます。
    const char* FilePath = "rom:/BreakpadSimple.nss";


    ElfReader rd(FilePath);
    ByteReader br(ENDIANNESS_LITTLE);
    br.SetAddressSize(rd.IsElf64File() ? 8 : 4);

    SectionMap sections;

    uint64_t cu_len, cu_offset = 0,
    debug_info_len = ReadDebugSections(rd, &sections);
    if (!debug_info_len)
    {
        fprintf(stderr, "%s: couldn't find .debug_info section\n", FilePath);
        exit(1);
    }

    // .debug_info の情報を出力するデモ
    do
    {
        FunctionMap offset2fi, addr2fi;
        std::vector<std::string> dirs;
        std::vector<SourceFileInfo> files;
        LineMap lm;
        CULineInfoHandler lih(&files, &dirs, &lm);
        CUFunctionInfoHandler fih(&files, &dirs, nullptr, &offset2fi, &addr2fi, &lih, sections, &br);

        CompilationUnit cu(FilePath, sections, cu_offset, &br, &fih);
        cu_len = cu.Start();
        for (auto &p : addr2fi) printFunction(*p.second, lm);

        cu_offset += cu_len;
    } while (cu_len && cu_offset < debug_info_len);


    // 取得したバックトレースのアドレスから、関数名を取得する
    size_t backtraceSize;
    uintptr_t backtraceAddress[16];

    // 関数名を取得する対象として、バックトレースを取得
    backtraceSize = nn::diag::GetBacktrace(backtraceAddress, 16);
    NN_LOG("backtrace:\n");
    for(int i = 0; i < backtraceSize; i++)
    {
        NN_LOG("   [%d] 0x%X\n", i, backtraceAddress[i]);
    }
    NN_LOG("\n\n");


    // ベースアドレスを調べるために、全モジュールの情報を得て、対応するモジュールを調べる。
    uintptr_t backtraceOffsetAddress = 0;
    const auto bufferSize = nn::diag::GetRequiredBufferSizeForGetAllModuleInfo();
    auto buffer = reinterpret_cast<nn::Bit8*>(std::malloc(bufferSize));

    nn::diag::ModuleInfo* modules;
    const auto moduleCount = nn::diag::GetAllModuleInfo(&modules, buffer, bufferSize);

    NN_LOG("Modules:\n");
    NN_LOG("  %-*s   %-*s   path\n", sizeof(uintptr_t) * 2, "base", sizeof(uintptr_t) * 2, "size");
    for (int i = 0; i < moduleCount; i++)
    {
        const auto& module = modules[i];
        NN_LOG("  0x%P 0x%P %s\n", module.baseAddress, module.size, module.path);

        for( int j = 0; j < backtraceSize; j++)
        {
            if( module.baseAddress < backtraceAddress[j] && module.baseAddress + module.size > backtraceAddress[j] )
            {
                backtraceOffsetAddress = backtraceAddress[j] - module.baseAddress;
                NN_LOG("[%d] 0x%X : ", j, backtraceAddress[j]);
                // オフセット値が取得できたので、このアドレスが含まれる関数名を debug_info から取得する
                SearchFunctionName(backtraceOffsetAddress, sections, br, FilePath, debug_info_len);
            }
        }
    }
    NN_LOG("\n");
    std::free(buffer);
}
