﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioSplitter.cpp,PageSampleAudioAudioSplitter}
 *
 * @brief
 * nn::audio::SplitterType の利用方法を解説するサンプルです。
 */

/**
 * @page PageSampleAudioAudioSplitter SplitterType の解説サンプル
 * @tableofcontents
 *
 * @brief
 * nn::audio::SplitterType の利用方法を解説するサンプルです。
 *
 * @section PageSampleAudioAudioSplitter_SectionBrief 概要
 * nn::audio::SplitterType を利用して、 オーディオグラフ構造を構築するサンプルです。
 *
 * @section PageSampleAudioAudioSplitter_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioSplitter
 * Samples/Sources/Applications/AudioSplitter @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioSplitter_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioSplitter_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に BGM や SE が再生されます。
 * 再生中はキーボードや DebugPad による入力を用いて、再生される BGM や SE、サイン波を操作できます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [Left][Right]   </td><td> Splitter から出力先へのミックスボリュームを調整 </td></tr>
 * <tr><td> [A]             </td><td> Splitter の出力先を変更                     </td></tr>
 * <tr><td> [B]             </td><td> Splitter を破棄・再追加する                  </td></tr>
 * <tr><td> [X]             </td><td> 説明文を印字                                </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                             </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioSplitter_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioSplitter_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioSplitter_SectionDetail 解説
 * このサンプルプログラムは nn::audio::SplitterType を用いて、
 * 単一の出力元を、複数の出力先に送付し複雑なルーティングを実現するための
 * 基本的な方法を解説するものです。
 *
 * サンプルプログラムでは以下の解説しています。
 * #0 サンプルの概要
 *    このサンプルプログラムで構成するグラフ構造を示しています。
 *    ここで示す構造を元に AudioRendererParameter をはじめとする、各種パラメータの設定を行います。
 *
 * #1 SplitterType を利用する準備
 *    AudioRendererParameter の設定について解説します。
 *    SplitterType を利用するには、 splitterCount および splitterSendChannelCount に値を設定する必要があります。
 *
 * #2 SplitterType の接続
 *    #0 で示したしたグラフ構造を構築します。
 *    SplitterType の接続方法などの、基本的な手順はここで解説します。
 *
 * また以下の 3 項目では、それぞれ SplitterType の設定を変更する方法を解説しています。
 *
 * #3 SplitterType の接続先切り替え
 *
 * #4 SplitterType のミックスボリューム調整
 *
 * #5 SplitterType の除去
 */

#include <cstdlib>
#include <cmath>
#include <new>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

const char Title[] = "AudioSplitter";

// - and add / remove them to / from the files lists
const char* g_BgmFileNames = "asset:/AudioSplitter/SampleBgm0-2ch.wav";

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[14 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

float clamp(float val, float min, float max)
{
    return (val <= min) ? min :
           (val >= max) ? max : val;
}

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA = nn::hid::KeyboardKey::A::Index;
    map.buttonB = nn::hid::KeyboardKey::B::Index;
    map.buttonX = nn::hid::KeyboardKey::X::Index;
    map.buttonY = nn::hid::KeyboardKey::Y::Index;
    map.buttonL = nn::hid::KeyboardKey::L::Index;
    map.buttonR = nn::hid::KeyboardKey::R::Index;
    map.buttonZL = nn::hid::KeyboardKey::U::Index;
    map.buttonZR = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void GetButtonState(nn::hid::NpadButtonSet& npadButtonCurrent, nn::hid::NpadButtonSet& npadButtonDown, nn::hid::AnalogStickState &analogStickStateL, nn::hid::AnalogStickState &analogStickStateR)
{
    // Npad の入力を取得します。
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
    {
        static nn::hid::NpadFullKeyState npadFullKeyState = {};
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        npadButtonCurrent |= state.buttons;
        npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
        analogStickStateL.x += state.analogStickL.x;
        analogStickStateL.y += state.analogStickL.y;
        analogStickStateR.x += state.analogStickR.x;
        analogStickStateR.y += state.analogStickR.y;
        npadFullKeyState = state;
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        static nn::hid::NpadHandheldState npadHandheldState = {};
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        npadButtonCurrent |= state.buttons;
        npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
        analogStickStateL.x += state.analogStickL.x;
        analogStickStateL.y += state.analogStickL.y;
        analogStickStateR.x += state.analogStickR.x;
        analogStickStateR.y += state.analogStickR.y;
        npadHandheldState = state;
    }

    // DebugPad の入力を取得します。
    {
        static nn::hid::DebugPadState debugPadState = {};
        nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
        nn::hid::DebugPadButtonSet debugPadButtonDown = {};
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        debugPadButtonCurrent |= state.buttons;
        debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
        analogStickStateL.x += state.analogStickL.x;
        analogStickStateL.y += state.analogStickL.y;
        analogStickStateR.x += state.analogStickR.x;
        analogStickStateR.y += state.analogStickR.y;
        debugPadState = state;
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
    }
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[Left/Right]    Update mix volume                       \n");
    NNS_LOG("[A]             Change Splitter destination             \n");
    NNS_LOG("[B]             Remove / Re-Add splitter                \n");
    NNS_LOG("[X]             Print this message                      \n");
    NNS_LOG("[Start/Space]   Shut down sample program                \n");
    NNS_LOG("--------------------------------------------------------\n");
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    //////////////////////////////////////////////////////////////////////////
    // #0 サンプルの概要
    //
    // nn::audio::SplitterType を利用したオーディオレンダラの利用方法を解説します。
    //
    // SplitterType は Voice/SubMix からの入力を 1 つ受け取り、
    // その入力内容を複数の宛先に出力することができます。
    // 出力先の総数は nn::audio::AcquireSplitter() で取得する際に決定され、
    // すべての出力先は同一のサンプルレートである必要があります。
    //
    // このサンプルでは、例として以下のグラフ構造の構築手順を解説します。
    //
    // [voice(2ch)] ---> (splitter) ---+---> [subMix1(subBus1 2ch)] ---+---> [finalMix(mainBus 2ch)] ---> [deviceSink]
    //                                 |        L(Delay)               |
    //                                 |                               |
    //                                 +---> [subMix2(subBus2 2ch)] ---+
    //                                          L(Reverb)
    //
    // またこのサンプルでは SplitterType への入力に Voice のみ扱っていますが
    // SubMixType も同様に SplitterType への入力として指定することが可能です。
    //////////////////////////////////////////////////////////////////////////
    const int SubMixCount = 2;
    const int SubMixChannelCount = 2;
    const int SubMixMixBufferCount = SubMixChannelCount * SubMixCount;
    const int FinalMixChannelCount = 2;
    const int FinalMixMixBufferCount = FinalMixChannelCount;
    const int VoiceChannelCount = 2;
    const int EffectCount = 2; // Delay, Reverb

    int8_t mainBus[FinalMixChannelCount];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;
    int8_t subBus1[SubMixChannelCount];
    subBus1[nn::audio::ChannelMapping_FrontLeft] = 0;
    subBus1[nn::audio::ChannelMapping_FrontRight] = 1;
    int8_t subBus2[SubMixChannelCount];
    subBus2[nn::audio::ChannelMapping_FrontLeft] = 0;
    subBus2[nn::audio::ChannelMapping_FrontRight] = 1;

    //////////////////////////////////////////////////////////////////////////
    // #1 SplitterType を利用する準備
    //
    // Splitter を利用する場合は nn::audio::AudioRendererParameter の splitterCount と splitterSendChannelCount を
    // 適切な値に設定する必要があります。
    // splitterCount にはオーディオレンダラで利用する SplitterType の総数を指定します。
    // splitterSendChannelCount には
    // Splitter への入力チャンネル数と、その splitter からの出力チャンネル数を掛け合わせた数を指定します。
    //////////////////////////////////////////////////////////////////////////
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    // 利用する SplitterType の総数を指定します。
    parameter.splitterCount = 1;
    // splitterSendChannelCount には SplitterType から出力されるチャンネル数の総数を指定します。
    // このサンプルでは SplitterType への入力は 2ch 、総出力チャンネル数は 4ch です。
    // よってここでは 2 * 4 = 8 となります。
    // 複数の SplitterType を利用する場合は、それらの SplitterType で利用するチャンネル数も加算する必要があります。
    parameter.splitterSendChannelCount = 8; // (input channel = 2) * (total output channel = 4)
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.voiceCount = VoiceChannelCount;
    parameter.sinkCount = 1;
    parameter.subMixCount = SubMixCount;
    parameter.mixBufferCount = SubMixMixBufferCount + FinalMixMixBufferCount;
    parameter.effectCount = EffectCount;
    NN_ABORT_UNLESS(
            nn::audio::IsValidAudioRendererParameter(parameter),
            "Invalid AudioRendererParameter specified.");

    // オーディオレンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    // オーディオレンダラを初期化します。
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer");

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    {
        NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory)));
        NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool));
    }
    nn::audio::MemoryPoolType effectMemoryPool;
    {
        NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &effectMemoryPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory)));
        NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&effectMemoryPool));
    }


    //////////////////////////////////////////////////////////////////////////
    // #2 Splitter の接続
    //
    // 上記したグラフ構造を構築します。
    // まず FinalMix, SubMix, DeviceSink, Effect の構築を行い、
    // その後、Voice と Splitter の接続を行います。
    /////////////////////////////////////////////////////////////////////////////
    nn::audio::FinalMixType finalMix;
    nn::audio::SubMixType subMix1;
    nn::audio::SubMixType subMix2;
    nn::audio::DeviceSinkType deviceSink;
    {
        // FinalMix, SubMix, DeviceSink の構築を行います。
        // これらの処理の詳細は AudioRenderer サンプルをご参照ください。
        NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&config, &finalMix, FinalMixChannelCount));
        NN_ABORT_UNLESS(nn::audio::AcquireSubMix(&config, &subMix1, parameter.sampleRate, SubMixChannelCount));
        NN_ABORT_UNLESS(nn::audio::AcquireSubMix(&config, &subMix2, parameter.sampleRate, SubMixChannelCount));
        nn::audio::SetSubMixDestination(&config, &subMix1, &finalMix);
        nn::audio::SetSubMixDestination(&config, &subMix2, &finalMix);
        for (auto i = 0; i < SubMixChannelCount; ++i) // mix 間の mix volume を指定します。
        {
            nn::audio::SetSubMixMixVolume(&subMix1, &finalMix, 0.707f / 2, subBus1[i], mainBus[i]);
            nn::audio::SetSubMixMixVolume(&subMix2, &finalMix, 0.707f / 2, subBus2[i], mainBus[i]);
        }
        nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, FinalMixChannelCount, "MainAudioOut");
        NN_ABORT_UNLESS(result.IsSuccess());
    }

    // subMix1 および subMix2 に、 Delay および Reverb をそれぞれ追加します。
    nn::audio::DelayType delay;
    nn::audio::ReverbType reverb;
    void* delayBuffer = nullptr;
    void* reverbBuffer = nullptr;
    {
        const nn::TimeSpan delayMaxTime = nn::TimeSpan::FromSeconds(1);
        size_t delayBufferSize = nn::audio::GetRequiredBufferSizeForDelay(delayMaxTime, nn::audio::GetSubMixSampleRate(&subMix1), nn::audio::GetSubMixBufferCount(&subMix1));
        size_t reverbBufferSize = nn::audio::GetRequiredBufferSizeForReverb(nn::audio::GetSubMixSampleRate(&subMix2), nn::audio::GetSubMixBufferCount(&subMix2));
        delayBuffer = g_EffectBufferAllocator.Allocate(delayBufferSize, nn::audio::BufferAlignSize);
        reverbBuffer = g_EffectBufferAllocator.Allocate(reverbBufferSize, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(delayBuffer);
        NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);

        nn::audio::AddDelay(&config, &delay, delayBuffer, delayBufferSize, &subMix1, delayMaxTime, nn::audio::GetSubMixBufferCount(&subMix1));
        nn::audio::SetDelayInputOutput(&delay, subBus1, subBus1, sizeof(subBus1));
        nn::audio::SetDelayTime(&delay, nn::TimeSpan::FromMilliSeconds(80));
        nn::audio::SetDelayChannelSpread(&delay, 0.8f);
        nn::audio::SetDelayDryGain(&delay, 0.4f);
        nn::audio::SetDelayFeedbackGain(&delay, 0.7f);
        nn::audio::SetDelayInGain(&delay, 1.0f);
        nn::audio::SetDelayLowPassAmount(&delay, 0.1f);
        nn::audio::SetDelayEnabled(&delay, true);

        nn::audio::AddReverb(&config, &reverb, reverbBuffer, reverbBufferSize, &subMix2, nn::audio::GetSubMixBufferCount(&subMix2));
        nn::audio::SetReverbInputOutput(&reverb, subBus2, subBus2, sizeof(subBus2));
        nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
        nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
        nn::audio::SetReverbPredelayTime(&reverb, nn::TimeSpan::FromMilliSeconds(20));
        nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
        nn::audio::SetReverbLateGain(&reverb, 0.8f);
        nn::audio::SetReverbDecayTime(&reverb, nn::TimeSpan::FromSeconds(3));
        nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.3f);
        nn::audio::SetReverbColoration(&reverb, 0.3f);
        nn::audio::SetReverbReverbGain(&reverb, 0.8f);
        nn::audio::SetReverbOutGain(&reverb, 0.9f);
        nn::audio::SetReverbDryGain(&reverb, 0.8f);
        nn::audio::SetReverbEnabled(&reverb, true);
    }

    // ここまでの処理で、以下の構成となっています。
    //
    // [voice(2ch)]      (splitter)          [subMix1(subBus1 2ch)] ---+---> [finalMix(mainBus 2ch)] ---> [deviceSink]
    //                                          L(Delay)               |
    //                                                                 |
    //                                       [subMix2(subBus2 2ch)] ---+
    //                                          L(Reverb)
    //
    // 次に VoiceType と SplitterType をグラフに追加します。
    nn::audio::SplitterType splitter;
    const int destIndex1 = 0;
    const int destIndex2 = 1;
    {
        // SplitterType を取得します。
        // SplitterType は入力元のサンプルデータを、複数の出力先に出力します。
        // ただし SplitterType はサンプルレート変換を行わないため、接続されるすべての出力は同一のサンプルレートを保持している必要があります。
        // 接続する SubMix / FinalMix のサンプルレートは、 nn::audio::AcquireSplitter() の sampleRate 引数で指定します。
        // ここで指定した sampleRate は nn::audio::GetSplitterSampleRate() で取得することができます。
        // このサンプルでは subMix1 と subMix2 のサンプルレートである parameter.sampleRate を指定します。
        //
        // また SplitterType への入力チャンネルは nn::audio::AcquireSplitter() の sourceChannelCount 引数で指定されます。
        // SplitterType はここで指定したチャンネル数以下の入力を受け付けることができます。
        // ここで指定した sourceChannelCount は、 nn::audio::GetSplitterSourceChannelCount() で取得することができます。
        // このサンプルでは入力元である voice は 2ch であるため VoiceChannelCount = 2 を指定します。
        //
        // 最後に SplitterType は複数の出力先を持つことができます。
        // 接続可能な出力先の数は nn::audio::AcquireSplitter() で取得する時に destinationCount の数で決定されます。
        // 出力先は subMix1, subMix2 の 2 つであるため SubMixCount = 2 を指定します。
        NN_ABORT_UNLESS(nn::audio::AcquireSplitter(&config, &splitter, parameter.sampleRate, VoiceChannelCount, SubMixCount));
        // 出力先はインデックスによって管理されます。このサンプルでは destinationCount に 2 を指定しているため、
        // SplitterType はインデックス 0 とインデックス 1 の、 2 つの出力先を持っています。
        // ここでは splitter のインデックス 0 に subMix1 を、インデックス 1 に subMix2 を、それぞれ接続します。
        nn::audio::SetSplitterDestination(&config, &splitter, destIndex1, &subMix1);
        nn::audio::SetSplitterDestination(&config, &splitter, destIndex2, &subMix2);
        // splitter から出力先へのミックスボリュームを設定します。
        // nn::audio::SetSplitterMixVolume() の destinationIndex と pDestination 引数には、
        // nn::audio::SetSplitterDestination() の destinationIndex と pDestination 引数に、同じ組み合わせを指定する必要があります。
        for (auto i = 0; i < nn::audio::GetSplitterSourceChannelCount(&splitter); ++i)
        {
            nn::audio::SetSplitterMixVolume(&splitter, destIndex1, &subMix1, 0.5f, i, subBus1[i]);
            nn::audio::SetSplitterMixVolume(&splitter, destIndex2, &subMix2, 0.5f, i, subBus2[i]);
        }
    }
    // Voice で再生する波形データを読み込みます。
    nn::audio::VoiceType voice;
    nn::audio::WaveBuffer waveBuffer;
    void* dataBgm = nullptr;
    {
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &dataBgm, g_BgmFileNames);
        nn::audio::AcquireVoiceSlot(&config, &voice, format.sampleRate, format.channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceVolume(&voice, 0.7f);
        nn::audio::SetVoiceDestination(&config, &voice, &splitter);

        waveBuffer.buffer = dataBgm;
        waveBuffer.size = dataSize;
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;
        waveBuffer.pContext = nullptr;
        waveBuffer.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    }

    // ここまでの処理で、以下の構成となりました。
    //
    // [voice(2ch)] ---> (splitter) ---+---> [subMix1(subBus1 2ch)] ---+---> [finalMix(mainBus 2ch)] ---> [deviceSink]
    //                                          L(Delay)               |
    //                                 |                               |
    //                                 +---> [subMix2(subBus2 2ch)] ---+
    //                                          L(Reverb)
    //
    // 設定したパラメータを反映し、オーディオレンダラをスタートします。
    NN_ABORT_UNLESS(nn::audio::RequestUpdateAudioRenderer(handle, &config).IsSuccess());
    NN_ABORT_UNLESS(nn::audio::StartAudioRenderer(handle).IsSuccess());

    PrintUsage();

    for (;;)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};
        GetButtonState(npadButtonCurrent, npadButtonDown, analogStickStateL, analogStickStateR);

        //////////////////////////////////////////////////////////////////////////
        // #3 SplitterType の接続先切り替え
        //
        // SplitterType の接続先は、オーディオレンダラ動作中に切り替えることが可能です。
        // 出力先は nn::audio::SetSplitterDestination() で設定します。
        //
        // ここでは splitter のインデックス 0 に割り当てられた出力先を、変更することで、
        // 上述のグラフ構造と以下のグラフ構造とを交互に切り替える処理を行います。
        //
        //                                 +-----------------------------------------+
        //                                 |                                         |
        //                                 |                                         v
        // [voice(2ch)] ---> (splitter) ---+     [subMix1(subBus1 2ch)] ---+---> [finalMix(mainBus 2ch)] ---> [deviceSink]
        //                                 |        L(Delay)               |
        //                                 |                               |
        //                                 +---> [subMix2(subBus2 2ch)] ---+
        //                                          L(Reverb)
        //
        // splitter の出力先の 1 つが finalMix になることで、 Delay のつながった subMix1 への入力が無くなるため、
        // Delay エフェクトの効いた音が出力されなくなるのを確認できます。
        //////////////////////////////////////////////////////////////////////////
        static bool SplitterConnected = true;
        static bool connectedToSubMix1 = true;
        if (npadButtonDown.Test< ::nn::hid::NpadButton::A >() && SplitterConnected)
        {
            if (connectedToSubMix1)
            {
                // Change connection to finalMix
                nn::audio::SetSplitterDestination(&config, &splitter, destIndex1, &finalMix);
            }
            else
            {
                // Change connection to subMix1
                nn::audio::SetSplitterDestination(&config, &splitter, destIndex1, &subMix1);
            }
            connectedToSubMix1 = !connectedToSubMix1;
            NNS_LOG("Splitter destination 1 is connected to \"%s\"\n", connectedToSubMix1 ? "subMix1" : "finalMix");
        }

        //////////////////////////////////////////////////////////////////////////
        // #4 SplitterType のミックスボリューム調整
        //
        // Splitter からの出力先へのミックスボリュームを調整します。
        // ミックスボリュームの指定は nn::audio::SetSplitterMixVolume() で行います。
        //
        // このサンプルでは splitter のインデックス 0 とインデックス 1 、
        // それぞれへのミックスボリュームのバランス調整を行います。
        // Delay エフェクトの効いた音と Reverb エフェクトの効いた音との混ざり具合から、
        // ミックスバランスが調整されるのを確認することができます。
        //////////////////////////////////////////////////////////////////////////
        const float volumeDelta = 0.005f;
        if ((npadButtonCurrent.Test< ::nn::hid::NpadButton::Right >() || npadButtonCurrent.Test< ::nn::hid::NpadButton::Left >())
            && SplitterConnected)
        {
            float curVol = connectedToSubMix1 ?
                nn::audio::GetSplitterMixVolume(&splitter, destIndex1, &subMix1, 0, 0) :
                nn::audio::GetSplitterMixVolume(&splitter, destIndex1, &finalMix, 0, 0);
            curVol = npadButtonCurrent.Test< ::nn::hid::NpadButton::Right >() ? curVol - volumeDelta :
                     npadButtonCurrent.Test< ::nn::hid::NpadButton::Left >() ? curVol + volumeDelta : curVol;
            curVol = clamp(curVol, 0.0f, 1.0f);
            NNS_LOG("SplitterMixVolumes %s:%.2f, submix2:%.2f\r", connectedToSubMix1 ? "subMix1" : "finalMix", curVol, 1.0f - curVol);

            for (auto i = 0; i < nn::audio::GetSplitterSourceChannelCount(&splitter); ++i)
            {
                if (connectedToSubMix1)
                {
                    nn::audio::SetSplitterMixVolume(&splitter, destIndex1, &subMix1, curVol, i, i);
                }
                else
                {
                    nn::audio::SetSplitterMixVolume(&splitter, destIndex1, &finalMix, curVol, i, i);
                }
                nn::audio::SetSplitterMixVolume(&splitter, destIndex2, &subMix2, 1.0f - curVol, i, i);
            }
        }

        //////////////////////////////////////////////////////////////////////////
        // # SplitterType の除去
        //
        // SplitterType の除去および再追加を行います。
        // SplitterType は nn::audio::ReleaseSplitter() で取り除くことが可能です。
        //
        // この例では SplitterType を取り除くと SubMix/FinalMix への入力がなくなるため、無音となります。
        //////////////////////////////////////////////////////////////////////////
        if (npadButtonDown.Test< ::nn::hid::NpadButton::B >())
        {
            if (SplitterConnected)
            {
                nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Pause);
                nn::audio::ReleaseSplitter(&config, &splitter);
            }
            else
            {
                // voice を入力として再接続します。
                // 接続方法は 「Splitter の接続」で解説した内容と同様です。
                nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
                NN_ABORT_UNLESS(nn::audio::AcquireSplitter(&config, &splitter, parameter.sampleRate, VoiceChannelCount, SubMixCount));
                nn::audio::SetSplitterDestination(&config, &splitter, destIndex1, &subMix1);
                nn::audio::SetSplitterDestination(&config, &splitter, destIndex2, &subMix2);
                for (auto i = 0; i < nn::audio::GetSplitterSourceChannelCount(&splitter); ++i)
                {
                    nn::audio::SetSplitterMixVolume(&splitter, destIndex1, &subMix1, 0.5f, i, subBus1[i]);
                    nn::audio::SetSplitterMixVolume(&splitter, destIndex2, &subMix2, 0.5f, i, subBus2[i]);
                }
                connectedToSubMix1 = true;
            }
            SplitterConnected = !SplitterConnected;
            NNS_LOG("Splitter is %s                           \n", SplitterConnected ? "connected" : "removed");
        }


        if (npadButtonDown.Test< ::nn::hid::NpadButton::X >())
        {
            PrintUsage();
        }

        if (npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break; // サンプルを終了します。
        }

        NN_ABORT_UNLESS(nn::audio::RequestUpdateAudioRenderer(handle, &config).IsSuccess());
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    if (dataBgm)
    {
        g_WaveBufferAllocator.Free(dataBgm);
        dataBgm = nullptr;
    }
    if (delayBuffer)
    {
        g_EffectBufferAllocator.Free(delayBuffer);
        delayBuffer = nullptr;
    }
    if (reverbBuffer)
    {
        g_EffectBufferAllocator.Free(reverbBuffer);
    }

    FinalizeFileSystem();

    g_WaveBufferAllocator.Finalize();
    g_EffectBufferAllocator.Finalize();
    g_Allocator.Finalize();
}  // NOLINT(readability/fn_size)
