﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioOut.cpp,PageSampleAudioAudioOut}
 *
 * @brief
 * オーディオ再生のサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioOut オーディオ再生
 * @tableofcontents
 *
 * @brief
 * オーディオ再生のサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioOut_SectionBrief 概要
 * オーディオ出力を利用して、オーディオ再生を行うサンプルです。
 *
 * @section PageSampleAudioAudioOut_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioOut
 * Samples/Sources/Applications/AudioOut @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioOut_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioOut_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、矩形波を再生されます。
 * キーボードや DebugPad による入力を用いてプログラムを操作することが出来ます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了 </td></tr>
 * <tr><td> [L][R]          </td><td> 音声出力の音量変更 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioOut_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioOut_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioOut_SectionDetail 解説
 * このサンプルプログラムは、動的に矩形波を生成し、オーディオ出力に出力するものです。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - 利用可能なオーディオ出力を列挙
 * - デフォルトのオーディオ出力をオープン
 * - オーディオ出力の情報を取得、確認
 * - 再生用バッファを準備、登録
 * - 再生を開始
 * - オーディオ再生を実行
 * - オーディオ出力の音量を変更
 * - 再生を終了
 * - オーディオ出力をクローズ
 * - 再生用バッファを破棄
 *
 * 最初にオーディオ出力を列挙しています。
 * システム上に複数の利用可能なオーディオ出力がある場合、名前を指定して
 * nn::audio::OpenAudioOut() を呼ぶことが可能です。
 * 異なるオーディオ出力を同時に利用することも可能です。
 * 本サンプルでは、名前の列挙とオープン、クローズ処理のみを行っており、
 * 実際のオーディオ再生はデフォルトのオーディオ出力を用いています。
 *
 * 再生のためのバッファは 50 ミリ秒の長さのものを 2 つ用意します。
 * バッファの長さや数については利用者側で自由に決めることができますが、短かすぎたり、
 * 数が少なすぎると、音が途切れるなどの問題が発生することがあります。
 * 用意したバッファは nn::audio::AppendAudioOutBuffer() により登録します。
 * 登録は nn::audio::StartAudioOut() の呼び出し前でも呼び出し後でも可能です。
 * 実際の再生は nn::audio::StartAudioOut() が呼ばれるまでは行われません。
 *
 * 再生開始後は nn::audio::GetReleasedAudioOutBuffer() により再生が完了したバッファ情報を取得し、
 * 動的に生成した矩形波データを再生バッファにコピーし、それぞれを再度登録します。
 * ダウンミックスやサンプルレート変換などが必要な場合は、このタイミングで行います。
 * この処理を繰り返すことでオーディオ再生を実現します。
 *
 * 音量の変更が要求されると、nn::audio::SetAudioOutVolume() によりオーディオ出力の音量の変更を行います。
 *
 * プログラムの終了が要求されると、上記の繰り返し処理を抜けて、再生を停止し、
 * オーディオ出力をクローズし、メモリの破棄を行います。
 * nn::audio::StopAudioOut() を呼ぶと、再生は停止し、登録済み（かつ未再生）バッファへのアクセスはなくなりますが、
 * 再度 nn::audio::StartAudioOut() を行うと、アクセスが発生することには注意が必要です。
 * nn::audio::CloseAudioOut() 後はアクセスが発生することはありません。
 */

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>
#include <nn/audio.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nns/audio/audio_HidUtilities.h>
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace
{

char g_HeapBuffer[128 * 1024];
const char Title[] = "AudioOut";

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("[L/R]           Change volume\n");
    NNS_LOG("-------------------------------------------------------\n");
}

//
// 状態名を返す関数です。
//
const char* GetAudioOutStateName(nn::audio::AudioOutState state)
{
    switch (state)
    {
        case nn::audio::AudioOutState_Started:
            return "Started";
        case nn::audio::AudioOutState_Stopped:
            return "Stopped";
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

//
// サンプルフォーマット名を返す関数です。
//
const char* GetSampleFormatName(nn::audio::SampleFormat format)
{
    switch (format)
    {
        case nn::audio::SampleFormat_Invalid:
            return "Invalid";
        case nn::audio::SampleFormat_PcmInt8:
            return "PcmInt8";
        case nn::audio::SampleFormat_PcmInt16:
            return "PcmInt16";
        case nn::audio::SampleFormat_PcmInt24:
            return "PcmInt24";
        case nn::audio::SampleFormat_PcmInt32:
            return "PcmInt32";
        case nn::audio::SampleFormat_PcmFloat:
            return "PcmFloat";
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

//
// nn::audio::SampleFormat_PcmInt8 に対応する矩形波生成関数です。未実装です。
//
void GenerateSquareWaveInt8(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    NN_UNUSED(buffer);
    NN_UNUSED(channelCount);
    NN_UNUSED(sampleRate);
    NN_UNUSED(sampleCount);
    NN_UNUSED(amplitude);
    NN_ABORT("Not implemented yet\n");
}

//
// nn::audio::SampleFormat_PcmInt16 に対応する矩形波生成関数です。
//
void GenerateSquareWaveInt16(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    static int s_TotalSampleCount[6] = { 0 };
    const int frequencies[6] = { 415, 698, 554, 104, 349, 277 };

    int16_t* buf = reinterpret_cast<int16_t*>(buffer);
    for (int ch = 0; ch < channelCount; ch++)
    {
        int waveLength = sampleRate / frequencies[ch]; // 1周期分の波形の長さ (サンプル数単位)

        for (int sample = 0; sample < sampleCount; sample++)
        {
            int16_t value = static_cast<int16_t>(s_TotalSampleCount[ch] < (waveLength / 2) ? amplitude : -amplitude);
            buf[sample*channelCount + ch] = value;
            s_TotalSampleCount[ch]++;
            if (s_TotalSampleCount[ch] == waveLength)
            {
                s_TotalSampleCount[ch] = 0;
            }
        }
    }
}

//
// nn::audio::SampleFormat_PcmInt24 に対応する矩形波生成関数です。未実装です。
//
void GenerateSquareWaveInt24(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    NN_UNUSED(buffer);
    NN_UNUSED(channelCount);
    NN_UNUSED(sampleRate);
    NN_UNUSED(sampleCount);
    NN_UNUSED(amplitude);
    NN_ABORT("Not implemented yet\n");
}

//
// nn::audio::SampleFormat_PcmInt32 に対応する矩形波生成関数です。未実装です。
//
void GenerateSquareWaveInt32(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    NN_UNUSED(buffer);
    NN_UNUSED(channelCount);
    NN_UNUSED(sampleRate);
    NN_UNUSED(sampleCount);
    NN_UNUSED(amplitude);
    NN_ABORT("Not implemented yet\n");
}

//
// nn::audio::SampleFormat_PcmFloat に対応する矩形波生成関数です。未実装です。
//
void GenerateSquareWaveFloat(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    NN_UNUSED(buffer);
    NN_UNUSED(channelCount);
    NN_UNUSED(sampleRate);
    NN_UNUSED(sampleCount);
    NN_UNUSED(amplitude);
    NN_ABORT("Not implemented yet\n");
}

//
// サンプルフォーマットに対応した矩形波生成関数を返します。
//
typedef void (*GenerateSquareWaveFunction)(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude);
GenerateSquareWaveFunction GetGenerateSquareWaveFunction(nn::audio::SampleFormat format)
{
    switch (format)
    {
        case nn::audio::SampleFormat_PcmInt8:
            return GenerateSquareWaveInt8;
        case nn::audio::SampleFormat_PcmInt16:
            return GenerateSquareWaveInt16;
        case nn::audio::SampleFormat_PcmInt24:
            return GenerateSquareWaveInt24;
        case nn::audio::SampleFormat_PcmInt32:
            return GenerateSquareWaveInt32;
        case nn::audio::SampleFormat_PcmFloat:
            return GenerateSquareWaveFloat;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

//
// 矩形波を生成する関数です。
//
void GenerateSquareWave(nn::audio::SampleFormat format, void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    NN_ASSERT_NOT_NULL(buffer);
    GenerateSquareWaveFunction func = GetGenerateSquareWaveFunction(format);
    if (func)
    {
        func(buffer, channelCount, sampleRate, sampleCount, amplitude);
    }
}

void* Allocate(size_t size)
{
    return std::malloc(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

}  // anonymous namespace

//
// メイン関数です。
//
extern "C" void nnMain()
{
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    nn::audio::AudioOut audioOut;

    nn::fs::SetAllocator(Allocate, Deallocate);
    InitializeHidDevices();
    int timeout = 0; //arg, for which the default is 0
    char** argvs = nn::os::GetHostArgv();
    for(int i = 0; i < nn::os::GetHostArgc(); ++i)
    {
        if(strcmp("timeout", argvs[i]) == 0)
        {
            if(i < nn::os::GetHostArgc())
                timeout = atoi(argvs[i + 1]);
        }
    }

    nn::TimeSpan endTime = nn::os::GetSystemTick().ToTimeSpan() + nn::TimeSpan::FromSeconds(timeout);
    // 利用可能なオーディオ出力のリストを取得します。
    {
        NNS_LOG("Available AudioOuts:\n");
        nn::audio::AudioOutInfo audioOutInfos[nn::audio::AudioOutCountMax];
        const int count = nn::audio::ListAudioOuts(audioOutInfos, sizeof(audioOutInfos) / sizeof(*audioOutInfos));
        for (int i = 0; i < count; ++i)
        {
            NNS_LOG("  %2d: %s\n", i, audioOutInfos[i].name);

            // オープン、クローズができることを確認します。
            nn::audio::AudioOutParameter parameter;
            nn::audio::InitializeAudioOutParameter(&parameter);
            NN_ABORT_UNLESS(
                nn::audio::OpenAudioOut(&audioOut, audioOutInfos[i].name, parameter).IsSuccess(),
                "Failed to open AudioOut."
            );
            nn::audio::CloseAudioOut(&audioOut);
        }
        NNS_LOG("\n");
    }

    // オーディオ出力をサンプルレートおよびチャンネル数を指定してオープンします。
    // 指定したサンプルレートがサポートされておらず、処理に失敗した場合はデフォルト値を利用します。
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.channelCount = 2;
    // parameter.channelCount = 6;  // 5.1ch 出力を利用する場合はチャンネル数に 6 を指定します。
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsFailure())
    {
        parameter.sampleRate = 0;
        parameter.channelCount = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsSuccess(),
            "Failed to open AudioOut."
        );
    }

    NNS_LOG("AudioOut is opened\n  State: %s\n", GetAudioOutStateName(nn::audio::GetAudioOutState(&audioOut)));

    // オーディオ出力の各種プロパティを取得します。
    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NNS_LOG("  Name: %s\n", nn::audio::GetAudioOutName(&audioOut));
    NNS_LOG("  ChannelCount: %d\n", channelCount);
    NNS_LOG("  SampleRate: %d\n", sampleRate);
    NNS_LOG("  SampleFormat: %s\n", GetSampleFormatName(sampleFormat));
    // このサンプルでは、サンプルフォーマットが 16bit であることを前提とします。
    NN_ASSERT(sampleFormat == nn::audio::SampleFormat_PcmInt16);

    // バッファに関するパラメータを準備します。
    const int frameRate = 20;                             // 20fps
    const int frameSampleCount = sampleRate / frameRate;  // 50msecs (in samples)
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const int bufferCount = 4;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;

    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    void* outBuffer[bufferCount];
    for (int i = 0; i < bufferCount; ++i)
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        GenerateSquareWave(sampleFormat, outBuffer[i], channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    // 再生を開始します。
    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
    );
    NNS_LOG("AudioOut is started\n  State: %s\n", GetAudioOutStateName(nn::audio::GetAudioOutState(&audioOut)));

    // 1 フレーム待ちます。
    const nn::TimeSpan interval(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / frameRate));
    nn::os::SleepThread(interval);

    PrintUsage();

    // オーディオ再生を行います。
    NNS_LOG("Start audio playback\n");

    for(;;)
    {
        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        // ボリュームを変更します。
        const auto volumeStep = 0.1f;
        const auto currentVolume = nn::audio::GetAudioOutVolume(&audioOut);
        auto updateVolume = currentVolume;

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::L >())
        {
            updateVolume = std::max(nn::audio::AudioOut::GetVolumeMin(), currentVolume - volumeStep);
        }

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::R >())
        {
            updateVolume = std::min(nn::audio::AudioOut::GetVolumeMax(), currentVolume + volumeStep);
        }

        if(updateVolume != currentVolume)
        {
            nn::audio::SetAudioOutVolume(&audioOut, updateVolume);
            NNS_LOG("Volume: %.2f\n", updateVolume);
        }

        //timeout condition
        if(timeout > 0 && (nn::os::GetSystemTick().ToTimeSpan() > endTime))
        {
            break;
        }
        systemEvent.Wait();

        // 再生が完了したバッファを取得します。
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        while (pAudioOutBuffer)
        {
            // 矩形波データを生成し、再度登録します。
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            NN_ASSERT(outSize == frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat));
            GenerateSquareWave(sampleFormat, pOutBuffer, channelCount, sampleRate, frameSampleCount, amplitude);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        }
    }

    NNS_LOG("Stop audio playback\n");

    // 再生を停止します。
    nn::audio::StopAudioOut(&audioOut);
    NNS_LOG("AudioOut is closed\n  State: %s\n", GetAudioOutStateName(nn::audio::GetAudioOutState(&audioOut)));

    // オーディオ出力をクローズします。
    nn::audio::CloseAudioOut(&audioOut);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを破棄します。
    for (int i = 0; i < bufferCount; ++i)
    {
        allocator.Free(outBuffer[i]);
    }

    return;
} // NOLINT(readability/fn_size)
