﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioMemoryPool.cpp,PageSampleAudioAudioMemoryPool}
 *
 * @brief
 * オーディオメモリプールのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioMemoryPool オーディオメモリプールのサンプル
 * @tableofcontents
 *
 * @brief
 * オーディオメモリプールのサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionBrief 概要
 * オーディオメモリプールを利用して、オーディオ再生を行うサンプルです。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioMemoryPool
 * Samples/Sources/Applications/AudioMemoryPool @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると BGM の再生を開始します。またボタン操作に応じて SE を再生します。
 *
 * このサンプルは nn::audio::MemoryPoolType の利用方法を説明することを目的としています。
 * 基本となるオーディオレンダラの使い方を事前に理解していることを前提としています。
 * オーディオレンダラの詳しい利用例は @link ../../../Samples/Sources/Applications/AudioRenderer
 * Samples/Sources/Applications/AudioRenderer @endlink を参照してください。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> BGM の再生状態をトグルする           </td></tr>
 * <tr><td> [X]             </td><td> SE 0 を再生する                    </td></tr>
 * <tr><td> [Y]             </td><td> SE 1 を再生する                    </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                    </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioMemoryPool_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioMemoryPool_SectionDetail 解説
 * このサンプルプログラムは nn::audio::MemoryPoolType の利用方法を解説するものです。
 * SE の再生、およびストリーム再生（波形データをファイルからメモリに逐次読み込みながら再生する）の 2 つの方法を例にとり解説します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムおよび HID の初期化
 * - オーディオレンダラを最低限の構成で初期化し、処理を開始する
 * - SE 用のメモリプールの準備と、再生用 Voice の準備
 * - ストリーム再生用のメモリプールの準備と、ストリーム再生に用いるファイルの準備
 * - ストリーム再生用 WaveBuffer の更新
 * - メモリプールの破棄
 *
 * メモリプール (nn::audio::MemoryPoolType) は「オーディオレンダラが利用するメモリ領域」を表現するための構造体です。
 * メモリプールの初期化時には、nn::audio::AcquireMemoryPool() の引数としてメモリ領域を渡します。この時渡す領域をオーディオレンダラは利用することができます。
 * よってオーディオレンダラが利用するデータで、メモリプールに置かれる必要があるものはすべて、この領域内に配置されている必要があります。
 * メモリプールに配置しなければならないデータは、具体的には以下のものが該当します。
 * - nn::audio::WaveBuffer.buffer および nn::audio::WaveBuffer.pContext で指定するバッファ
 * - nn::audio::AcquireVoiceSlot() の pParameter として指定するバッファ
 * - nn::audio::AddDelay()、 nn::audio::AddReverb()、 nn::audio::AddI3dl2Reverb()、nn::audio::AddAux() のエフェクト追加関数で buffer として指定するバッファ
 * - nn::audio::AddCircularBufferSink() で buffer に指定するバッファ
 *
 * メモリプールには「アタッチ要求状態」「アタッチ状態」「デタッチ要求状態」「デタッチ状態」という 4 種類の状態が存在します。
 * オーディオレンダラからの参照は「アタッチ状態」の時のみ可能です。
 * メモリプールの初期化直後は「デタッチ状態」ですから、必要なデータのメモリに配置後「アタッチ状態」に設定する必要があります。
 * 具体的な設定の流れはこのプログラムで解説します。
 *
 * またメモリプールが「アタッチ状態」にある最中に、それが保持するメモリ領域へはアクセスしないでください。
 * nn::fs もしくは nn::audio::CopyMemoryPoolData() を利用する場合に限り、メモリ領域へはアクセスが可能です。
 * ただしこの場合においても、オーディオレンダラが利用している最中のデータへのアクセスは避けてください。
 */

#include <cstdlib>
#include <cmath>
#include <queue>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BytePtr.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int BgmCount = 1;
const int SeCount = 2;

// choose number of files to play
const char Title[] = "AudioMemoryPool";

// - and add / remove them to / from the files lists
const char* g_BgmFileNames[BgmCount] =
{
    "asset:/AudioMemoryPool/SampleBgm0-1ch.wav",
};
const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;

char* g_MountRomCacheBuffer = NULL;
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             Toggle streaming sound PlayState\n");
    NNS_LOG("[X]             Play SE 0\n");
    NNS_LOG("[Y]             Play SE 1\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

nn::hid::NpadButtonSet GetPadButtonDown()
{
    nn::hid::NpadButtonSet npadButtonDown = {};

    // Npad の入力を取得します。
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
    {
        static nn::hid::NpadFullKeyState npadFullKeyState = {};
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
        npadFullKeyState = state;
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        static nn::hid::NpadHandheldState npadHandheldState = {};
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
        npadHandheldState = state;
    }

    // DebugPad の入力を取得します。
    {
        static nn::hid::DebugPadState debugPadState = {};
        nn::hid::DebugPadButtonSet debugPadButtonDown = {};
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
        debugPadState = state;
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
    }

    return npadButtonDown;
}

class StreamUtility
{
    nn::fs::FileHandle handle;

    nns::audio::WavFormat format;
    int64_t fileSize;
    size_t readOffset;

public:
    void Open(const char* filename)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
        readOffset = 0;

        // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
        const std::size_t WavHeaderDataSize = 1024;
        uint8_t tmpBuffer[WavHeaderDataSize];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, tmpBuffer, WavHeaderDataSize));

        nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(&format, tmpBuffer, WavHeaderDataSize);
        NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
        NN_ABORT_UNLESS_EQUAL(format.channelCount, 1);    // このサンプルでは 1ch を仮定しています
        NN_ABORT_UNLESS_EQUAL(format.bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています
    }

    void Close()
    {
        nn::fs::CloseFile(handle);
        handle.handle = nullptr;
        fileSize = 0;
        readOffset = 0;
        memset(&format, 0, sizeof(nns::audio::WavFormat));
    }

    std::size_t Read(void* buffer, size_t size)
    {
        NN_ABORT_UNLESS_NOT_NULL(buffer);

        const size_t dataSize = static_cast<size_t>(this->fileSize - this->format.dataOffset);
        const size_t dataOffset = static_cast<size_t>(this->format.dataOffset);
        size_t offset = this->readOffset;

        std::size_t read = 0;
        memset(buffer, 0xff, size);
        for (std::size_t readSize = 0; readSize < size; readSize += read)
        {
            read = std::min(dataSize - offset, size - readSize);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(this->handle, dataOffset + offset, nn::util::BytePtr(buffer, readSize).Get(), read));
            offset += read;
            if (offset >= dataSize)
            {
                offset = 0;
            }
        }

        readOffset = offset;
        return size;
    }

    int GetSampleRate() const
    {
        return format.sampleRate;
    }

    int GetChannelCount() const
    {
        return format.channelCount;
    }
};

struct StreamWaveBuffer
{
    nn::audio::WaveBuffer waveBuffer;
    void* bufferAddress;
    size_t bufferSize;
};

void ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* pOutHeader, int64_t* pOutFileSize, size_t* pOutDataSize, void* buffer, size_t bufferSize, const char* filename)
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read));

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
    NN_ABORT_UNLESS(static_cast<size_t>(size) < bufferSize);

    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, sizeof(adpcmheader), buffer, static_cast<size_t>(size) - sizeof(adpcmheader)));

    nn::fs::CloseFile(handle);
    nn::audio::ParseAdpcmHeader(pOutHeader, adpcmheader, sizeof(adpcmheader));

    *pOutFileSize = size;
    *pOutDataSize = static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void LoadAdpcmBuffers(
    nn::audio::AdpcmHeaderInfo** ppOutAdpcmHeaders,
    nn::audio::WaveBuffer * pOutWaveBuffers,
    int seCount,
    nn::audio::MemoryPoolType* pPool)
{
    // seCount 個の ADPCM 波形データを
    // pPool で指定されたメモリプールの先頭に、順番にロードします。
    nn::util::BytePtr base(nn::audio::GetMemoryPoolAddress(pPool));
    nn::util::BytePtr poolAEndAddress(nn::audio::GetMemoryPoolAddress(pPool), nn::audio::GetMemoryPoolSize(pPool));
    for (int i = 0; i < seCount; ++i)
    {
        size_t dataSize = 0;
        int64_t fileSize = 0;
        const char* seName = g_SeFileNames[i];
        nn::audio::AdpcmHeaderInfo** pHeader = &ppOutAdpcmHeaders[i];
        *pHeader = base.Get<nn::audio::AdpcmHeaderInfo>();
        base += nn::util::align_up(sizeof(nn::audio::AdpcmHeaderInfo), nn::audio::BufferAlignSize);

        ReadAdpcmFile(*pHeader, &fileSize, &dataSize, base.Get(), base.Distance(poolAEndAddress.Get()), seName);
        pOutWaveBuffers[i].buffer = base.Get();
        pOutWaveBuffers[i].size = dataSize;
        pOutWaveBuffers[i].startSampleOffset = 0;
        pOutWaveBuffers[i].endSampleOffset = (*pHeader)->sampleCount;
        pOutWaveBuffers[i].loop = false;
        pOutWaveBuffers[i].isEndOfStream = false;
        pOutWaveBuffers[i].pContext = &(*pHeader)->loopContext;
        pOutWaveBuffers[i].contextSize = sizeof(nn::audio::AdpcmContext);
        base += static_cast<ptrdiff_t>(nn::util::align_up(fileSize, nn::audio::BufferAlignSize));
    }
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    InitializeFileSystem();
    InitializeHidDevices();

    //////////////////////////////////////////////////////////////////////////
    // オーディオレンダラの初期化
    //
    // オーディオレンダラを基本的な初期化を行います。
    // オーディオレンダラの詳しい利用例は @link ../../../Samples/Sources/Applications/AudioRenderer
    // Samples/Sources/Applications/AudioRenderer @endlink を参照してください。
    //////////////////////////////////////////////////////////////////////////
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2; // FinalMix(2)
    parameter.voiceCount = 6;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(parameter), "Invalid AudioRendererParameter specified.");

    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize));

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, channelCount);

    nn::audio::DeviceSinkType deviceSink;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut"));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

    //////////////////////////////////////////////////////////////////////////
    // SE 用のメモリプールの準備と、再生用 Voice の準備
    //
    // SE 波形を配置するメモリプールを準備します。
    // メモリプールは利用数に応じてシステムにオーバーヘッドが生じます。
    // よって関連するデータはまとめて１つのメモリプールに配置するなどし、
    // メモリプールの数は少なく保つことが推奨されます。
    //
    // このサンプルでは 1MB のサイズを持つ 1 つのメモリプールに 2 つの SE 波形をロードし、
    // それぞれを nn::audio::WaveBuffer に設定して再生を行います。
    //////////////////////////////////////////////////////////////////////////

    nn::audio::MemoryPoolType memoryPoolForSe;
    // メモリプールで保持するメモリ領域は、アドレス・サイズともにアライン要求があります。
    // - アドレス：nn::audio::MemoryPoolType::AddressAlignment
    // - サイズ：nn::audio::MemoryPoolType::SizeGranularity
    const size_t poolBufferSize = 1 * 1024 * 1024;
    NN_ABORT_UNLESS(nn::util::is_aligned(poolBufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    void* sePoolAddress = g_Allocator.Allocate(poolBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(sePoolAddress);
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &memoryPoolForSe, sePoolAddress, poolBufferSize));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&memoryPoolForSe));

    // 準備したメモリプールの中に複数の波形データを読み込みます。
    nn::audio::AdpcmHeaderInfo* adpcmHeader[SeCount];
    nn::audio::WaveBuffer seWaveBuffers[SeCount];
    LoadAdpcmBuffers(adpcmHeader, seWaveBuffers, SeCount, &memoryPoolForSe);

    // Voice を初期化し、波形データを追加します。
    nn::audio::VoiceType seVoices[SeCount];
    for (int i = 0; i < SeCount; ++i)
    {
        nn::audio::VoiceType* pVoice = &seVoices[i];
        nn::audio::WaveBuffer* pWaveBuffer = &seWaveBuffers[i];
        nn::audio::AdpcmHeaderInfo* pHeader = adpcmHeader[i];
        NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(
            &config,
            pVoice,
            pHeader->sampleRate,
            1,
            nn::audio::SampleFormat_Adpcm,
            nn::audio::VoiceType::PriorityHighest,
            &pHeader->parameter,
            sizeof(nn::audio::AdpcmParameter)));
        nn::audio::SetVoiceDestination(&config, pVoice, &finalMix);
        nn::audio::SetVoicePlayState(pVoice, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(pVoice, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(pVoice, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontRight]);
        nn::audio::AppendWaveBuffer(pVoice, pWaveBuffer);
    }

    //////////////////////////////////////////////////////////////////////////
    // ストリーム再生用のメモリプールの準備と、ストリーム再生に用いるファイルの準備
    //
    // BGM をストリーム再生します。
    // メモリプールが管理するメモリ領域へのアクセスは、
    // 基本的に「デタッチ状態」にある場合にのみ、安全にアクセスできます。
    // ただし nn::fs によるアクセスおよび nn::audio::CopyMemoryPoolData() を利用した場合のみ、
    // 「アタッチ状態」であっても安全にアクセス可能となっています。
    //
    // ここではサンプルでは 1 つのメモリプールに、
    // ストリーミング用の nn::audio::WaveBuffer を 4 つ準備し、
    // それぞれに対して波形データを読み込みます。
    //////////////////////////////////////////////////////////////////////////
    const size_t streamBufferSize = 12 * 1024;
    const int streamBufferCount = 4;
    nn::audio::MemoryPoolType memoryPoolForStream;
    void* streamPoolAddress = g_Allocator.Allocate(streamBufferSize * streamBufferCount, nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(streamPoolAddress);
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &memoryPoolForStream, streamPoolAddress, streamBufferSize * streamBufferCount));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&memoryPoolForStream));

    StreamUtility streamUtil;
    streamUtil.Open(g_BgmFileNames[0]);
    nn::audio::VoiceType streamVoice;
    NN_ABORT_UNLESS(nn::audio::AcquireVoiceSlot(
        &config,
        &streamVoice,
        streamUtil.GetSampleRate(),
        streamUtil.GetChannelCount(),
        nn::audio::SampleFormat_PcmInt16,
        nn::audio::VoiceType::PriorityHighest,
        nullptr,
        0));
    nn::audio::SetVoiceDestination(&config, &streamVoice, &finalMix);
    nn::audio::SetVoicePlayState(&streamVoice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(&streamVoice, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontLeft]);
    nn::audio::SetVoiceMixVolume(&streamVoice, &finalMix, 0.707f / 2, 0, mainBus[nn::audio::ChannelMapping_FrontRight]);
    StreamWaveBuffer streamBuffer[streamBufferCount];
    {
        nn::util::BytePtr base(nn::audio::GetMemoryPoolAddress(&memoryPoolForStream));
        for (int i = 0; i < streamBufferCount; ++i)
        {
            size_t readSize = streamUtil.Read(base.Get(), streamBufferSize);
            streamBuffer[i].waveBuffer.buffer = base.Get();
            streamBuffer[i].waveBuffer.size = readSize;
            streamBuffer[i].waveBuffer.startSampleOffset = 0;
            streamBuffer[i].waveBuffer.endSampleOffset = static_cast<int32_t>(readSize / sizeof(int16_t));
            streamBuffer[i].waveBuffer.loop = false;
            streamBuffer[i].waveBuffer.isEndOfStream = false;
            streamBuffer[i].waveBuffer.pContext = nullptr;
            streamBuffer[i].waveBuffer.contextSize = 0;
            streamBuffer[i].bufferAddress = base.Get();
            streamBuffer[i].bufferSize = streamBufferSize;
            base += readSize;

            nn::audio::AppendWaveBuffer(&streamVoice, &streamBuffer[i].waveBuffer);
        }
    }

    PrintUsage();
    bool running = true;
    while (running)
    {
        auto buttonDown = GetPadButtonDown();

        // StreamVoice の再生状態の更新
        if(buttonDown.Test< ::nn::hid::NpadButton::A >())
        {
            nn::audio::VoiceType::PlayState state = nn::audio::GetVoicePlayState(&streamVoice);
            state = (state == nn::audio::VoiceType::PlayState_Play) ? nn::audio::VoiceType::PlayState_Pause : nn::audio::VoiceType::PlayState_Play;
            nn::audio::SetVoicePlayState(&streamVoice, state);
        }

        // SE の再生
        if (buttonDown.Test< ::nn::hid::NpadButton::X >())
        {
            if (nn::audio::GetReleasedWaveBuffer(&seVoices[0]))
            {
                nn::audio::AppendWaveBuffer(&seVoices[0], &seWaveBuffers[0]);
            }
        }
        if (buttonDown.Test< ::nn::hid::NpadButton::Y >())
        {
            if (nn::audio::GetReleasedWaveBuffer(&seVoices[1]))
            {
                nn::audio::AppendWaveBuffer(&seVoices[1], &seWaveBuffers[1]);
            }
        }

        if(buttonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            running = false; // 終了処理へ
        }

        //////////////////////////////////////////////////////////////////////////
        // ストリーム再生用 WaveBuffer の更新
        //
        // 再生が完了した nn::audio::WaveBuffer に対して、
        // 逐次データの読み込みを行います。
        // このサンプルでは「アタッチ状態」にあるメモリプール内に
        // nn::fs で直接データを読み込んでいます。
        //////////////////////////////////////////////////////////////////////////
        if (const nn::audio::WaveBuffer* waveBuffer = nn::audio::GetReleasedWaveBuffer(&streamVoice))
        {
            // 更新対象の WaveBuffer のインデックスを取得
            int idx = 0;
            for (; idx < streamBufferCount; ++idx)
            {
                if (streamBuffer[idx].waveBuffer.buffer == waveBuffer->buffer)
                {
                    break;
                }
            }
            NN_ABORT_UNLESS_LESS(idx, streamBufferCount);

            // streamBuffer[idx].bufferAddress で指定されるバッファは、 memoryPoolForStream に含まれています。
            // 「アタッチ状態」にある memoryPoolForStream 内に配置されたバッファに対して
            // nn::fs で直接データを読み込みます。
            // またこの方法以外に nn::audio::CopyMemoryPoolData() を用いて、
            // アクセスすることも可能です。
            streamUtil.Read(streamBuffer[idx].bufferAddress, streamBuffer[idx].bufferSize);
            // バッファの更新が完了した WaveBuffer を streamVoice に再追加します。
            nn::audio::AppendWaveBuffer(&streamVoice, waveBuffer);
        }

        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

    } // for (;;)

    // サンプルの終了処理を開始します。
    streamUtil.Close();

    //////////////////////////////////////////////////////////////////////////
    // メモリプールの破棄
    //
    // メモリプールは、その内部にオーディオレンダラが利用中のデータがある場合、デタッチが完了しません。
    // またオーディオレンダラはアプリケーションとは非同期にオーディオ処理を実行していますので、
    // 利用の停止まで待つ必要が出てくる場合があります。
    //
    // またメモリプールで利用中の領域は gfx のメモリプールとの共存ができません。
    // 利用の完了したメモリプールは必ず破棄してください。
    // ただし nn::audio::CloseAudioRenderer() 呼び出し時に、
    // 対象のオーディオレンダラに紐づくすべてのメモリプールは、自動的にデタッチ＆削除されます。
    // この場合は以下に解説する削除処理を省略することも可能です。
    //////////////////////////////////////////////////////////////////////////

    // 利用していたメモリプールにデタッチ要求をします。
    nn::audio::RequestDetachMemoryPool(&memoryPoolForSe);
    nn::audio::RequestDetachMemoryPool(&memoryPoolForStream);

    // メモリプール内に配置されたリソースを利用する、すべての処理を停止します。
    // これらの停止が完了するまではメモリプールのデタッチは完了しません。
    nn::audio::SetVoicePlayState(&seVoices[0], nn::audio::VoiceType::PlayState_Stop);
    nn::audio::SetVoicePlayState(&seVoices[1], nn::audio::VoiceType::PlayState_Stop);
    nn::audio::SetVoicePlayState(&streamVoice, nn::audio::VoiceType::PlayState_Stop);

    // メモリプールがデタッチされるのを待ちます。
    while (nn::audio::IsMemoryPoolAttached(&memoryPoolForSe) ||
        nn::audio::IsMemoryPoolAttached(&memoryPoolForStream))
    {
        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));
    }

    // メモリプールのデタッチが完了しました。除去を行います。
    nn::audio::ReleaseMemoryPool(&config, &memoryPoolForSe);
    nn::audio::ReleaseMemoryPool(&config, &memoryPoolForStream);

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    if (sePoolAddress)
    {
        g_Allocator.Free(sePoolAddress);
        sePoolAddress = nullptr;
    }
    if (streamPoolAddress)
    {
        g_Allocator.Free(streamPoolAddress);
        streamPoolAddress = nullptr;
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
    g_Allocator.Finalize();

}  // NOLINT(readability/fn_size)
