﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioEffectAuxUtility.cpp,PageSampleAudioAudioEffectAuxUtility}
 *
 * @brief
 * Aux を利用したオーディオエフェクトのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioEffectAuxUtility Aux を利用したオーディオエフェクト
 * @tableofcontents
 *
 * @brief
 * Aux を利用したオーディオエフェクトのサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionBrief 概要
 * Aux と、CPU 処理として実装されたオーディオエフェクトを利用して、再生サンプルにエフェクト処理を施すサンプルです。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioEffectAuxUtility
 * Samples/Sources/Applications/AudioEffectAuxUtility @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionHowToOperate 操作方法
 * キーボードや DebugPad による入力を用いて SE を再生することができます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> SampleSe0 を一度再生                 </td></tr>
 * <tr><td> [B]             </td><td> SampleSe1 を一度再生                 </td></tr>
 * <tr><td> [R]             </td><td> AuxI3dl2Reverb エフェクトの設定を更新</td></tr>
 * <tr><td> [X]             </td><td> AuxReverb エフェクトの設定を更新     </td></tr>
 * <tr><td> [Y]             </td><td> AuxDelay エフェクトの設定を更新      </td></tr>
 * <tr><td> [Up][Down]      </td><td> エフェクト対象の SE の再生音量を調整 </td></tr>
 * <tr><td> [Left][Right]   </td><td> エフェクト対象の SE の左右のパン調整 </td></tr>
 * <tr><td> [L]             </td><td> エフェクトを施す対象の SE を切り替え </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムを終了                     </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioEffectAuxUtility_SectionDetail 解説
 * このサンプルプログラムは、Aux を利用したオーディオエフェクト処理を施すものです。
 *
 * Aux エフェクトとについてその利用方法を示しています。
 *
 * @subsection PageSampleAudioAudioEffectAuxUtility_SubsectionAuxEffect Aux エフェクト
 * Aux エフェクトはオーディオレンダラからオーディオサンプルデータの取得、およびその書き戻しを可能にします。
 * この機能によってユーザーは独自の信号処理を挿入することが可能になります。
 *
 * 利用の流れは以下の通りです。
 *
 * - Aux に利用するバッファを確保する。
 * - SubMix/FinalMix に Aux エフェクトを追加する。
 * - オーディオレンダラからのサンプルデータを受け取る。
 * - 信号処理を施す。（ここでは、簡単なボリューム調整とパン処理を施します。）
 * - オーディオレンダラにサンプルデータを書き戻す。
 *
 * 最初に Aux エフェクトに利用するバッファを確保しておきます。
 * このバッファに書き出されたサンプルデータをユーザーは処理することになります。
 * ここでバッファサイズを大きくとると、最大遅延量が増加します。
 * 一方で小さくとりすぎると、ユーザーの独自処理に使える時間がすくなくなるため、音途切れの原因となります。
 *
 * 次に Aux エフェクトを SubMix/FinalMix に追加します。
 * ここで Aux エフェクトに送るチャンネルも設定しておきます。
 * オーディオレンダラから受け取るデータはブロックインターリーブされています。
 * ここで指定したチャンネル分だけのチャンネルブロックが含まれることになります。
 *
 * 最後にメインループの中で信号処理を行います。
 * この処理に時間がかかりすぎると音途切れの原因となります。
 * またユーザーはオーディオレンダラに受け取った分と同数のサンプルデータを書き戻す必要があります。
 *
 * @subsection PageSampleAudioAudioEffectAuxUtility_SubsectionI3dl2ReverbEffect AuxI3dl2Reverb エフェクト
 * I3dl2Reverb エフェクトは Interactive 3D Audio Level 2 の仕様に部分的に準拠したリバーブエフェクトです。
 *
 * 利用方法は、nn::audio::InitializeAuxI3dl2Reverb(), nn::audio::ProcessAuxI3dl2Reverb() を参照してください。
 *
 */

#include <cstdlib>
#include <cmath>
#include <limits>
#include <algorithm>
#include <string>

#include <nn/nn_Abort.h>
#include <nns/nns_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {
// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int SeCount = 2;

const char Title[] = "AudioEffectAuxUtility";

// - and add / remove them to / from the files lists
const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

const float VolumeDelta = 0.005f;
const float BalanceDelta = 0.005f;
const int ReverbParameterSetCount = 2;
const int DelayParameterSetCount = 2;

nn::audio::I3dl2ReverbParameterSet i3dl2ReverbParameterUserSet =
{
    -1000.f,    //roomGain;
    -1000.f,    //roomHfGain
    1.0f,       //lateReverbDecayTime
    0.3f,       //lateReverbHfDecayRatio
    -500.f,     //reflectionsGain;
    0.01f,      //reflectionsDelayTime;
    -310.f,     //reverbGain;
    0.04f,      //reverbDelayTime;
    80.f,       //reverbDiffusion;
    60.f,       //reverbDensity;
    5000.f,     //hfReference
    0.2f,       //dryGain;
};

nn::audio::ReverbParameterSet auxReverbParameterSets[ReverbParameterSetCount] =
{
    {
        nn::audio::ReverbType::EarlyMode_SmallRoom, // earlyMode
        0.6f,                                       // earlyGain
        10.0f,                                      // predelayTimeMilliSeconds
        nn::audio::ReverbType::LateMode_Room,       // lateMode
        0.7f,                                       // lateGain
        1.0f,                                       // decayTimeSeconds
        0.1f,                                       // highFreqDecayRatio
        0.8f,                                       // coloration
        0.9f,                                       // reverbGain
        1.0f,                                       // outGain
        0.5f,                                       // dryGain
    },
    {
        nn::audio::ReverbType::EarlyMode_Hall,      // earlyMode
        0.8f,                                       // earlyGain
        20.0f,                                      // predelayTimeMilliSeconds
        nn::audio::ReverbType::LateMode_Hall,       // lateMode
        0.8f,                                       // lateGain
        3.0f,                                       // decayTimeSeconds
        0.3f,                                       // highFreqDecayRatio
        0.0f,                                       // coloration
        0.8f,                                       // reverbGain
        0.9f,                                       // outGain
        0.8f,                                       // dryGain
    }
};

nn::audio::DelayParameterSet auxDelayParameterSets[DelayParameterSetCount] =
{
    {
        nn::TimeSpan::FromMilliSeconds(400),        // delayTimeMilliSeconds
        0.8f,       // inGain
        0.4f,       // feedbackGain
        0.7f,       // dryGain
        0.0f,       // channelSpread
        0.8f        // lowPassAmount
    },
    {
        nn::TimeSpan::FromMilliSeconds(10),         // delayTimeMilliSeconds
        0.7f,       // inGain
        0.7f,       // feedbackGain
        0.7f,       // dryGain
        0.0f,       // channelSpread
        0.8f        // lowPassAmount
    },
};

bool g_EnabledAuxI3dl2Effect = true;
bool g_EnabledAuxReverbEffect = true;
bool g_EnabledAuxDelayEffect = true;

struct AudioEffectSampleStatus
{
    int i3dl2ReverbParameterIndex;
    int auxReverbParameterIndex;
    int auxDelayParameterIndex;
    std::string i3dl2ReverbState;
    std::string auxReverbState;
    std::string auxDelayState;
    float volume;
    float position;
    bool seSourceFlag;
};
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA      = nn::hid::KeyboardKey::A::Index;
    map.buttonB      = nn::hid::KeyboardKey::B::Index;
    map.buttonX      = nn::hid::KeyboardKey::X::Index;
    map.buttonY      = nn::hid::KeyboardKey::Y::Index;
    map.buttonL      = nn::hid::KeyboardKey::L::Index;
    map.buttonR      = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft   = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight  = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp     = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown   = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart  = nn::hid::KeyboardKey::Space::Index;
    map.buttonSelect = nn::hid::KeyboardKey::Minus::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("---------------------------------------------- \n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("Renderer Sample Rate %d\n", RenderRate);
    NNS_LOG("---------------------------------------------- \n");
    NNS_LOG("[A]             StartSound     (SampleSe0)     \n");
    NNS_LOG("[B]             StartSound     (SampleSe1)     \n");
    NNS_LOG("[Up][Down]      [Aux] Control SE Volume        \n");
    NNS_LOG("[Left][Right]   [Aux] Control SE Panning       \n");
    NNS_LOG("[L]             Switch effect target SE        \n");
    NNS_LOG("[R]             [I3dl2Reverb] Update Parameters\n");
    NNS_LOG("[X]             [Reverb] Update Parameters     \n");
    NNS_LOG("[Y]             [Delay] Update Parameters      \n");
    NNS_LOG("[-]             Print Usage                    \n");
    NNS_LOG("[Start/Space]   Finish Sample Program          \n");
    NNS_LOG("---------------------------------------------- \n");
}

void ProcessUserEffect(nn::audio::AuxType aux, int32_t* readBuffer, int sampleCount, int channelCount, int auxInputChannelCount,  float balance, float volume, nn::audio::AuxI3dl2ReverbType* pOutI3dl2Reverb, nn::audio::AuxReverbType* pOutAuxReverb, nn::audio::AuxDelayType* pOutAuxDelay)
{
    // Aux エフェクトによって受け取ったサンプルデータを加工する処理を行います。
    // ここではごく単純なボリューム調整とパン処理を行います。

    // オーディオレンダラからサンプルデータを読み込みます。
    int readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, channelCount * sampleCount * 2);
    while( readCount )
    {
        // Aux エフェクトによって得られるサンプルデータは、チャンネルごとにブロックインタリーブされています。
        // 各ブロックは Aux エフェクトを追加した FinalMix / SubMix が保持するサンプル数です。
        // 今回は FinalMix に追加したので、各ブロックに含まれるサンプル数は parameter.sampleCount に一致します。
        int samplesPerFrame = sampleCount;

        // 含まれるチャンネル数は SetAuxInputOutput() で設定した数となります。
        int channelBlockCount = auxInputChannelCount;

        // ReadAuxSendBuffer() で読み込まれたサンプルが、何フレーム分に相当するか取得します。
        int frameCount = readCount / (samplesPerFrame * channelCount);

        for( auto frame = 0; frame < frameCount; ++frame )
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * channelBlockCount;

            for( auto ch = 0; ch < channelBlockCount; ++ch )
            {
                int32_t* addr = bufferBase + ch * samplesPerFrame;
                float vol = (ch == 0) ? (balance * volume) : ((1.0f - balance) * volume);
                for (auto i = 0; i < samplesPerFrame; ++i)
                {
                    addr[i] = static_cast<int32_t>(addr[i] * vol);
                }
            }

            // AuxI3dl2Reverb process a frame
            if (g_EnabledAuxI3dl2Effect)
            {
                nn::audio::ProcessAuxI3dl2Reverb(pOutI3dl2Reverb, bufferBase, bufferBase, samplesPerFrame);
            }
            if(g_EnabledAuxReverbEffect)
            {
                nn::audio::ProcessAuxReverb(pOutAuxReverb, bufferBase, bufferBase, samplesPerFrame);
            }
            if (g_EnabledAuxDelayEffect)
            {
                nn::audio::ProcessAuxDelay(pOutAuxDelay, bufferBase, bufferBase, samplesPerFrame);
            }
        }

        // オーディオレンダラに加工したサンプルを書き戻します。
        int writeCount = nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        if( writeCount != readCount )
        {
            // オーディオレンダラにすべてのサンプルを書き戻すことができませんでした。
            // 一般的には原因として以下の点が考えられます。
            //
            // - nn::audio::ReadAuxSendBuffer() の読み出し頻度が、処理負荷増等の理由により、低くなっている。
            // - その結果 nn::audio::ReadAuxSendBuffer() で読み込まれるサンプル数が増えている。
            // - 増えたすべてのサンプルを、 一度に nn::audio::WriteAuxReturnBuffer() で書き戻そうとしている。
            // - その結果 ReturnBuffer のバッファが不足しており、readCount に指定したすべてのサンプルを書き戻すことができなかった。
            //
            // この問題を回避するためには、以下の改善を検討してください。
            // - nn::audio::ReadAuxSendBuffer() で読み込むサンプル数を減らす
            // - nn::audio::ReadAuxSendBuffer() を呼び出す頻度を上げる( == Aux エフェクト処理の更新頻度を上げる)
            //
            NNS_LOG("Notice: Could not write back whole sample\n");
        }
        readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, channelCount * sampleCount);
    }
}

void UpdateUserEffectParameter(const nn::hid::NpadButtonSet& buttons, float& volume, float& balance)
{
    if ( !(buttons.Test< ::nn::hid::NpadButton::Up >() ||
           buttons.Test< ::nn::hid::NpadButton::Down >() ||
           buttons.Test< ::nn::hid::NpadButton::Left >() ||
           buttons.Test< ::nn::hid::NpadButton::Right >()))
    {
        return;
    }

    if(buttons.Test< ::nn::hid::NpadButton::Up >())
    {
        volume = std::min(1.0f, volume + VolumeDelta);
    }
    if(buttons.Test< ::nn::hid::NpadButton::Down >())
    {
        volume = std::max(0.0f, volume - VolumeDelta);
    }
    if(buttons.Test< ::nn::hid::NpadButton::Left >())
    {
        balance = std::min(1.0f, balance + BalanceDelta);
    }
    if(buttons.Test< ::nn::hid::NpadButton::Right >())
    {
        balance = std::max(0.0f, balance - BalanceDelta);
    }

    NNS_LOG("UserEffect: Volume:%.2f Position:%.2f\n", volume, balance);
}

int UpdateI3dl2ReverbParameter(nn::audio::AuxI3dl2ReverbType* pReverb, int i3dl2UpdateMode, std::string& message)
{
    // Reverb のパラメータを更新します。
    nn::audio::I3dl2ReverbParameterSet parameterSet;
    parameterSet = nn::audio::GetAuxI3dl2ReverbParameters(pReverb); //use the original parameter set when re-enable

    switch (i3dl2UpdateMode)
    {
    case nn::audio::I3dl2ReverbType::Preset_SmallRoom:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_SmallRoom);
        message = "SmallRoom";
        break;
    case nn::audio::I3dl2ReverbType::Preset_LargeRoom:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_LargeRoom);
        message = "LargeRoom";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Hall:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_Hall);
        message = "Hall";
        break;
    case nn::audio::I3dl2ReverbType::Preset_CavernousCathedral:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_CavernousCathedral);
        message = "Cathedral";
        break;
    case nn::audio::I3dl2ReverbType::Preset_MetalCorridor:
        nn::audio::LoadI3dl2ReverbPreset(&parameterSet, nn::audio::I3dl2ReverbType::Preset_MetalCorridor);
        message = "Corridor";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count:
        parameterSet = i3dl2ReverbParameterUserSet;
        message = "UserDefine ";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count + 1:
        // Reverb エフェクトの無効化します。
        g_EnabledAuxI3dl2Effect = false;
        message = "disabled";
        break;
    case nn::audio::I3dl2ReverbType::Preset_Count + 2:
        // 有効にする際に、Reverb エフェクトの内部状態はリセットされます。
        // よって前回無効化したタイミングで鳴っていた残響音はすべてクリアされます。
        g_EnabledAuxI3dl2Effect = true;
        nn::audio::ResetAuxI3dl2Reverb(pReverb);
        message = "re-enabled";
        break;
    default:
        break;
    }

    nn::audio::SetAuxI3dl2ReverbParameters(pReverb, &parameterSet);
    return (i3dl2UpdateMode + 1) % 8;
}

int UpdateAuxReverbParameter(nn::audio::AuxReverbType* reverb, int reverbUpdateMode, std::string& message)
{
    // Reverb のパラメータを更新します。
    nn::audio::ReverbParameterSet* parameterSet = nullptr;
    switch (reverbUpdateMode)
    {
    case 0:
        parameterSet = &auxReverbParameterSets[reverbUpdateMode];
        message = "SmallRoom";
        break;
    case 1:
        parameterSet = &auxReverbParameterSets[reverbUpdateMode];
        message = "Hall";
        break;
    case 2:
        // Reverb エフェクトの無効化します。
        g_EnabledAuxReverbEffect = false;
        message = "disabled";
        break;
    case 3:
        // 有効にする際に、Reverb エフェクトの内部状態はリセットされます。
        // よって前回無効化したタイミングで鳴っていた残響音はすべてクリアされます。
        g_EnabledAuxReverbEffect = true;
        message = "re-enabled";
        break;
    default:
        break;
    }

    if (parameterSet != nullptr)
    {
        // Reverb エフェクトパラメータを更新します。
        // 更新は、次回の nn::audio::RequestUpdateAudioRenderer() 時に
        // オーディオレンダラに反映されます。
        nn::audio::SetAuxReverbParameters(reverb, parameterSet);
    }


    return (reverbUpdateMode + 1) % 4;
}

int UpdateAuxDelayParameter(nn::audio::AuxDelayType* delay, int delayUpdateMode, std::string& message)
{
    nn::audio::DelayParameterSet* parameterSet = nullptr;
    switch (delayUpdateMode)
    {
    case 0:
        parameterSet = &auxDelayParameterSets[delayUpdateMode];
        message = "long";
        break;
    case 1:
        parameterSet = &auxDelayParameterSets[delayUpdateMode];
        message = "short";
        break;
    case 2:
        g_EnabledAuxDelayEffect = false;
        message = "delay disabled";
        break;
    case 3:
        g_EnabledAuxDelayEffect = true;
        message = "delay re-enabled";
        break;
    default:
        break;
    }

    if (parameterSet != nullptr)
    {
        NNS_LOG("DelayTimeMax: %d\n", nn::audio::GetAuxDelayTimeMax(delay));
        nn::audio::SetAuxDelayParameters(delay, parameterSet);
    }


    return (delayUpdateMode + 1) % 4;
}


void UpdateEffectInputOutput(nn::audio::AuxType* aux, int8_t* targetBus, int busCount)
{
    nn::audio::SetAuxInputOutput(aux, targetBus,targetBus, busCount);
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    AudioEffectSampleStatus sampleStatus = { 0 };

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6;
    parameter.voiceCount = 24;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 3; // Aux, BufferMixer * 2
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t subBusA[2];
    subBusA[nn::audio::ChannelMapping_FrontLeft] = 2;
    subBusA[nn::audio::ChannelMapping_FrontRight] = 3;
    int8_t subBusB[2];
    subBusB[nn::audio::ChannelMapping_FrontLeft] = 0;
    subBusB[nn::audio::ChannelMapping_FrontRight] = 1;


    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備し AudioRenderer に登録します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 4);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    // エフェクトバッファ用のメモリプールを準備します。
    // MemoryPool は nn::audio::AcquireMemoryPool() で渡される address 、 size の領域を書き換えることはありません。
    // そのため g_EffectBufferPoolMemory として準備した領域をすべてをメモリプールに利用することができます。
    // またこのサンプルでは WaveBuffer, Effect とメモリの用途別にアロケータを準備しています。
    // MemoryPool はその領域全体を「アタッチ」「デタッチ」しますので、
    // MemoryPool の操作に必要な粒度に応じてメモリ領域を分けておくことをおすすめします。
    // MemoryPool の詳しい利用法は @link ../../../Samples/Sources/Applications/AudioMemoryPool
    // Samples/Sources/Applications/AudioMemoryPool @endlink を参照してください。
    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = nn::audio::AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
    rt = nn::audio::RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    // Aux エフェクトを用意します。
    nn::audio::AuxType aux;

    // Aux エフェクトで利用する Send Buffer & Return Buffer を確保します。
    // ここでは 4 オーディオフレーム分のバッファを確保しますので、最大で 5ms * 4 フレーム = 20ms の遅延が発生します。
    // このバッファサイズを小さくすることで遅延量も減りますが、後述するユーザー側の処理をより高頻度に行う必要がでてきます。
    const int AuxBufferFrameCount = 16;
    size_t auxBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, AuxBufferFrameCount, sizeof(subBusA));
    void* sendBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(sendBuffer);
    void* returnBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(returnBuffer);

    // Aux エフェクトを FinalMix に追加します。
    result = nn::audio::AddAux(&config, &aux, &finalMix, sendBuffer, returnBuffer, auxBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    // Aux エフェクトへの入力チャンネルを設定します。
    int auxInputChannelCount = sizeof(subBusA);
    nn::audio::SetAuxInputOutput(&aux, subBusA, subBusA, auxInputChannelCount);


    // I3dl2Reverb を pAux に追加します。
    nn::audio::AuxI3dl2ReverbType i3dl2Reverb;
    int8_t auxSubBusA[2];
    auxSubBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBusA[nn::audio::ChannelMapping_FrontRight] = 1;
    int auxI3dl2ReverbChannelCount = sizeof(auxSubBusA);

    size_t i3dl2ReverbSize = nn::audio::GetRequiredBufferSizeForAuxI3dl2Reverb(parameter.sampleRate, auxI3dl2ReverbChannelCount);
    void* i3dl2ReverbBuffer = g_EffectBufferAllocator.Allocate(i3dl2ReverbSize);
    NN_ABORT_UNLESS(i3dl2ReverbBuffer != nullptr);

    nn::audio::InitializeAuxI3dl2Reverb(&i3dl2Reverb, i3dl2ReverbBuffer, i3dl2ReverbSize, parameter.sampleRate, auxI3dl2ReverbChannelCount);

    nn::audio::SetAuxI3dl2ReverbInputOutput(&i3dl2Reverb, auxSubBusA, auxSubBusA, auxI3dl2ReverbChannelCount);

    nn::audio::I3dl2ReverbParameterSet i3dl2Parameter;
    nn::audio::LoadI3dl2ReverbPreset(&i3dl2Parameter, nn::audio::I3dl2ReverbType::Preset_SmallRoom);
    nn::audio::SetAuxI3dl2ReverbParameters(&i3dl2Reverb, &i3dl2Parameter);

    // Reverb を pAux に追加します。
    nn::audio::AuxReverbType auxReverb;
    int8_t auxSubBusB[2];
    auxSubBusB[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBusB[nn::audio::ChannelMapping_FrontRight] = 1;
    int auxReverbChannelCount = sizeof(auxSubBusB);

    size_t auxReverbSize = nn::audio::GetRequiredBufferSizeForAuxReverb(parameter.sampleRate, auxReverbChannelCount);
    void* auxReverbBuffer = g_EffectBufferAllocator.Allocate(auxReverbSize);
    NN_ABORT_UNLESS(auxReverbBuffer != nullptr);

    nn::audio::InitializeAuxReverb(&auxReverb, auxReverbBuffer, auxReverbSize, parameter.sampleRate, auxReverbChannelCount);

    nn::audio::SetAuxReverbInputOutput(&auxReverb, auxSubBusB, auxSubBusB, auxReverbChannelCount);
    nn::audio::SetAuxReverbParameters(&auxReverb, &auxReverbParameterSets[0]);

    nn::audio::AuxDelayType auxDelay;
    int8_t auxSubBusC[2];
    auxSubBusC[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxSubBusC[nn::audio::ChannelMapping_FrontRight] = 1;
    int auxDelayChannelCount = sizeof(auxSubBusC);

    size_t auxDelaySize = nn::audio::GetRequiredBufferSizeForAuxDelay(auxDelayParameterSets[0].delayTime, parameter.sampleRate, auxDelayChannelCount);
    void* auxDelayBuffer = g_EffectBufferAllocator.Allocate(auxDelaySize);
    NN_ABORT_UNLESS(auxDelayBuffer != nullptr);

    nn::audio::InitializeAuxDelay(&auxDelay, auxDelayBuffer, auxDelaySize, auxDelayParameterSets[0].delayTime, parameter.sampleRate, auxDelayChannelCount);

    nn::audio::SetAuxDelayInputOutput(&auxDelay, auxSubBusC, auxSubBusC, auxDelayChannelCount);
    NNS_LOG("DelayTimeMax: %d\n", nn::audio::GetAuxDelayTimeMax(&auxDelay));
    nn::audio::SetAuxDelayParameters(&auxDelay, &auxDelayParameterSets[0]);

    // BufferMixer エフェクトを用意します。
    nn::audio::BufferMixerType mixerA;
    nn::audio::BufferMixerType mixerB;
    nn::audio::AddBufferMixer(&config, &mixerA, &finalMix);
    nn::audio::AddBufferMixer(&config, &mixerB, &finalMix);
    nn::audio::SetBufferMixerVolume(&mixerA, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerA, 1, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerB, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerB, 1, 1.0f);
    nn::audio::SetBufferMixerInputOutput(&mixerA, subBusA, mainBus, sizeof(mainBus));
    nn::audio::SetBufferMixerInputOutput(&mixerB, subBusB, mainBus, sizeof(mainBus));

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // SE を読み込みます。
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        auto bus = (i == 0) ? subBusA : subBusB;
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, bus[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, bus[nn::audio::ChannelMapping_FrontRight]);
    }

    PrintUsage();

    // オーディオレンダラからサンプルデータを受け取るバッファを準備します。
    int32_t* readBuffer = static_cast<int32_t*>(g_Allocator.Allocate(auxBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, auxBufferSize);

    sampleStatus.volume = 1.0f;
    sampleStatus.position = 0.5f;
    sampleStatus.i3dl2ReverbState = "default";
    sampleStatus.auxReverbState = "default";
    sampleStatus.auxDelayState = "default";
    bool running = true;

    while (running)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        // 各 Effect のパラメータを更新します。
        {
            // I3dl2Reverb のパラメータを更新します。
            if(npadButtonDown.Test<::nn::hid::NpadButton::R>())
            {
                sampleStatus.i3dl2ReverbParameterIndex = UpdateI3dl2ReverbParameter(&i3dl2Reverb, sampleStatus.i3dl2ReverbParameterIndex, sampleStatus.i3dl2ReverbState);
            }
            // AuxReverb のパラメータを更新します。
            if(npadButtonDown.Test<::nn::hid::NpadButton::X>())
            {
                sampleStatus.auxReverbParameterIndex = UpdateAuxReverbParameter(&auxReverb, sampleStatus.auxReverbParameterIndex, sampleStatus.auxReverbState);
            }
            // AuxDelay
            if(npadButtonDown.Test<::nn::hid::NpadButton::Y>())
            {
                sampleStatus.auxDelayParameterIndex = UpdateAuxDelayParameter(&auxDelay, sampleStatus.auxDelayParameterIndex, sampleStatus.auxDelayState);
            }

            // Aux エフェクトのパラメータを更新します。
            UpdateUserEffectParameter(npadButtonCurrent, sampleStatus.volume, sampleStatus.position);

            // Aux については、独自定義のエフェクト処理も行います。
            ProcessUserEffect(aux, readBuffer, parameter.sampleCount, channelCount, auxInputChannelCount, sampleStatus.position, sampleStatus.volume, &i3dl2Reverb, &auxReverb, &auxDelay);

            // 全 Effect の InputOutput を更新します。
            if(npadButtonDown.Test<::nn::hid::NpadButton::L>())
            {
                int8_t* targetBus = (sampleStatus.seSourceFlag) ? subBusA : subBusB;
                UpdateEffectInputOutput(&aux, targetBus, channelCount);
                sampleStatus.seSourceFlag = !sampleStatus.seSourceFlag;
            }

            if (npadButtonDown.Test<::nn::hid::NpadButton::Minus>())
            {
                PrintUsage();
            }

            if ( npadButtonDown.Test<::nn::hid::NpadButton::A>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::B>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::X>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::Y>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::L>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::R>())
            {
                NNS_LOG("Effected SE No.%d : I3dl2Reverb:%-15s\nAuxReverb:%-15s\nAuxDelay:%-15s\n",
                    sampleStatus.seSourceFlag ? 1 : 0,
                    sampleStatus.i3dl2ReverbState.c_str(),
                    sampleStatus.auxReverbState.c_str(),
                    sampleStatus.auxDelayState.c_str());
            }
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]) != nullptr)
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // 各エフェクトに施したパラメータ変更をオーディオレンダラに反映します。
        // 反映するまでは、パラメータ変更は有効になりません。
        // nn::audio::RequestUpdateAudioRenderer() の呼び出し中は、各エフェクトへのアクセスは避けてください。
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS(result.IsSuccess());

        //  サンプルを終了する
        if(npadButtonDown.Test<::nn::hid::NpadButton::Plus>())
        {
            running = false;
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
        if (header[i])
        {
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }

    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
    if (sendBuffer)
    {
        g_EffectBufferAllocator.Free(sendBuffer);
        sendBuffer = nullptr;
    }
    if (returnBuffer)
    {
        g_EffectBufferAllocator.Free(returnBuffer);
        returnBuffer = nullptr;
    }
    if (readBuffer)
    {
        g_Allocator.Free(readBuffer);
        readBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
